/* ServerThread.java
 * Copyright (C) 2007, 2008 Gustav Behm <gbehm (at) kth.se>
 *
 * This file is part of Netzack.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


package se.kth.netzack;
import se.kth.netzack.*;
import java.net.*;
import java.lang.*;
import java.io.*;
import java.util.*;
import java.nio.*;
import javax.swing.Timer;
import java.awt.event.*;



public class ServerThread extends Thread implements ActionListener {

    ServerSocket socket;
    Server server;

    Vector peers = new Vector(10, 5);
    Vector prune = new Vector(5, 2);

    Object lock;

    private final Timer timer;


    ServerThread(Server s, Object l) {
	super("ServerThread");

	lock = l;

	try {
	    socket = new ServerSocket(0);
	}
	catch(IOException e) {
	    Netzack.debug(e);
	}

	timer = new Timer(1000, this);

	server = s;
	start();
    }

    public int getPort() {
	return socket.getLocalPort();
    }

    public void run() {
	synchronized(lock) {
	    lock.notify();
	}

	try {
	    while(true) {
		Socket s = socket.accept();
		synchronized(peers) {
		    peers.add(new Peer(s, server));
		}
	    }
	}
	catch(IOException E) {

	}
    }

    public void close() {
	try {
	    socket.close();
	}
	catch(IOException e){

	}
    }

    public void dispatch(Packet packet) {
	synchronized(peers) {
	    if(peers.isEmpty())
		return;

	    Iterator i = peers.iterator();
	    while(i.hasNext()) {
		Peer p = (Peer)i.next();
		if(!packet.isFrom(p)) {
		    p.send(packet);
		}

	    }
	}
    }


    public void checkConnections() {
	synchronized(peers) {
	    Iterator i = peers.iterator();
	    while(i.hasNext()) {
		Peer p = (Peer)i.next();
		if(!p.isConnected()) {
		    //Issue reconnect
		    if(p.tries < 3) {
			if(p.tries == 0) {
			    prune.add(p);
			    timer.restart();
			}

			p.tries++;
			server.reestablish(p.id);
		    }

		}
	    }
	}

    }


    public void send(Packet packet) {
	synchronized(peers) {
	    Iterator i = peers.iterator();
	    while(i.hasNext()) {
		Peer p = (Peer)i.next();
		if(!p.send(packet)) {
		    //Issue reconnect
		    /*if(p.tries < 3) {
			if(p.tries == 0) {
			    prune.add(p);
			    timer.restart();
			}

			p.tries++;
			server.reestablish(p.id);
			}*/

		}
	    }
	}

    }


    public Peer connect(InetAddress addr, int port) {

	try{
	    Socket s = new Socket(addr, port);
	    Peer p = new Peer(s, server);
	    synchronized(peers) {
		peers.add(p);
	    }
	    Netzack.debug("Connecting to " + addr.toString() + ":" + port);
	    return p;
	}
	catch(IOException e) {
	    Netzack.debug(e);
	    Netzack.debug(addr.toString());
	}

	return null;
    }



    public void reconnect(int to, InetAddress addr, int port) {
	synchronized(peers) {
	    Iterator i = peers.iterator();
	    while(i.hasNext()) {
		Peer p = (Peer)i.next();
		if(p.id == to) {
		    Peer peer = connect(addr, port);
		    Packet packet = Packet.reconnectPacket(server.id, to);
		    peer.send(packet);
		    peers.remove(p);
		    break;
		}
	    }
	}
    }

    public void actionPerformed(ActionEvent e) {
	synchronized(peers) {
	    Iterator i = prune.iterator();
	    while(i.hasNext()) {
		Peer p = (Peer)i.next();
		Netzack.debug("Pruning dead peer #" + p.id);
		peers.remove(p);
	    }
	}
	prune.clear();
	timer.stop();
    }

    public void reconnected(int to) {
	synchronized(peers) {
	    Iterator i = peers.iterator();
	    while(i.hasNext()) {
		Peer p = (Peer)i.next();
		if(p.id == to) {
		    peers.remove(p);
		    break;
		}
	    }
	}
    }

}
