# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module CodeCoverage
      class SourceFileClassifier
        PATTERNS = {
          'frontend' => [
            %r{^app/assets/javascripts/.*\.(js|jsx|ts|tsx|vue)$},
            %r{^app/assets/stylesheets/.*\.(css|scss)$},
            %r{^ee/app/assets/javascripts/.*\.(js|jsx|ts|tsx|vue)$},
            %r{^ee/app/assets/stylesheets/.*\.(css|scss)$},
            %r{^jh/app/assets/javascripts/.*\.(js|jsx|ts|tsx|vue)$},
            %r{^spec/frontend/},
            %r{^ee/spec/frontend/},
            %r{^spec/frontend_integration/},
            %r{^app/assets/javascripts/.*\.graphql$},
            /\.stories\.js$/
          ],
          'backend' => [
            %r{^app/(models|controllers|services|workers|helpers|mailers|policies|presenters|uploaders|validators|enums|events|experiments|facades|channels)/.*\.rb$},
            %r{^app/serializers/.*\.rb$},
            %r{^app/graphql/.*\.rb$},
            %r{^app/components/.*\.rb$},
            %r{^app/views/.*\.(haml|erb)$},
            %r{^lib/.*\.rb$},
            %r{^lib/api/.*\.rb$},
            %r{^ee/app/.*\.rb$},
            %r{^ee/lib/.*\.rb$},
            %r{^jh/app/.*\.rb$},
            %r{^jh/lib/.*\.rb$},
            %r{^spec/.*_spec\.rb$},
            %r{^ee/spec/.*_spec\.rb$},
            %r{^lib/tasks/.*\.rake$}
          ],
          'database' => [
            %r{^db/migrate/.*\.rb$},
            %r{^db/post_migrate/.*\.rb$},
            %r{^ee/db/geo/migrate/.*\.rb$},
            %r{^db/structure\.sql$},
            %r{^db/seeds\.rb$},
            %r{^db/fixtures/}
          ],
          'infrastructure' => [
            /^\.gitlab-ci\.yml$/,
            %r{^\.gitlab/ci/.*\.(yml|yaml)$},
            /Dockerfile/,
            /\.dockerfile$/,
            %r{^scripts/pipeline/}
          ],
          'qa' => [
            %r{^qa/.*\.rb$}
          ],
          'workhorse' => [
            %r{^workhorse/.*\.go$}
          ],
          'tooling' => [
            %r{^tooling/.*\.(rb|js)$},
            %r{^rubocop/.*\.rb$},
            %r{^danger/.*\.rb$},
            /^\.rubocop\.yml$/
          ],
          'configuration' => [
            %r{^config/.*\.(yml|yaml|rb)$}
          ]
        }.freeze

        # Classifies a collection of file paths into their respective types
        #
        # @param file_paths [Array<String>] List of file paths to classify
        # @return [Hash<String, String>] Hash mapping file path to file type
        def classify(file_paths)
          file_paths.each_with_object({}) do |file_path, result|
            result[file_path] = determine_type(file_path)
          end
        end

        private

        # Determines the type of a single file based on pattern matching
        #
        # @param file_path [String] The file path to classify
        # @return [String] The file type category
        def determine_type(file_path)
          PATTERNS.each do |type, patterns|
            return type if patterns.any? { |pattern| file_path.match?(pattern) }
          end

          'other'
        end
      end
    end
  end
end
