/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*- */
/* dbus-sysdeps-util.c Would be in dbus-sysdeps.c, but not used in libdbus
 * 
 * Copyright (C) 2002, 2003, 2004, 2005  Red Hat, Inc.
 * Copyright (C) 2003 CodeFactory AB
 *
 * Licensed under the Academic Free License version 2.1
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#undef open

#define STRSAFE_NO_DEPRECATE

#include "dbus-sysdeps.h"
#include "dbus-internals.h"
#include "dbus-protocol.h"
#include "dbus-string.h"
#include "dbus-sysdeps.h"
#include "dbus-sysdeps-win.h"
#include "dbus-memory.h"

#include <io.h>
#include <sys/stat.h>
#include <aclapi.h>
#include <winsock2.h>

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <winsock2.h>   // WSA error codes

#if defined __MINGW32__ || (defined _MSC_VER && _MSC_VER <= 1310)
/* save string functions version
   using DBusString needs to much time because of uncommon api 
*/ 
#define errno_t int

errno_t strcat_s(char *dest, size_t size, char *src) 
{
  _dbus_assert(strlen(dest) + strlen(src) +1 <= size);
  strcat(dest,src);
  return 0;
}

errno_t strcpy_s(char *dest, size_t size, char *src)
{
  _dbus_assert(strlen(src) +1 <= size);
  strcpy(dest,src);  
  return 0;
}
#endif

/**
 * Does the chdir, fork, setsid, etc. to become a daemon process.
 *
 * @param pidfile #NULL, or pidfile to create
 * @param print_pid_fd file descriptor to print daemon's pid to, or -1 for none
 * @param error return location for errors
 * @returns #FALSE on failure
 */
dbus_bool_t
_dbus_become_daemon (const DBusString *pidfile,
                     DBusPipe         *print_pid_pipe,
                     DBusError        *error)
{
  return TRUE;
}

/**
 * Creates a file containing the process ID.
 *
 * @param filename the filename to write to
 * @param pid our process ID
 * @param error return location for errors
 * @returns #FALSE on failure
 */
dbus_bool_t
_dbus_write_pid_file (const DBusString *filename,
                      unsigned long     pid,
                      DBusError        *error)
{
  const char *cfilename;
  DBusFile file;
  FILE *f;

  cfilename = _dbus_string_get_const_data (filename);

  if (!_dbus_file_open(&file, cfilename, O_WRONLY|O_CREAT|O_EXCL|O_BINARY, 0644))
    {
      dbus_set_error (error, _dbus_error_from_errno (errno),
                      "Failed to open \"%s\": %s", cfilename,
                      _dbus_strerror (errno));
      return FALSE;
    }

  if ((f = fdopen (file.FDATA, "w")) == NULL)
    {
      dbus_set_error (error, _dbus_error_from_errno (errno),
                      "Failed to fdopen fd %d: %s", file.FDATA, _dbus_strerror (errno));
      _dbus_file_close (&file, NULL);
      return FALSE;
    }

  if (fprintf (f, "%lu\n", pid) < 0)
    {
      dbus_set_error (error, _dbus_error_from_errno (errno),
                      "Failed to write to \"%s\": %s", cfilename,
                      _dbus_strerror (errno));

      fclose (f);
      return FALSE;
    }

  if (fclose (f) == EOF)
    {
      dbus_set_error (error, _dbus_error_from_errno (errno),
                      "Failed to close \"%s\": %s", cfilename,
                      _dbus_strerror (errno));
      return FALSE;
    }

  return TRUE;
}

/**
 * Writes the given pid_to_write to a pidfile (if non-NULL) and/or to a
 * pipe (if non-NULL). Does nothing if pidfile and print_pid_pipe are both
 * NULL.
 *
 * @param pidfile the file to write to or #NULL
 * @param print_pid_pipe the pipe to write to or #NULL
 * @param pid_to_write the pid to write out
 * @param error error on failure
 * @returns FALSE if error is set
 */
dbus_bool_t
_dbus_write_pid_to_file_and_pipe (const DBusString *pidfile,
                                  DBusPipe         *print_pid_pipe,
                                  dbus_pid_t        pid_to_write,
                                  DBusError        *error)
{
  if (pidfile)
    {
      _dbus_verbose ("writing pid file %s\n", _dbus_string_get_const_data (pidfile));
      if (!_dbus_write_pid_file (pidfile,
                                 pid_to_write,
                                 error))
        {
          _dbus_verbose ("pid file write failed\n");
          _DBUS_ASSERT_ERROR_IS_SET(error);
          return FALSE;
        }
    }
  else
    {
      _dbus_verbose ("No pid file requested\n");
    }

  if (print_pid_pipe != NULL && _dbus_pipe_is_valid (print_pid_pipe))
    {
      DBusString pid;
      int bytes;

      _dbus_verbose ("writing our pid to pipe %d\n", print_pid_pipe->fd_or_handle);

      if (!_dbus_string_init (&pid))
        {
          _DBUS_SET_OOM (error);
          return FALSE;
        }

      if (!_dbus_string_append_int (&pid, pid_to_write) ||
          !_dbus_string_append (&pid, "\n"))
        {
          _dbus_string_free (&pid);
          _DBUS_SET_OOM (error);
          return FALSE;
        }

      bytes = _dbus_string_get_length (&pid);
      if (_dbus_pipe_write (print_pid_pipe, &pid, 0, bytes, error) != bytes)
        {
          /* _dbus_pipe_write sets error only on failure, not short write */
          if (error != NULL && !dbus_error_is_set(error))
            {
              dbus_set_error (error, DBUS_ERROR_FAILED,
                              "Printing message bus PID: did not write enough bytes\n");
            }
          _dbus_string_free (&pid);
          return FALSE;
        }

      _dbus_string_free (&pid);
    }
  else
    {
      _dbus_verbose ("No pid pipe to write to\n");
    }

  return TRUE;
}

/**
 * Verify that after the fork we can successfully change to this user.
 *
 * @param user the username given in the daemon configuration
 * @returns #TRUE if username is valid
 */
dbus_bool_t
_dbus_verify_daemon_user (const char *user)
{
  return TRUE;
}

/**
 * Changes the user and group the bus is running as.
 *
 * @param user the user to become
 * @param error return location for errors
 * @returns #FALSE on failure
 */
dbus_bool_t
_dbus_change_to_daemon_user  (const char    *user,
                              DBusError     *error)
{
  return TRUE;
}

/**
 * Changes the user and group the bus is running as.
 *
 * @param uid the new user ID
 * @param gid the new group ID
 * @param error return location for errors
 * @returns #FALSE on failure
 */
dbus_bool_t
_dbus_change_identity  (dbus_uid_t     uid,
                        dbus_gid_t     gid,
                        DBusError     *error)
{
  return TRUE;
}

/** Checks if user is at the console
*
* @param username user to check
* @param error return location for errors
* @returns #TRUE is the user is at the consolei and there are no errors
*/
dbus_bool_t
_dbus_user_at_console(const char *username,
                      DBusError  *error)
{
#ifdef DBUS_WINCE
	return TRUE;
#else
  dbus_bool_t retval = FALSE;
  wchar_t *wusername;
  DWORD sid_length;
  PSID user_sid, console_user_sid;
  HWINSTA winsta;

  wusername = _dbus_win_utf8_to_utf16 (username, error);
  if (!wusername)
    return FALSE;

  // TODO remove
  if (!_dbus_win_account_to_sid (wusername, &user_sid, error))
    goto out0;

  /* Now we have the SID for username. Get the SID of the
   * user at the "console" (window station WinSta0)
   */
  if (!(winsta = OpenWindowStation ("WinSta0", FALSE, READ_CONTROL)))
    {
      _dbus_win_set_error_from_win_error (error, GetLastError ());
      goto out2;
    }

  sid_length = 0;
  GetUserObjectInformation (winsta, UOI_USER_SID,
                            NULL, 0, &sid_length);
  if (sid_length == 0)
    {
      /* Nobody is logged on */
      goto out2;
    }

  if (sid_length < 0 || sid_length > 1000)
    {
      dbus_set_error_const (error, DBUS_ERROR_FAILED, "Invalid SID length");
      goto out3;
    }

  console_user_sid = dbus_malloc (sid_length);
  if (!console_user_sid)
    {
      _DBUS_SET_OOM (error);
      goto out3;
    }

  if (!GetUserObjectInformation (winsta, UOI_USER_SID,
                                 console_user_sid, sid_length, &sid_length))
    {
      _dbus_win_set_error_from_win_error (error, GetLastError ());
      goto out4;
    }

  if (!IsValidSid (console_user_sid))
    {
      dbus_set_error_const (error, DBUS_ERROR_FAILED, "Invalid SID");
      goto out4;
    }

  retval = EqualSid (user_sid, console_user_sid);

out4:
  dbus_free (console_user_sid);
out3:
  CloseWindowStation (winsta);
out2:
  dbus_free (user_sid);
out0:
  dbus_free (wusername);

  return retval;
#endif //DBUS_WINCE
}

/**
 * Removes a directory; Directory must be empty
 * 
 * @param filename directory filename
 * @param error initialized error object
 * @returns #TRUE on success
 */
dbus_bool_t
_dbus_delete_directory (const DBusString *filename,
                        DBusError        *error)
{
  const char *filename_c;

  _DBUS_ASSERT_ERROR_IS_CLEAR (error);

  filename_c = _dbus_string_get_const_data (filename);

  if (rmdir (filename_c) != 0)
    {
      dbus_set_error (error, DBUS_ERROR_FAILED,
                      "Failed to remove directory %s: %s\n",
                      filename_c, _dbus_strerror (errno));
      return FALSE;
    }

  return TRUE;
}

/** Installs a signal handler
 *
 * @param sig the signal to handle
 * @param handler the handler
 */
void
_dbus_set_signal_handler (int               sig,
                          DBusSignalHandler handler)
{
  _dbus_verbose ("_dbus_set_signal_handler() has to be implemented\n");
}

/** Checks if a file exists
*
* @param file full path to the file
* @returns #TRUE if file exists
*/
dbus_bool_t 
_dbus_file_exists (const char *file)
{
  HANDLE h = CreateFile(
          file, /* LPCTSTR lpFileName*/
          0, /* DWORD dwDesiredAccess */
          0, /* DWORD dwShareMode*/
          NULL, /* LPSECURITY_ATTRIBUTES lpSecurityAttributes */
          OPEN_EXISTING, /* DWORD dwCreationDisposition */
          FILE_ATTRIBUTE_NORMAL, /* DWORD dwFlagsAndAttributes */
          NULL /* HANDLE hTemplateFile */
        );

    /* file not found, use local copy of session.conf  */
    if (h != INVALID_HANDLE_VALUE && GetLastError() != ERROR_PATH_NOT_FOUND)
      {
        CloseHandle(h);
        return TRUE;
      }
    else
        return FALSE;  
}

/**
 * stat() wrapper.
 *
 * @param filename the filename to stat
 * @param statbuf the stat info to fill in
 * @param error return location for error
 * @returns #FALSE if error was set
 */
dbus_bool_t
_dbus_stat(const DBusString *filename,
           DBusStat         *statbuf,
           DBusError        *error)
{
#ifdef DBUS_WINCE
	return TRUE;
	//TODO
#else
  const char *filename_c;
#if !defined(DBUS_WIN) && !defined(DBUS_WINCE)

  struct stat sb;
#else

  WIN32_FILE_ATTRIBUTE_DATA wfad;
  char *lastdot;
  DWORD rc;
  PSID owner_sid, group_sid;
  PSECURITY_DESCRIPTOR sd;
#endif

  _DBUS_ASSERT_ERROR_IS_CLEAR (error);

  filename_c = _dbus_string_get_const_data (filename);

  if (!GetFileAttributesEx (filename_c, GetFileExInfoStandard, &wfad))
    {
      _dbus_win_set_error_from_win_error (error, GetLastError ());
      return FALSE;
    }

  if (wfad.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
    statbuf->mode = _S_IFDIR;
  else
    statbuf->mode = _S_IFREG;

  statbuf->mode |= _S_IREAD;
  if (wfad.dwFileAttributes & FILE_ATTRIBUTE_READONLY)
    statbuf->mode |= _S_IWRITE;

  lastdot = strrchr (filename_c, '.');
  if (lastdot && stricmp (lastdot, ".exe") == 0)
    statbuf->mode |= _S_IEXEC;

  statbuf->mode |= (statbuf->mode & 0700) >> 3;
  statbuf->mode |= (statbuf->mode & 0700) >> 6;

  statbuf->nlink = 1;

  sd = NULL;
  rc = GetNamedSecurityInfo ((char *) filename_c, SE_FILE_OBJECT,
                             OWNER_SECURITY_INFORMATION |
                             GROUP_SECURITY_INFORMATION,
                             &owner_sid, &group_sid,
                             NULL, NULL,
                             &sd);
  if (rc != ERROR_SUCCESS)
    {
      _dbus_win_set_error_from_win_error (error, rc);
      if (sd != NULL)
        LocalFree (sd);
      return FALSE;
    }

#ifdef ENABLE_UID_TO_SID
  /* FIXME */
  statbuf->uid = _dbus_win_sid_to_uid_t (owner_sid);
  statbuf->gid = _dbus_win_sid_to_uid_t (group_sid);
#endif

  LocalFree (sd);

  statbuf->size = ((dbus_int64_t) wfad.nFileSizeHigh << 32) + wfad.nFileSizeLow;

  statbuf->atime =
    (((dbus_int64_t) wfad.ftLastAccessTime.dwHighDateTime << 32) +
     wfad.ftLastAccessTime.dwLowDateTime) / 10000000 - DBUS_INT64_CONSTANT (116444736000000000);

  statbuf->mtime =
    (((dbus_int64_t) wfad.ftLastWriteTime.dwHighDateTime << 32) +
     wfad.ftLastWriteTime.dwLowDateTime) / 10000000 - DBUS_INT64_CONSTANT (116444736000000000);

  statbuf->ctime =
    (((dbus_int64_t) wfad.ftCreationTime.dwHighDateTime << 32) +
     wfad.ftCreationTime.dwLowDateTime) / 10000000 - DBUS_INT64_CONSTANT (116444736000000000);

  return TRUE;
#endif //DBUS_WINCE
}


#ifdef HAVE_DIRENT_H

// mingw ships with dirent.h
#include <dirent.h>
#define _dbus_opendir opendir
#define _dbus_readdir readdir
#define _dbus_closedir closedir

#else

#ifdef HAVE_IO_H
#include <io.h> // win32 file functions
#endif

#include <sys/types.h>
#include <stdlib.h>

/* This file is part of the KDE project
Copyright (C) 2000 Werner Almesberger

libc/sys/linux/sys/dirent.h - Directory entry as returned by readdir

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
Boston, MA 02110-1301, USA.
*/
#define HAVE_NO_D_NAMLEN	/* no struct dirent->d_namlen */
#define HAVE_DD_LOCK  		/* have locking mechanism */

#define MAXNAMLEN 255		/* sizeof(struct dirent.d_name)-1 */

#define __dirfd(dir) (dir)->dd_fd

/* struct dirent - same as Unix */
struct dirent
  {
    long d_ino;                    /* inode (always 1 in WIN32) */
    off_t d_off;                /* offset to this dirent */
    unsigned short d_reclen;    /* length of d_name */
    char d_name[_MAX_FNAME+1];    /* filename (null terminated) */
  };

/* typedef DIR - not the same as Unix */
typedef struct
  {
    long handle;                /* _findfirst/_findnext handle */
    short offset;                /* offset into directory */
    short finished;             /* 1 if there are not more files */
    struct _finddata_t fileinfo;  /* from _findfirst/_findnext */
    char *dir;                  /* the dir we are reading */
    struct dirent dent;         /* the dirent to return */
  }
DIR;

/**********************************************************************
* Implement dirent-style opendir/readdir/closedir on Window 95/NT
*
* Functions defined are opendir(), readdir() and closedir() with the
* same prototypes as the normal dirent.h implementation.
*
* Does not implement telldir(), seekdir(), rewinddir() or scandir().
* The dirent struct is compatible with Unix, except that d_ino is
* always 1 and d_off is made up as we go along.
*
* The DIR typedef is not compatible with Unix.
**********************************************************************/

DIR * _dbus_opendir(const char *dir)
{
  DIR *dp;
  char *filespec;
  long handle;
  int index;

  filespec = malloc(strlen(dir) + 2 + 1);
  strcpy(filespec, dir);
  index = strlen(filespec) - 1;
  if (index >= 0 && (filespec[index] == '/' || filespec[index] == '\\'))
    filespec[index] = '\0';
  strcat(filespec, "\\*");

  dp = (DIR *)malloc(sizeof(DIR));
  dp->offset = 0;
  dp->finished = 0;
  dp->dir = strdup(dir);

  if ((handle = _findfirst(filespec, &(dp->fileinfo))) < 0)
    {
      if (errno == ENOENT)
        dp->finished = 1;
      else
        return NULL;
    }

  dp->handle = handle;
  free(filespec);

  return dp;
}

struct dirent * _dbus_readdir(DIR *dp)
  {
    if (!dp || dp->finished)
      return NULL;

    if (dp->offset != 0)
      {
        if (_findnext(dp->handle, &(dp->fileinfo)) < 0)
          {
            dp->finished = 1;
            errno = 0;
            return NULL;
          }
      }
    dp->offset++;

    strncpy(dp->dent.d_name, dp->fileinfo.name, _MAX_FNAME);
    dp->dent.d_ino = 1;
    dp->dent.d_reclen = strlen(dp->dent.d_name);
    dp->dent.d_off = dp->offset;

    return &(dp->dent);
  }


int _dbus_closedir(DIR *dp)
{
  if (!dp)
    return 0;
  _findclose(dp->handle);
  if (dp->dir)
    free(dp->dir);
  if (dp)
    free(dp);

  return 0;
}

#endif //#ifdef HAVE_DIRENT_H

/**
 * Internals of directory iterator
 */
struct DBusDirIter
  {
    DIR *d; /**< The DIR* from opendir() */

  };

/**
 * Open a directory to iterate over.
 *
 * @param filename the directory name
 * @param error exception return object or #NULL
 * @returns new iterator, or #NULL on error
 */
DBusDirIter*
_dbus_directory_open (const DBusString *filename,
                      DBusError        *error)
{
  DIR *d;
  DBusDirIter *iter;
  const char *filename_c;

  _DBUS_ASSERT_ERROR_IS_CLEAR (error);

  filename_c = _dbus_string_get_const_data (filename);

  d = _dbus_opendir (filename_c);
  if (d == NULL)
    {
      dbus_set_error (error, _dbus_error_from_errno (errno),
                      "Failed to read directory \"%s\": %s",
                      filename_c,
                      _dbus_strerror (errno));
      return NULL;
    }
  iter = dbus_new0 (DBusDirIter, 1);
  if (iter == NULL)
    {
      _dbus_closedir (d);
      dbus_set_error (error, DBUS_ERROR_NO_MEMORY,
                      "Could not allocate memory for directory iterator");
      return NULL;
    }

  iter->d = d;

  return iter;
}

/**
 * Get next file in the directory. Will not return "." or ".."  on
 * UNIX. If an error occurs, the contents of "filename" are
 * undefined. The error is never set if the function succeeds.
 *
 * @todo for thread safety, I think we have to use
 * readdir_r(). (GLib has the same issue, should file a bug.)
 *
 * @param iter the iterator
 * @param filename string to be set to the next file in the dir
 * @param error return location for error
 * @returns #TRUE if filename was filled in with a new filename
 */
dbus_bool_t
_dbus_directory_get_next_file (DBusDirIter      *iter,
                               DBusString       *filename,
                               DBusError        *error)
{
  struct dirent *ent;

  _DBUS_ASSERT_ERROR_IS_CLEAR (error);

again:
  errno = 0;
  ent = _dbus_readdir (iter->d);
  if (ent == NULL)
    {
      if (errno != 0)
        dbus_set_error (error,
                        _dbus_error_from_errno (errno),
                        "%s", _dbus_strerror (errno));
      return FALSE;
    }
  else if (ent->d_name[0] == '.' &&
           (ent->d_name[1] == '\0' ||
            (ent->d_name[1] == '.' && ent->d_name[2] == '\0')))
    goto again;
  else
    {
      _dbus_string_set_length (filename, 0);
      if (!_dbus_string_append (filename, ent->d_name))
        {
          dbus_set_error (error, DBUS_ERROR_NO_MEMORY,
                          "No memory to read directory entry");
          return FALSE;
        }
      else
        return TRUE;
    }
}

/**
 * Closes a directory iteration.
 */
void
_dbus_directory_close (DBusDirIter *iter)
{
  _dbus_closedir (iter->d);
  dbus_free (iter);
}

/**
 * Checks whether the filename is an absolute path
 *
 * @param filename the filename
 * @returns #TRUE if an absolute path
 */
dbus_bool_t
_dbus_path_is_absolute (const DBusString *filename)
{
  if (_dbus_string_get_length (filename) > 0)
    return _dbus_string_get_byte (filename, 1) == ':'
           || _dbus_string_get_byte (filename, 0) == '\\'
           || _dbus_string_get_byte (filename, 0) == '/';
  else
    return FALSE;
}

/** @} */ /* End of DBusInternalsUtils functions */

/**
 * @addtogroup DBusString
 *
 * @{
 */
/**
 * Get the directory name from a complete filename
 * @param filename the filename
 * @param dirname string to append directory name to
 * @returns #FALSE if no memory
 */
dbus_bool_t
_dbus_string_get_dirname(const DBusString *filename,
                         DBusString       *dirname)
{
  int sep;

  _dbus_assert (filename != dirname);
  _dbus_assert (filename != NULL);
  _dbus_assert (dirname != NULL);

  /* Ignore any separators on the end */
  sep = _dbus_string_get_length (filename);
  if (sep == 0)
    return _dbus_string_append (dirname, "."); /* empty string passed in */

  while (sep > 0 &&
         (_dbus_string_get_byte (filename, sep - 1) == '/' ||
          _dbus_string_get_byte (filename, sep - 1) == '\\'))
    --sep;

  _dbus_assert (sep >= 0);

  if (sep == 0 ||
      (sep == 2 &&
       _dbus_string_get_byte (filename, 1) == ':' &&
       isalpha (_dbus_string_get_byte (filename, 0))))
    return _dbus_string_copy_len (filename, 0, sep + 1,
                                  dirname, _dbus_string_get_length (dirname));

  {
    int sep1, sep2;
    _dbus_string_find_byte_backward (filename, sep, '/', &sep1);
    _dbus_string_find_byte_backward (filename, sep, '\\', &sep2);

    sep = MAX (sep1, sep2);
  }
  if (sep < 0)
    return _dbus_string_append (dirname, ".");

  while (sep > 0 &&
         (_dbus_string_get_byte (filename, sep - 1) == '/' ||
          _dbus_string_get_byte (filename, sep - 1) == '\\'))
    --sep;

  _dbus_assert (sep >= 0);

  if ((sep == 0 ||
       (sep == 2 &&
        _dbus_string_get_byte (filename, 1) == ':' &&
        isalpha (_dbus_string_get_byte (filename, 0))))
      &&
      (_dbus_string_get_byte (filename, sep) == '/' ||
       _dbus_string_get_byte (filename, sep) == '\\'))
    return _dbus_string_copy_len (filename, 0, sep + 1,
                                  dirname, _dbus_string_get_length (dirname));
  else
    return _dbus_string_copy_len (filename, 0, sep - 0,
                                  dirname, _dbus_string_get_length (dirname));
}


/**
 * Checks to see if the UNIX user ID matches the UID of
 * the process. Should always return #FALSE on Windows.
 *
 * @param uid the UNIX user ID
 * @returns #TRUE if this uid owns the process.
 */
dbus_bool_t
_dbus_unix_user_is_process_owner (dbus_uid_t uid)
{
  return FALSE;
}

/*=====================================================================
  unix emulation functions - should be removed sometime in the future
 =====================================================================*/

/**
 * Checks to see if the UNIX user ID is at the console.
 * Should always fail on Windows (set the error to
 * #DBUS_ERROR_NOT_SUPPORTED).
 *
 * @param uid UID of person to check 
 * @param error return location for errors
 * @returns #TRUE if the UID is the same as the console user and there are no errors
 */
dbus_bool_t
_dbus_unix_user_is_at_console (dbus_uid_t         uid,
                               DBusError         *error)
{
  return FALSE;
}


/**
 * Parse a UNIX group from the bus config file. On Windows, this should
 * simply always fail (just return #FALSE).
 *
 * @param groupname the groupname text
 * @param gid_p place to return the gid
 * @returns #TRUE on success
 */
dbus_bool_t
_dbus_parse_unix_group_from_config (const DBusString  *groupname,
                                    dbus_gid_t        *gid_p)
{
  return FALSE;
}

/**
 * Parse a UNIX user from the bus config file. On Windows, this should
 * simply always fail (just return #FALSE).
 *
 * @param username the username text
 * @param uid_p place to return the uid
 * @returns #TRUE on success
 */
dbus_bool_t
_dbus_parse_unix_user_from_config (const DBusString  *username,
                                   dbus_uid_t        *uid_p)
{
  return FALSE;
}


/**
 * Gets all groups corresponding to the given UNIX user ID. On UNIX,
 * just calls _dbus_groups_from_uid(). On Windows, should always
 * fail since we don't know any UNIX groups.
 *
 * @param uid the UID
 * @param group_ids return location for array of group IDs
 * @param n_group_ids return location for length of returned array
 * @returns #TRUE if the UID existed and we got some credentials
 */
dbus_bool_t
_dbus_unix_groups_from_uid (dbus_uid_t            uid,
                            dbus_gid_t          **group_ids,
                            int                  *n_group_ids)
{
  return FALSE;
}



/** @} */ /* DBusString stuff */

/************************************************************************
 
 error handling
 
 ************************************************************************/


/**
 * Assigns an error name and message corresponding to a Win32 error
 * code to a DBusError. Does nothing if error is #NULL.
 *
 * @param error the error.
 * @param code the Win32 error code
 */
void
_dbus_win_set_error_from_win_error (DBusError *error,
                                    int        code)
{
  char *msg;

  /* As we want the English message, use the A API */
  FormatMessageA (FORMAT_MESSAGE_ALLOCATE_BUFFER |
                  FORMAT_MESSAGE_IGNORE_INSERTS |
                  FORMAT_MESSAGE_FROM_SYSTEM,
                  NULL, code, MAKELANGID (LANG_ENGLISH, SUBLANG_ENGLISH_US),
                  (LPTSTR) &msg, 0, NULL);
  if (msg)
    {
      char *msg_copy;

      msg_copy = dbus_malloc (strlen (msg));
      strcpy (msg_copy, msg);
      LocalFree (msg);

      dbus_set_error (error, "win32.error", "%s", msg_copy);
    }
  else
    dbus_set_error (error, "win32.error", "Unknown error code %d or FormatMessage failed", code);
}

void
_dbus_win_warn_win_error (const char *message,
                          int         code)
{
  DBusError error;

  dbus_error_init (&error);
  _dbus_win_set_error_from_win_error (&error, code);
  _dbus_warn ("%s: %s\n", message, error.message);
  dbus_error_free (&error);
}

/**
 * A wrapper around strerror() because some platforms
 * may be lame and not have strerror().
 *
 * @param error_number errno.
 * @returns error description.
 */
const char*
_dbus_strerror (int error_number)
{
#ifdef DBUS_WINCE
  // TODO
  return "unknown";
#else
  const char *msg;

  switch (error_number)
    {
    case WSAEINTR:
      return "Interrupted function call";
    case WSAEACCES:
      return "Permission denied";
    case WSAEFAULT:
      return "Bad address";
    case WSAEINVAL:
      return "Invalid argument";
    case WSAEMFILE:
      return "Too many open files";
    case WSAEWOULDBLOCK:
      return "Resource temporarily unavailable";
    case WSAEINPROGRESS:
      return "Operation now in progress";
    case WSAEALREADY:
      return "Operation already in progress";
    case WSAENOTSOCK:
      return "Socket operation on nonsocket";
    case WSAEDESTADDRREQ:
      return "Destination address required";
    case WSAEMSGSIZE:
      return "Message too long";
    case WSAEPROTOTYPE:
      return "Protocol wrong type for socket";
    case WSAENOPROTOOPT:
      return "Bad protocol option";
    case WSAEPROTONOSUPPORT:
      return "Protocol not supported";
    case WSAESOCKTNOSUPPORT:
      return "Socket type not supported";
    case WSAEOPNOTSUPP:
      return "Operation not supported";
    case WSAEPFNOSUPPORT:
      return "Protocol family not supported";
    case WSAEAFNOSUPPORT:
      return "Address family not supported by protocol family";
    case WSAEADDRINUSE:
      return "Address already in use";
    case WSAEADDRNOTAVAIL:
      return "Cannot assign requested address";
    case WSAENETDOWN:
      return "Network is down";
    case WSAENETUNREACH:
      return "Network is unreachable";
    case WSAENETRESET:
      return "Network dropped connection on reset";
    case WSAECONNABORTED:
      return "Software caused connection abort";
    case WSAECONNRESET:
      return "Connection reset by peer";
    case WSAENOBUFS:
      return "No buffer space available";
    case WSAEISCONN:
      return "Socket is already connected";
    case WSAENOTCONN:
      return "Socket is not connected";
    case WSAESHUTDOWN:
      return "Cannot send after socket shutdown";
    case WSAETIMEDOUT:
      return "Connection timed out";
    case WSAECONNREFUSED:
      return "Connection refused";
    case WSAEHOSTDOWN:
      return "Host is down";
    case WSAEHOSTUNREACH:
      return "No route to host";
    case WSAEPROCLIM:
      return "Too many processes";
    case WSAEDISCON:
      return "Graceful shutdown in progress";
    case WSATYPE_NOT_FOUND:
      return "Class type not found";
    case WSAHOST_NOT_FOUND:
      return "Host not found";
    case WSATRY_AGAIN:
      return "Nonauthoritative host not found";
    case WSANO_RECOVERY:
      return "This is a nonrecoverable error";
    case WSANO_DATA:
      return "Valid name, no data record of requested type";
    case WSA_INVALID_HANDLE:
      return "Specified event object handle is invalid";
    case WSA_INVALID_PARAMETER:
      return "One or more parameters are invalid";
    case WSA_IO_INCOMPLETE:
      return "Overlapped I/O event object not in signaled state";
    case WSA_IO_PENDING:
      return "Overlapped operations will complete later";
    case WSA_NOT_ENOUGH_MEMORY:
      return "Insufficient memory available";
    case WSA_OPERATION_ABORTED:
      return "Overlapped operation aborted";
#ifdef WSAINVALIDPROCTABLE

    case WSAINVALIDPROCTABLE:
      return "Invalid procedure table from service provider";
#endif
#ifdef WSAINVALIDPROVIDER

    case WSAINVALIDPROVIDER:
      return "Invalid service provider version number";
#endif
#ifdef WSAPROVIDERFAILEDINIT

    case WSAPROVIDERFAILEDINIT:
      return "Unable to initialize a service provider";
#endif

    case WSASYSCALLFAILURE:
      return "System call failure";
    }
  msg = strerror (error_number);
  if (msg == NULL)
    msg = "unknown";

  return msg;
#endif //DBUS_WINCE
}



/* lan manager error codes */
const char*
_dbus_lm_strerror(int error_number)
{
#ifdef DBUS_WINCE
  // TODO
  return "unknown";
#else
  const char *msg;
  switch (error_number)
    {
    case NERR_NetNotStarted:
      return "The workstation driver is not installed.";
    case NERR_UnknownServer:
      return "The server could not be located.";
    case NERR_ShareMem:
      return "An internal error occurred. The network cannot access a shared memory segment.";
    case NERR_NoNetworkResource:
      return "A network resource shortage occurred.";
    case NERR_RemoteOnly:
      return "This operation is not supported on workstations.";
    case NERR_DevNotRedirected:
      return "The device is not connected.";
    case NERR_ServerNotStarted:
      return "The Server service is not started.";
    case NERR_ItemNotFound:
      return "The queue is empty.";
    case NERR_UnknownDevDir:
      return "The device or directory does not exist.";
    case NERR_RedirectedPath:
      return "The operation is invalid on a redirected resource.";
    case NERR_DuplicateShare:
      return "The name has already been shared.";
    case NERR_NoRoom:
      return "The server is currently out of the requested resource.";
    case NERR_TooManyItems:
      return "Requested addition of items exceeds the maximum allowed.";
    case NERR_InvalidMaxUsers:
      return "The Peer service supports only two simultaneous users.";
    case NERR_BufTooSmall:
      return "The API return buffer is too small.";
    case NERR_RemoteErr:
      return "A remote API error occurred.";
    case NERR_LanmanIniError:
      return "An error occurred when opening or reading the configuration file.";
    case NERR_NetworkError:
      return "A general network error occurred.";
    case NERR_WkstaInconsistentState:
      return "The Workstation service is in an inconsistent state. Restart the computer before restarting the Workstation service.";
    case NERR_WkstaNotStarted:
      return "The Workstation service has not been started.";
    case NERR_BrowserNotStarted:
      return "The requested information is not available.";
    case NERR_InternalError:
      return "An internal error occurred.";
    case NERR_BadTransactConfig:
      return "The server is not configured for transactions.";
    case NERR_InvalidAPI:
      return "The requested API is not supported on the remote server.";
    case NERR_BadEventName:
      return "The event name is invalid.";
    case NERR_DupNameReboot:
      return "The computer name already exists on the network. Change it and restart the computer.";
    case NERR_CfgCompNotFound:
      return "The specified component could not be found in the configuration information.";
    case NERR_CfgParamNotFound:
      return "The specified parameter could not be found in the configuration information.";
    case NERR_LineTooLong:
      return "A line in the configuration file is too long.";
    case NERR_QNotFound:
      return "The printer does not exist.";
    case NERR_JobNotFound:
      return "The print job does not exist.";
    case NERR_DestNotFound:
      return "The printer destination cannot be found.";
    case NERR_DestExists:
      return "The printer destination already exists.";
    case NERR_QExists:
      return "The printer queue already exists.";
    case NERR_QNoRoom:
      return "No more printers can be added.";
    case NERR_JobNoRoom:
      return "No more print jobs can be added.";
    case NERR_DestNoRoom:
      return "No more printer destinations can be added.";
    case NERR_DestIdle:
      return "This printer destination is idle and cannot accept control operations.";
    case NERR_DestInvalidOp:
      return "This printer destination request contains an invalid control function.";
    case NERR_ProcNoRespond:
      return "The print processor is not responding.";
    case NERR_SpoolerNotLoaded:
      return "The spooler is not running.";
    case NERR_DestInvalidState:
      return "This operation cannot be performed on the print destination in its current state.";
    case NERR_QInvalidState:
      return "This operation cannot be performed on the printer queue in its current state.";
    case NERR_JobInvalidState:
      return "This operation cannot be performed on the print job in its current state.";
    case NERR_SpoolNoMemory:
      return "A spooler memory allocation failure occurred.";
    case NERR_DriverNotFound:
      return "The device driver does not exist.";
    case NERR_DataTypeInvalid:
      return "The data type is not supported by the print processor.";
    case NERR_ProcNotFound:
      return "The print processor is not installed.";
    case NERR_ServiceTableLocked:
      return "The service database is locked.";
    case NERR_ServiceTableFull:
      return "The service table is full.";
    case NERR_ServiceInstalled:
      return "The requested service has already been started.";
    case NERR_ServiceEntryLocked:
      return "The service does not respond to control actions.";
    case NERR_ServiceNotInstalled:
      return "The service has not been started.";
    case NERR_BadServiceName:
      return "The service name is invalid.";
    case NERR_ServiceCtlTimeout:
      return "The service is not responding to the control function.";
    case NERR_ServiceCtlBusy:
      return "The service control is busy.";
    case NERR_BadServiceProgName:
      return "The configuration file contains an invalid service program name.";
    case NERR_ServiceNotCtrl:
      return "The service could not be controlled in its present state.";
    case NERR_ServiceKillProc:
      return "The service ended abnormally.";
    case NERR_ServiceCtlNotValid:
      return "The requested pause or stop is not valid for this service.";
    case NERR_NotInDispatchTbl:
      return "The service control dispatcher could not find the service name in the dispatch table.";
    case NERR_BadControlRecv:
      return "The service control dispatcher pipe read failed.";
    case NERR_ServiceNotStarting:
      return "A thread for the new service could not be created.";
    case NERR_AlreadyLoggedOn:
      return "This workstation is already logged on to the local-area network.";
    case NERR_NotLoggedOn:
      return "The workstation is not logged on to the local-area network.";
    case NERR_BadUsername:
      return "The user name or group name parameter is invalid.";
    case NERR_BadPassword:
      return "The password parameter is invalid.";
    case NERR_UnableToAddName_W:
      return "@W The logon processor did not add the message alias.";
    case NERR_UnableToAddName_F:
      return "The logon processor did not add the message alias.";
    case NERR_UnableToDelName_W:
      return "@W The logoff processor did not delete the message alias.";
    case NERR_UnableToDelName_F:
      return "The logoff processor did not delete the message alias.";
    case NERR_LogonsPaused:
      return "Network logons are paused.";
    case NERR_LogonServerConflict:
      return "A centralized logon-server conflict occurred.";
    case NERR_LogonNoUserPath:
      return "The server is configured without a valid user path.";
    case NERR_LogonScriptError:
      return "An error occurred while loading or running the logon script.";
    case NERR_StandaloneLogon:
      return "The logon server was not specified. Your computer will be logged on as STANDALONE.";
    case NERR_LogonServerNotFound:
      return "The logon server could not be found.";
    case NERR_LogonDomainExists:
      return "There is already a logon domain for this computer.";
    case NERR_NonValidatedLogon:
      return "The logon server could not validate the logon.";
    case NERR_ACFNotFound:
      return "The security database could not be found.";
    case NERR_GroupNotFound:
      return "The group name could not be found.";
    case NERR_UserNotFound:
      return "The user name could not be found.";
    case NERR_ResourceNotFound:
      return "The resource name could not be found.";
    case NERR_GroupExists:
      return "The group already exists.";
    case NERR_UserExists:
      return "The user account already exists.";
    case NERR_ResourceExists:
      return "The resource permission list already exists.";
    case NERR_NotPrimary:
      return "This operation is only allowed on the primary domain controller of the domain.";
    case NERR_ACFNotLoaded:
      return "The security database has not been started.";
    case NERR_ACFNoRoom:
      return "There are too many names in the user accounts database.";
    case NERR_ACFFileIOFail:
      return "A disk I/O failure occurred.";
    case NERR_ACFTooManyLists:
      return "The limit of 64 entries per resource was exceeded.";
    case NERR_UserLogon:
      return "Deleting a user with a session is not allowed.";
    case NERR_ACFNoParent:
      return "The parent directory could not be located.";
    case NERR_CanNotGrowSegment:
      return "Unable to add to the security database session cache segment.";
    case NERR_SpeGroupOp:
      return "This operation is not allowed on this special group.";
    case NERR_NotInCache:
      return "This user is not cached in user accounts database session cache.";
    case NERR_UserInGroup:
      return "The user already belongs to this group.";
    case NERR_UserNotInGroup:
      return "The user does not belong to this group.";
    case NERR_AccountUndefined:
      return "This user account is undefined.";
    case NERR_AccountExpired:
      return "This user account has expired.";
    case NERR_InvalidWorkstation:
      return "The user is not allowed to log on from this workstation.";
    case NERR_InvalidLogonHours:
      return "The user is not allowed to log on at this time.";
    case NERR_PasswordExpired:
      return "The password of this user has expired.";
    case NERR_PasswordCantChange:
      return "The password of this user cannot change.";
    case NERR_PasswordHistConflict:
      return "This password cannot be used now.";
    case NERR_PasswordTooShort:
      return "The password does not meet the password policy requirements. Check the minimum password length, password complexity and password history requirements.";
    case NERR_PasswordTooRecent:
      return "The password of this user is too recent to change.";
    case NERR_InvalidDatabase:
      return "The security database is corrupted.";
    case NERR_DatabaseUpToDate:
      return "No updates are necessary to this replicant network/local security database.";
    case NERR_SyncRequired:
      return "This replicant database is outdated; synchronization is required.";
    case NERR_UseNotFound:
      return "The network connection could not be found.";
    case NERR_BadAsgType:
      return "This asg_type is invalid.";
    case NERR_DeviceIsShared:
      return "This device is currently being shared.";
    case NERR_NoComputerName:
      return "The computer name could not be added as a message alias. The name may already exist on the network.";
    case NERR_MsgAlreadyStarted:
      return "The Messenger service is already started.";
    case NERR_MsgInitFailed:
      return "The Messenger service failed to start.";
    case NERR_NameNotFound:
      return "The message alias could not be found on the network.";
    case NERR_AlreadyForwarded:
      return "This message alias has already been forwarded.";
    case NERR_AddForwarded:
      return "This message alias has been added but is still forwarded.";
    case NERR_AlreadyExists:
      return "This message alias already exists locally.";
    case NERR_TooManyNames:
      return "The maximum number of added message aliases has been exceeded.";
    case NERR_DelComputerName:
      return "The computer name could not be deleted.";
    case NERR_LocalForward:
      return "Messages cannot be forwarded back to the same workstation.";
    case NERR_GrpMsgProcessor:
      return "An error occurred in the domain message processor.";
    case NERR_PausedRemote:
      return "The message was sent, but the recipient has paused the Messenger service.";
    case NERR_BadReceive:
      return "The message was sent but not received.";
    case NERR_NameInUse:
      return "The message alias is currently in use. Try again later.";
    case NERR_MsgNotStarted:
      return "The Messenger service has not been started.";
    case NERR_NotLocalName:
      return "The name is not on the local computer.";
    case NERR_NoForwardName:
      return "The forwarded message alias could not be found on the network.";
    case NERR_RemoteFull:
      return "The message alias table on the remote station is full.";
    case NERR_NameNotForwarded:
      return "Messages for this alias are not currently being forwarded.";
    case NERR_TruncatedBroadcast:
      return "The broadcast message was truncated.";
    case NERR_InvalidDevice:
      return "This is an invalid device name.";
    case NERR_WriteFault:
      return "A write fault occurred.";
    case NERR_DuplicateName:
      return "A duplicate message alias exists on the network.";
    case NERR_DeleteLater:
      return "@W This message alias will be deleted later.";
    case NERR_IncompleteDel:
      return "The message alias was not successfully deleted from all networks.";
    case NERR_MultipleNets:
      return "This operation is not supported on computers with multiple networks.";
    case NERR_NetNameNotFound:
      return "This shared resource does not exist.";
    case NERR_DeviceNotShared:
      return "This device is not shared.";
    case NERR_ClientNameNotFound:
      return "A session does not exist with that computer name.";
    case NERR_FileIdNotFound:
      return "There is not an open file with that identification number.";
    case NERR_ExecFailure:
      return "A failure occurred when executing a remote administration command.";
    case NERR_TmpFile:
      return "A failure occurred when opening a remote temporary file.";
    case NERR_TooMuchData:
      return "The data returned from a remote administration command has been truncated to 64K.";
    case NERR_DeviceShareConflict:
      return "This device cannot be shared as both a spooled and a non-spooled resource.";
    case NERR_BrowserTableIncomplete:
      return "The information in the list of servers may be incorrect.";
    case NERR_NotLocalDomain:
      return "The computer is not active in this domain.";
#ifdef NERR_IsDfsShare

    case NERR_IsDfsShare:
      return "The share must be removed from the Distributed File System before it can be deleted.";
#endif

    case NERR_DevInvalidOpCode:
      return "The operation is invalid for this device.";
    case NERR_DevNotFound:
      return "This device cannot be shared.";
    case NERR_DevNotOpen:
      return "This device was not open.";
    case NERR_BadQueueDevString:
      return "This device name list is invalid.";
    case NERR_BadQueuePriority:
      return "The queue priority is invalid.";
    case NERR_NoCommDevs:
      return "There are no shared communication devices.";
    case NERR_QueueNotFound:
      return "The queue you specified does not exist.";
    case NERR_BadDevString:
      return "This list of devices is invalid.";
    case NERR_BadDev:
      return "The requested device is invalid.";
    case NERR_InUseBySpooler:
      return "This device is already in use by the spooler.";
    case NERR_CommDevInUse:
      return "This device is already in use as a communication device.";
    case NERR_InvalidComputer:
      return "This computer name is invalid.";
    case NERR_MaxLenExceeded:
      return "The string and prefix specified are too long.";
    case NERR_BadComponent:
      return "This path component is invalid.";
    case NERR_CantType:
      return "Could not determine the type of input.";
    case NERR_TooManyEntries:
      return "The buffer for types is not big enough.";
    case NERR_ProfileFileTooBig:
      return "Profile files cannot exceed 64K.";
    case NERR_ProfileOffset:
      return "The start offset is out of range.";
    case NERR_ProfileCleanup:
      return "The system cannot delete current connections to network resources.";
    case NERR_ProfileUnknownCmd:
      return "The system was unable to parse the command line in this file.";
    case NERR_ProfileLoadErr:
      return "An error occurred while loading the profile file.";
    case NERR_ProfileSaveErr:
      return "@W Errors occurred while saving the profile file. The profile was partially saved.";
    case NERR_LogOverflow:
      return "Log file %1 is full.";
    case NERR_LogFileChanged:
      return "This log file has changed between reads.";
    case NERR_LogFileCorrupt:
      return "Log file %1 is corrupt.";
    case NERR_SourceIsDir:
      return "The source path cannot be a directory.";
    case NERR_BadSource:
      return "The source path is illegal.";
    case NERR_BadDest:
      return "The destination path is illegal.";
    case NERR_DifferentServers:
      return "The source and destination paths are on different servers.";
    case NERR_RunSrvPaused:
      return "The Run server you requested is paused.";
    case NERR_ErrCommRunSrv:
      return "An error occurred when communicating with a Run server.";
    case NERR_ErrorExecingGhost:
      return "An error occurred when starting a background process.";
    case NERR_ShareNotFound:
      return "The shared resource you are connected to could not be found.";
    case NERR_InvalidLana:
      return "The LAN adapter number is invalid.";
    case NERR_OpenFiles:
      return "There are open files on the connection.";
    case NERR_ActiveConns:
      return "Active connections still exist.";
    case NERR_BadPasswordCore:
      return "This share name or password is invalid.";
    case NERR_DevInUse:
      return "The device is being accessed by an active process.";
    case NERR_LocalDrive:
      return "The drive letter is in use locally.";
    case NERR_AlertExists:
      return "The specified client is already registered for the specified event.";
    case NERR_TooManyAlerts:
      return "The alert table is full.";
    case NERR_NoSuchAlert:
      return "An invalid or nonexistent alert name was raised.";
    case NERR_BadRecipient:
      return "The alert recipient is invalid.";
    case NERR_AcctLimitExceeded:
      return "A user's session with this server has been deleted.";
    case NERR_InvalidLogSeek:
      return "The log file does not contain the requested record number.";
    case NERR_BadUasConfig:
      return "The user accounts database is not configured correctly.";
    case NERR_InvalidUASOp:
      return "This operation is not permitted when the Netlogon service is running.";
    case NERR_LastAdmin:
      return "This operation is not allowed on the last administrative account.";
    case NERR_DCNotFound:
      return "Could not find domain controller for this domain.";
    case NERR_LogonTrackingError:
      return "Could not set logon information for this user.";
    case NERR_NetlogonNotStarted:
      return "The Netlogon service has not been started.";
    case NERR_CanNotGrowUASFile:
      return "Unable to add to the user accounts database.";
    case NERR_TimeDiffAtDC:
      return "This server's clock is not synchronized with the primary domain controller's clock.";
    case NERR_PasswordMismatch:
      return "A password mismatch has been detected.";
    case NERR_NoSuchServer:
      return "The server identification does not specify a valid server.";
    case NERR_NoSuchSession:
      return "The session identification does not specify a valid session.";
    case NERR_NoSuchConnection:
      return "The connection identification does not specify a valid connection.";
    case NERR_TooManyServers:
      return "There is no space for another entry in the table of available servers.";
    case NERR_TooManySessions:
      return "The server has reached the maximum number of sessions it supports.";
    case NERR_TooManyConnections:
      return "The server has reached the maximum number of connections it supports.";
    case NERR_TooManyFiles:
      return "The server cannot open more files because it has reached its maximum number.";
    case NERR_NoAlternateServers:
      return "There are no alternate servers registered on this server.";
    case NERR_TryDownLevel:
      return "Try down-level (remote admin protocol) version of API instead.";
    case NERR_UPSDriverNotStarted:
      return "The UPS driver could not be accessed by the UPS service.";
    case NERR_UPSInvalidConfig:
      return "The UPS service is not configured correctly.";
    case NERR_UPSInvalidCommPort:
      return "The UPS service could not access the specified Comm Port.";
    case NERR_UPSSignalAsserted:
      return "The UPS indicated a line fail or low battery situation. Service not started.";
    case NERR_UPSShutdownFailed:
      return "The UPS service failed to perform a system shut down.";
    case NERR_BadDosRetCode:
      return "The program below returned an MS-DOS error code:";
    case NERR_ProgNeedsExtraMem:
      return "The program below needs more memory:";
    case NERR_BadDosFunction:
      return "The program below called an unsupported MS-DOS function:";
    case NERR_RemoteBootFailed:
      return "The workstation failed to boot.";
    case NERR_BadFileCheckSum:
      return "The file below is corrupt.";
    case NERR_NoRplBootSystem:
      return "No loader is specified in the boot-block definition file.";
    case NERR_RplLoadrNetBiosErr:
      return "NetBIOS returned an error:      The NCB and SMB are dumped above.";
    case NERR_RplLoadrDiskErr:
      return "A disk I/O error occurred.";
    case NERR_ImageParamErr:
      return "Image parameter substitution failed.";
    case NERR_TooManyImageParams:
      return "Too many image parameters cross disk sector boundaries.";
    case NERR_NonDosFloppyUsed:
      return "The image was not generated from an MS-DOS diskette formatted with /S.";
    case NERR_RplBootRestart:
      return "Remote boot will be restarted later.";
    case NERR_RplSrvrCallFailed:
      return "The call to the Remoteboot server failed.";
    case NERR_CantConnectRplSrvr:
      return "Cannot connect to the Remoteboot server.";
    case NERR_CantOpenImageFile:
      return "Cannot open image file on the Remoteboot server.";
    case NERR_CallingRplSrvr:
      return "Connecting to the Remoteboot server...";
    case NERR_StartingRplBoot:
      return "Connecting to the Remoteboot server...";
    case NERR_RplBootServiceTerm:
      return "Remote boot service was stopped; check the error log for the cause of the problem.";
    case NERR_RplBootStartFailed:
      return "Remote boot startup failed; check the error log for the cause of the problem.";
    case NERR_RPL_CONNECTED:
      return "A second connection to a Remoteboot resource is not allowed.";
    case NERR_BrowserConfiguredToNotRun:
      return "The browser service was configured with MaintainServerList=No.";
    case NERR_RplNoAdaptersStarted:
      return "Service failed to start since none of the network adapters started with this service.";
    case NERR_RplBadRegistry:
      return "Service failed to start due to bad startup information in the registry.";
    case NERR_RplBadDatabase:
      return "Service failed to start because its database is absent or corrupt.";
    case NERR_RplRplfilesShare:
      return "Service failed to start because RPLFILES share is absent.";
    case NERR_RplNotRplServer:
      return "Service failed to start because RPLUSER group is absent.";
    case NERR_RplCannotEnum:
      return "Cannot enumerate service records.";
    case NERR_RplWkstaInfoCorrupted:
      return "Workstation record information has been corrupted.";
    case NERR_RplWkstaNotFound:
      return "Workstation record was not found.";
    case NERR_RplWkstaNameUnavailable:
      return "Workstation name is in use by some other workstation.";
    case NERR_RplProfileInfoCorrupted:
      return "Profile record information has been corrupted.";
    case NERR_RplProfileNotFound:
      return "Profile record was not found.";
    case NERR_RplProfileNameUnavailable:
      return "Profile name is in use by some other profile.";
    case NERR_RplProfileNotEmpty:
      return "There are workstations using this profile.";
    case NERR_RplConfigInfoCorrupted:
      return "Configuration record information has been corrupted.";
    case NERR_RplConfigNotFound:
      return "Configuration record was not found.";
    case NERR_RplAdapterInfoCorrupted:
      return "Adapter ID record information has been corrupted.";
    case NERR_RplInternal:
      return "An internal service error has occurred.";
    case NERR_RplVendorInfoCorrupted:
      return "Vendor ID record information has been corrupted.";
    case NERR_RplBootInfoCorrupted:
      return "Boot block record information has been corrupted.";
    case NERR_RplWkstaNeedsUserAcct:
      return "The user account for this workstation record is missing.";
    case NERR_RplNeedsRPLUSERAcct:
      return "The RPLUSER local group could not be found.";
    case NERR_RplBootNotFound:
      return "Boot block record was not found.";
    case NERR_RplIncompatibleProfile:
      return "Chosen profile is incompatible with this workstation.";
    case NERR_RplAdapterNameUnavailable:
      return "Chosen network adapter ID is in use by some other workstation.";
    case NERR_RplConfigNotEmpty:
      return "There are profiles using this configuration.";
    case NERR_RplBootInUse:
      return "There are workstations, profiles, or configurations using this boot block.";
    case NERR_RplBackupDatabase:
      return "Service failed to backup Remoteboot database.";
    case NERR_RplAdapterNotFound:
      return "Adapter record was not found.";
    case NERR_RplVendorNotFound:
      return "Vendor record was not found.";
    case NERR_RplVendorNameUnavailable:
      return "Vendor name is in use by some other vendor record.";
    case NERR_RplBootNameUnavailable:
      return "(boot name, vendor ID) is in use by some other boot block record.";
    case NERR_RplConfigNameUnavailable:
      return "Configuration name is in use by some other configuration.";
    case NERR_DfsInternalCorruption:
      return "The internal database maintained by the Dfs service is corrupt.";
    case NERR_DfsVolumeDataCorrupt:
      return "One of the records in the internal Dfs database is corrupt.";
    case NERR_DfsNoSuchVolume:
      return "There is no DFS name whose entry path matches the input Entry Path.";
    case NERR_DfsVolumeAlreadyExists:
      return "A root or link with the given name already exists.";
    case NERR_DfsAlreadyShared:
      return "The server share specified is already shared in the Dfs.";
    case NERR_DfsNoSuchShare:
      return "The indicated server share does not support the indicated DFS namespace.";
    case NERR_DfsNotALeafVolume:
      return "The operation is not valid on this portion of the namespace.";
    case NERR_DfsLeafVolume:
      return "The operation is not valid on this portion of the namespace.";
    case NERR_DfsVolumeHasMultipleServers:
      return "The operation is ambiguous because the link has multiple servers.";
    case NERR_DfsCantCreateJunctionPoint:
      return "Unable to create a link.";
    case NERR_DfsServerNotDfsAware:
      return "The server is not Dfs Aware.";
    case NERR_DfsBadRenamePath:
      return "The specified rename target path is invalid.";
    case NERR_DfsVolumeIsOffline:
      return "The specified DFS link is offline.";
    case NERR_DfsNoSuchServer:
      return "The specified server is not a server for this link.";
    case NERR_DfsCyclicalName:
      return "A cycle in the Dfs name was detected.";
    case NERR_DfsNotSupportedInServerDfs:
      return "The operation is not supported on a server-based Dfs.";
    case NERR_DfsDuplicateService:
      return "This link is already supported by the specified server-share.";
    case NERR_DfsCantRemoveLastServerShare:
      return "Can't remove the last server-share supporting this root or link.";
    case NERR_DfsVolumeIsInterDfs:
      return "The operation is not supported for an Inter-DFS link.";
    case NERR_DfsInconsistent:
      return "The internal state of the Dfs Service has become inconsistent.";
    case NERR_DfsServerUpgraded:
      return "The Dfs Service has been installed on the specified server.";
    case NERR_DfsDataIsIdentical:
      return "The Dfs data being reconciled is identical.";
    case NERR_DfsCantRemoveDfsRoot:
      return "The DFS root cannot be deleted. Uninstall DFS if required.";
    case NERR_DfsChildOrParentInDfs:
      return "A child or parent directory of the share is already in a Dfs.";
    case NERR_DfsInternalError:
      return "Dfs internal error.";
      /* the following are not defined in mingw */
#if 0

    case NERR_SetupAlreadyJoined:
      return "This machine is already joined to a domain.";
    case NERR_SetupNotJoined:
      return "This machine is not currently joined to a domain.";
    case NERR_SetupDomainController:
      return "This machine is a domain controller and cannot be unjoined from a domain.";
    case NERR_DefaultJoinRequired:
      return "The destination domain controller does not support creating machine accounts in OUs.";
    case NERR_InvalidWorkgroupName:
      return "The specified workgroup name is invalid.";
    case NERR_NameUsesIncompatibleCodePage:
      return "The specified computer name is incompatible with the default language used on the domain controller.";
    case NERR_ComputerAccountNotFound:
      return "The specified computer account could not be found.";
    case NERR_PersonalSku:
      return "This version of Windows cannot be joined to a domain.";
    case NERR_PasswordMustChange:
      return "The password must change at the next logon.";
    case NERR_AccountLockedOut:
      return "The account is locked out.";
    case NERR_PasswordTooLong:
      return "The password is too long.";
    case NERR_PasswordNotComplexEnough:
      return "The password does not meet the complexity policy.";
    case NERR_PasswordFilterError:
      return "The password does not meet the requirements of the password filter DLLs.";
#endif

    }
  msg = strerror (error_number);
  if (msg == NULL)
    msg = "unknown";

  return msg;
#endif //DBUS_WINCE
}
