% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ExperimentRegistry.R
\name{makeExperimentRegistry}
\alias{makeExperimentRegistry}
\alias{ExperimentRegistry}
\title{ExperimentRegistry Constructor}
\usage{
makeExperimentRegistry(
  file.dir = "registry",
  work.dir = getwd(),
  conf.file = findConfFile(),
  packages = character(0L),
  namespaces = character(0L),
  source = character(0L),
  load = character(0L),
  seed = NULL,
  make.default = TRUE
)
}
\arguments{
\item{file.dir}{[\code{character(1)}]\cr
  Path where all files of the registry are saved.
  Default is directory \dQuote{registry} in the current working directory.
  The provided path will get normalized unless it is given relative to the home directory
  (i.e., starting with \dQuote{~}). Note that some templates do not handle relative paths well.

  If you pass \code{NA}, a temporary directory will be used.
  This way, you can create disposable registries for \code{\link{btlapply}} or examples.
  By default, the temporary directory \code{\link[base]{tempdir}()} will be used.
  If you want to use another directory, e.g. a directory which is shared between nodes,
  you can set it in your configuration file by setting the variable \code{temp.dir}.}

\item{work.dir}{[\code{character(1)}]\cr
  Working directory for R process for running jobs.
  Defaults to the working directory currently set during Registry construction (see \code{\link[base]{getwd}}).
  \code{loadRegistry} uses the stored \code{work.dir}, but you may also explicitly overwrite it,
  e.g., after switching to another system.

  The provided path will get normalized unless it is given relative to the home directory
  (i.e., starting with \dQuote{~}). Note that some templates do not handle relative paths well.}

\item{conf.file}{[\code{character(1)}]\cr
  Path to a configuration file which is sourced while the registry is created.
  In the configuration file you can define how \pkg{batchtools} interacts with the system via \code{\link{ClusterFunctions}}.
  Separating the configuration of the underlying host system from the R code allows to easily move computation to another site.

  The file lookup is implemented in the internal (but exported) function \code{findConfFile} which returns the first file found of the following candidates:
  \enumerate{
   \item{File \dQuote{batchtools.conf.R} in the path specified by the environment variable \dQuote{R_BATCHTOOLS_SEARCH_PATH}.}
   \item{File \dQuote{batchtools.conf.R} in the current working directory.}
   \item{File \dQuote{config.R} in the user configuration directory as reported by \code{rappdirs::user_config_dir("batchtools", expand = FALSE)} (depending on OS, e.g., on linux this usually resolves to \dQuote{~/.config/batchtools/config.R}).}
   \item{\dQuote{.batchtools.conf.R} in the home directory (\dQuote{~}).}
   \item{\dQuote{config.R} in the site config directory as reported by \code{rappdirs::site_config_dir("batchtools")} (depending on OS). This file can be used for admins to set sane defaults for a computation site.}
  }
  Set to \code{NA} if you want to suppress reading any configuration file.
  If a configuration file is found, it gets sourced inside the environment of the registry after the defaults for all variables are set.
  Therefore you can set and overwrite slots, e.g. \code{default.resources = list(walltime = 3600)} to set default resources or \dQuote{max.concurrent.jobs} to
  limit the number of jobs allowed to run simultaneously on the system.}

\item{packages}{[\code{character}]\cr
Packages that will always be loaded on each node.
Uses \code{\link[base]{require}} internally.
Default is \code{character(0)}.}

\item{namespaces}{[\code{character}]\cr
Same as \code{packages}, but the packages will not be attached.
Uses \code{\link[base]{requireNamespace}} internally.
Default is \code{character(0)}.}

\item{source}{[\code{character}]\cr
Files which should be sourced on the slaves prior to executing a job.
Calls \code{\link[base]{sys.source}} using the \code{\link[base]{.GlobalEnv}}.}

\item{load}{[\code{character}]\cr
Files which should be loaded on the slaves prior to executing a job.
Calls \code{\link[base]{load}} using the \code{\link[base]{.GlobalEnv}}.}

\item{seed}{[\code{integer(1)}]\cr
Start seed for jobs. Each job uses the (\code{seed} + \code{job.id}) as seed.
Default is a random integer between 1 and 32768.
Note that there is an additional seeding mechanism to synchronize instantiation of
\code{\link{Problem}}s in a \code{\link{ExperimentRegistry}}.}

\item{make.default}{[\code{logical(1)}]\cr
If set to \code{TRUE}, the created registry is saved inside the package
namespace and acts as default registry. You might want to switch this
off if you work with multiple registries simultaneously.
Default is \code{TRUE}.}
}
\value{
[\code{ExperimentRegistry}].
}
\description{
\code{makeExperimentRegistry} constructs a special \code{\link{Registry}} which
is suitable for the definition of large scale computer experiments.

Each experiments consists of a \code{\link{Problem}} and an \code{\link{Algorithm}}.
These can be parametrized with \code{\link{addExperiments}} to actually define computational
jobs.
}
\examples{
\dontshow{ batchtools:::example_push_temp(1) }
tmp = makeExperimentRegistry(file.dir = NA, make.default = FALSE)

# Definde one problem, two algorithms and add them with some parameters:
addProblem(reg = tmp, "p1",
  fun = function(job, data, n, mean, sd, ...) rnorm(n, mean = mean, sd = sd))
addAlgorithm(reg = tmp, "a1", fun = function(job, data, instance, ...) mean(instance))
addAlgorithm(reg = tmp, "a2", fun = function(job, data, instance, ...) median(instance))
ids = addExperiments(reg = tmp, list(p1 = data.table::CJ(n = c(50, 100), mean = -2:2, sd = 1:4)))

# Overview over defined experiments:
tmp$problems
tmp$algorithms
summarizeExperiments(reg = tmp)
summarizeExperiments(reg = tmp, by = c("problem", "algorithm", "n"))
ids = findExperiments(prob.pars = (n == 50), reg = tmp)
print(unwrap(getJobPars(ids, reg = tmp)))

# Submit jobs
submitJobs(reg = tmp)
waitForJobs(reg = tmp)

# Reduce the results of algorithm a1
ids.mean = findExperiments(algo.name = "a1", reg = tmp)
reduceResults(ids.mean, fun = function(aggr, res, ...) c(aggr, res), reg = tmp)

# Join info table with all results and calculate mean of results
# grouped by n and algorithm
ids = findDone(reg = tmp)
pars = unwrap(getJobPars(ids, reg = tmp))
results = unwrap(reduceResultsDataTable(ids, fun = function(res) list(res = res), reg = tmp))
tab = ljoin(pars, results)
tab[, list(mres = mean(res)), by = c("n", "algorithm")]
}
\concept{Registry Experiment}
