/* lowlevelfunc.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2004 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: lowlevelfunc.cc,v 1.31 2004/10/27 20:40:57 ralf Exp $ */

#include "lowlevelfunc.h"
#include <locale.h>

#ifdef USE_MEM_SYSTEM
typedef struct _safemem {void *ptr;
                         size_t size;
                         struct _safemem *next;} SafeMem;
SafeMem *firstmemstr;
unsigned long maxmemusage,memusage;
short memsystem_inited = 0;
void _memsysteminit();

#ifdef WANT_THREADS
#include "mutex.h"
MutEx memlock;
#endif

#endif

void *_allocsafe(size_t size)
{ /* gibt Speicher zurck oder beendet sofort das Programm */
  void *ptr;
#ifdef USE_MEM_SYSTEM
  SafeMem *tsmptr;
#endif

  ptr=(void*)malloc(size);
  if(ptr==NULL) {
    fprintf( stderr, "Worker Error:No Memory!!!!Aborting\n" );
    exit(1);
  }
#ifdef USE_MEM_SYSTEM
#ifdef WANT_THREADS
  memlock.lock();
#endif
  if ( memsystem_inited != 1 ) _memsysteminit();
  tsmptr=(SafeMem*)malloc(sizeof(SafeMem));
  tsmptr->ptr=ptr;
  tsmptr->size=size;
  tsmptr->next = firstmemstr;
  firstmemstr=tsmptr;
  memusage+=size;
  if(memusage>maxmemusage) maxmemusage=memusage;
#ifdef WANT_THREADS
  memlock.unlock();
#endif
#endif
  return ptr;
}

void waittime(unsigned long msec)
{
#ifdef HAVE_NANOSLEEP
  struct timespec wt;
  wt.tv_nsec=msec*1000000;
  wt.tv_sec=0;
  nanosleep( &wt, NULL );
#else

#ifdef HAVE_USLEEP
  usleep( msec * 1000 );
#else

  /* configure doesn't allow continue without nanosleep and usleep
   * but worker could run without a working waittime (but not very nice)
   * just throw a warning
   */
#warning empty wait-function

#endif /* HAVE_USLEEP */

#endif /* HAVE_NANOSLEEP */
}

#ifdef USE_MEM_SYSTEM
void _freesafe(void *p)
{
  SafeMem *tsmptr,*found,*prefound;

#ifdef WANT_THREADS
  memlock.lock();
#endif
  if ( memsystem_inited != 1 ) _memsysteminit();
  tsmptr=firstmemstr;
  found=NULL;
  prefound=NULL;
  while(tsmptr!=NULL) {
    if(tsmptr->ptr==p) {
      found=tsmptr;
      break;
    }
    prefound=tsmptr;
    tsmptr=tsmptr->next;
  }
  if(found!=NULL) {
    memusage-=found->size;
    if(firstmemstr==found) {
      firstmemstr=found->next;
      free(found);
    } else {
      prefound->next=found->next;
      free(found);
    }
  } else {
    fprintf( stderr, "Freigabe eines unauthorisierten Speichers!\n");
  }
#ifdef WANT_THREADS
  memlock.unlock();
#endif
  free(p);
}

void _memsystemcheck()
{
  SafeMem *tsmptr;
  int nr;

  if ( memsystem_inited != 1 ) return;
#ifdef WANT_THREADS
  memlock.lock();
#endif
  tsmptr=firstmemstr;
  nr=0;
  while(tsmptr!=NULL) {
    nr++;
/*    printf("lasse nr %d free:\n",nr);
    free(tsmptr->ptr);*/
    tsmptr=tsmptr->next;
  }
#ifdef DEBUG
  printf("Es gibt noch %d nicht freigelassene Speicherstcke!\n",nr);
  printf("Maximal benutzter Speicher:%ld Bytes\n",maxmemusage);
  printf("Noch benutzter Speicher:%ld Bytes\n",memusage);
#else
  if(nr>0) {
    printf("Es gibt noch %d nicht freigelassene Speicherstcke!\n",nr);
  }
#endif
#ifdef WANT_THREADS
  memlock.unlock();
#endif
}

void _memsysteminit()
{
  printf("Init Mem-System\n");
  firstmemstr=NULL;
  maxmemusage=0;
  memusage=0;
  memsystem_inited = 1;
  atexit( _memsystemcheck );
}
#endif

char *dupstring(const char *str)
{
  char *tstr;
  tstr=(char*)_allocsafe(strlen(str)+1);
  strcpy(tstr,str);
  return tstr;
}

int WriteLong2Str(char *buffer,loff_t size)
{
  int i=0,digit,j;
  loff_t x=size;
  do {
    digit = abs( (int)( x % 10 ) );
    x/=10;
    buffer[i++]='0'+digit;
  } while(x!=0);
  if(size<0) buffer[i++]='-';
  // now reverse the buffer for human readable
  for(j=0;j<(i/2);j++) {
    digit=buffer[i-1-j];
    buffer[i-1-j]=buffer[j];
    buffer[j]=digit;
  }
  return i;
}

const char *GetLong2Str( loff_t size )
{
  static char nrbuf[ A_BYTESFORNUMBER( size ) ];

  nrbuf[ WriteLong2Str( nrbuf, size ) ] = '\0';
  return nrbuf;
}

int MakeLong2NiceStr(char *buffer,loff_t size)
{
  int s = NiceLongSize( size );
  char ch;
  int l,i,pos;
  static char own_thousands_sep = '\0';
  struct lconv *current_locale;

  if ( own_thousands_sep == '\0' ) {
    // thousands separator isn't set so try to determine it
    current_locale = localeconv();
    /* check if the locale defines thousands_sep */
    if ( ( current_locale->thousands_sep != NULL ) &&
	 ( *( current_locale->thousands_sep ) != 0 ) ) {
      /* yes, it does. just copy the char. */
      own_thousands_sep = *( current_locale->thousands_sep );
    } else {
      // default is the dot
      own_thousands_sep = '.';
    }
  }

  // first write the normal str into buf
  l=WriteLong2Str(buffer,size);
  // start at the right of needed room (the lowest number)
  pos=s-1;
  // for all chars in the normal str
  for(i=0;i<l;i++) {
    // save the char
    ch=buffer[l-1-i];
    // when taking over the thousand/million/billion number write the
    // own_thousands_sep (depending on the current locale) into the dest
    // this will NOT overwrite any important information because
    // 1.we saved the ch we will write
    // 2.at the end we will be at pos 0 (pos starts at s-1)
    //   and if we have all points written, we will take over the ch
    if ( ( ( i == 3 ) || ( i == 6 ) || ( i == 9 ) ) &&
	 ( isdigit( ch ) ) ) buffer[pos--] = own_thousands_sep;
    buffer[pos--]=ch;
  }
  return s;
}

int NiceLongSize(loff_t value)
{
  int size;
  loff_t x;
  x=value;
  size=1;
  
  if(value<0) {
    size++;
    if(x<-999) size++;
    if(x<-999999) size++;
    if(x<-999999999) size++;
    while(x<-9) {
      size++;
      x=x/10;
    }
  } else {
    if(x>999) size++;
    if(x>999999) size++;
    if(x>999999999) size++;
    while(x>9) {
      size++;
      x=x/10;
    }
  }
  return size;
}

char *catstring(const char *str1,const char *str2)
{
  char *tstr;
  tstr=(char*)_allocsafe(strlen(str1)+strlen(str2)+1);
  strcpy(tstr,str1);
  strcpy(tstr+strlen(str1),str2);
  return tstr;
}

char *shrinkstring(const char *str,int maxlen)
{
  int len=strlen(str),llen,rlen;
  char *newstr;
  if(len<=maxlen) return dupstring(str);
  if(maxlen>4) {
    newstr=(char*)_allocsafe(maxlen+1);
    llen=(maxlen-3)/2;
    rlen=maxlen-llen-3;
    strncpy( newstr, str, llen );
    strcpy( newstr + llen, "..." );
    strcpy( newstr + llen + 3, str + len - rlen );
  } else {
    newstr=dupstring(str);
    newstr[(maxlen>0)?maxlen:0]=0;
  }
  return newstr;
}

double diffgtod( struct timeval *tv1, struct timeval *tv0 )
{
  long s, us;

  if ( ( tv1 == NULL ) || ( tv0 == NULL ) ) return 0.0;
  
  s = tv1->tv_sec - tv0->tv_sec;
  us = tv1->tv_usec - tv0->tv_usec;
  
  return ((double)s) * 1000000.0 + (double)us;
}

long ldiffgtod( struct timeval *tv1, struct timeval *tv0 )
{
  long s, us;

  if ( ( tv1 == NULL ) || ( tv0 == NULL ) ) return 0;
  
  s = tv1->tv_sec - tv0->tv_sec;
  us = tv1->tv_usec - tv0->tv_usec;
  
  return s * 1000000 + us;
}

long ldiffgtod_m( const struct timeval *tv1, const struct timeval *tv0 )
{
  long s, us;

  if ( ( tv1 == NULL ) || ( tv0 == NULL ) ) return 0;
  
  s = tv1->tv_sec - tv0->tv_sec;
  us = tv1->tv_usec - tv0->tv_usec;
  
  return s * 1000 + us / 1000;
}

/* this function calculate the quote mode at the end of str
   0 = no quotes
   1 = single quote
   2 = double quote
   -1 = error
*/
int AGUIX_getQuoteMode( const char *str )
{
  int quotemode = 0; /* 0 means no quote
			1 means in single quotes
			2 means in double quotes */
  int i, len;

  if ( str == NULL ) return -1;

  len = strlen( str );

  for ( i = 0; i < len; i++ ) {
    switch ( quotemode ) {
    case 1:
      // even backslash doesn't matter in single-quotes so just check for closing quote
      if ( str[i] == '\'' ) {
	// you cannot do single quote inside double quote so we return to "no quote"
	quotemode = 0;
      }
      break;
    case 2:
      // this is more complicated
      // the backslash doesn't protect everything just special characters
      // anyway if the following char isn't special we don't care on it
      // and if it's a special we also don't care because it's backslashed
      // so it's safe to skip next char in any case
      if ( str[i] == '\\' ) {
	i++;
      } else if ( str[i] == '"' ) {
	// remember that we skiped this char if it was backslashed
	// so we really have to leave quote mode and for the same reason
	// as above it's not possibly to do double quote in single quote
	// we can go to "no quotes" mode
	quotemode = 0;
      }
      break;
    default:
      if ( str[i] == '\\' ) {
	// not in quotes => ignore next
	i++;
      } else if ( str[i] == '"' ) {
	// double quote begin
	quotemode = 2;
      } else if ( str[i] == '\'' ) {
	// single quote begin
	quotemode = 1;
      }
      break;
    }
  }
  return quotemode;
}

/* this function will cat 2 strings:
   str1 have to be a string which STARTS unquoted (can begin with quotes
   but it's interpreted as beginning quotes not ending for a another unknown
   string!
   str2 is an untrusted string and will be single quoted in any case
   depending on quote-status at the end of str1, str2 is correctly
   quoted in single-quotes so it can be used in shell-script to avoid problems
   for filenames like test`rm -rf /`
   
   str1 is not changed but between str1 and str2 and at the end
   I add the correct quotes
   str2 can be changed (single quotes become '\'' (to avoid any known and unknown shell actions) )

   return string will always end in same quotes like str1
*/
char *AGUIX_catTrustedAndUnTrusted( const char *str1, const char *str2 )
{
  int quotemode = 0; /* 0 means no quote
			1 means in single quotes
			2 means in double quotes */
  int i, o, lenstr1, lenstr2, sqcount;
  char *newstr2, *newstr, *tstr1;

  if ( ( str1 == NULL ) || ( str2 == NULL ) ) return NULL;

  // first fix backslashed str1 because it a last backslash
  // will protect the quoting for str2
  tstr1 = AGUIX_fixBackslashed( str1 );
  if ( tstr1 == NULL ) return NULL;

  lenstr1 = strlen( tstr1 );

  quotemode = AGUIX_getQuoteMode( tstr1 );
  if ( quotemode < 0 ) return NULL;

  // now we know in which quote mode str1 is at the end
  // se we can do the following:
  // in single quotes we have just to replace singlequotes with '\''
  // in double quotes we have to close double close and add single quoted filename
  // in no quotes we just have to add single quoted filename

  // since filename will always be inside single quotes first replace all single quotes
  // in filename with '\''
  lenstr2 = strlen( str2 );
  sqcount = 0;
  for ( i = 0; i < lenstr2; i++ ) {
    if ( str2[i] == '\'' ) sqcount++;
  }
  // for every single quote we need 3 additional chars
  newstr2 = (char*)_allocsafe( lenstr2 + sqcount * 3 + 1 );
  for ( i = 0, o = 0; i < lenstr2; i++ ) {
    if ( str2[i] == '\'' ) {
      newstr2[o++] = '\'';
      newstr2[o++] = '\\';
      newstr2[o++] = '\'';
      newstr2[o++] = '\'';
    } else {
      newstr2[o++] = str2[i];
    }
  }
  newstr2[o] = '\0';

  // newfilename is now okay
  // space needed for newstr1:
  // str1len
  // perhaps a closing double quote and a beginning single quote
  // strlen( newfilename )
  // perhaps a closing single quote and beginning double quote
  newstr = (char*)_allocsafe( lenstr1 + 2 + strlen( newstr2 ) + 2 + 1 );
  switch ( quotemode ) {
  case 1:
    // single quote so no need for quote changing
    // remember newfilename is correctly quoted to live inside
    // single quote
    sprintf( newstr, "%s%s", tstr1, newstr2 );
    break;
  case 2:
    // double quotes so close double quotes, begin single quote
    // and after newfilename close single quote and start double quote
    sprintf( newstr, "%s\"'%s'\"", tstr1, newstr2 );
    break;
  default:
    // no quote so just start single quote and after newfilename single quote
    sprintf( newstr, "%s'%s'", tstr1, newstr2 );
    break;
  }
  _freesafe( newstr2 );
  _freesafe( tstr1 );
  return newstr;
}

/* this function replace all single quotes with '\'' so it can be
   used in single quotes */
char *AGUIX_prepareForSingleQuote( const char *str1 )
{
  int i, o, lenstr1, sqcount;
  char *newstr1;

  if ( str1 == NULL ) return NULL;

  // single quotes in filename with '\''
  lenstr1 = strlen( str1 );
  sqcount = 0;
  for ( i = 0; i < lenstr1; i++ ) {
    if ( str1[i] == '\'' ) sqcount++;
  }
  // for every single quote we need 3 additional chars
  newstr1 = (char*)_allocsafe( lenstr1 + sqcount * 3 + 1 );
  for ( i = 0, o = 0; i < lenstr1; i++ ) {
    if ( str1[i] == '\'' ) {
      newstr1[o++] = '\'';
      newstr1[o++] = '\\';
      newstr1[o++] = '\'';
      newstr1[o++] = '\'';
    } else {
      newstr1[o++] = str1[i];
    }
  }
  newstr1[o] = '\0';
  return newstr1;
}

/* this function will remove all quotes from the string
   rules:
   single quotes: allowed, no special character
   double quotes: allowed, backslash escapes next character
   single and double quotes can be mixed
*/
char *AGUIX_unquoteString( const char *str1 )
{
  char *newstr;
  int quotemode = 0; /* 0 means no quote
			1 means in single quotes
			2 means in double quotes */
  int i, o, len;

  if ( str1 == NULL ) return NULL;
  newstr = (char*)_allocsafe( strlen( str1 ) + 1 );

  len = strlen( str1 );

  for ( i = 0, o = 0; i < len; i++ ) {
    switch ( quotemode ) {
    case 1:
      // even backslash doesn't matter in single-quotes so just check for closing quote
      if ( str1[i] == '\'' ) {
	// you cannot do single quote inside double quote so we return to "no quote"
	quotemode = 0;
      } else {
	newstr[o++] = str1[i];
      }
      break;
    case 2:
      // this is more complicated
      // the backslash doesn't protect everything just special characters
      // anyway if the following char isn't special we don't care on it
      // and if it's a special we also don't care because it's backslashed
      // so it's safe to skip next char in any case
      if ( str1[i] == '\\' ) {
	// only backslash $, `, ", <backslash
	// (just like shell (at least bash))
	switch( str1[i + 1] ) {
	case '$':
	case '`':
	case '"':
	case '\\':
	  i++;
	default:
	  newstr[o++] = str1[i];
	  break;
	}
      } else if ( str1[i] == '"' ) {
	// remember that we skiped this char if it was backslashed
	// so we really have to leave quote mode and for the same reason
	// as above it's not possibly to do double quote in single quote
	// we can go to "no quotes" mode
	quotemode = 0;
      } else {
	newstr[o++] = str1[i];
      }
      break;
    default:
      if ( str1[i] == '\\' ) {
	// backslash in nonquoted env escapes everything
	i++;
	newstr[o++] = str1[i];
      } else if ( str1[i] == '"' ) {
	// double quote begin
	quotemode = 2;
      } else if ( str1[i] == '\'' ) {
	// single quote begin
	quotemode = 1;
      } else {
	newstr[o++] = str1[i];
      }
      break;
    }
  }
  newstr[o] = '\0';

  return newstr;
}

/*
 * this function will cat 2 strings
 * the first one can have single and double quotes
 * the second will be protected according to the quote-mode at the end
 * of str1
 *
 * what's done here is not enough for direct shell output because special
 * shell stuff like $... is not protected
 * use catTrustedAndUnTrusted instead
 *
 * this function is enough for secureCommand... from execlass
 *
 * following rules apply:
 * 1.str1 not quoted at the end: whitespaces, single,double quotes and backslash
 *   are protected with backslash
 * 2.str1 ends with single quote: single quotes are replaced with '\''
 * 3.str1 ends with double quote: double quotes and backslash are protected with backslash
 */
char *AGUIX_catQuotedAndUnQuoted( const char *str1, const char *str2 )
{
  int quotemode = 0; /* 0 means no quote
			1 means in single quotes
			2 means in double quotes */
  int i, o, lenstr1, lenstr2, newlen;
  char *newstr, *tstr1;

  if ( ( str1 == NULL ) || ( str2 == NULL ) ) return NULL;

  // first fix backslashed str1
  tstr1 = AGUIX_fixBackslashed( str1 );
  if ( tstr1 == NULL ) return NULL;

  lenstr1 = strlen( tstr1 );

  quotemode = AGUIX_getQuoteMode( tstr1 );
  if ( quotemode < 0 ) return NULL;

  /* to know the length of the new string
   * count all whitespaces, single and double quotes in it
   * and add the needed space to protect them
   */
  lenstr2 = strlen( str2 );
  newlen = 0;
  for ( i = 0; i < lenstr2; i++, newlen++ ) {
    if ( str2[i] == '\'' ) {
      newlen += 3; /* in quotemode 1 I have to protect it
		    * with '\'' so I need 3 additional chars */
    } else if ( str2[i] == '\"' ) {
      newlen++;
    } else if ( str2[i] == '\\' ) {
      newlen++;
    } else if ( str2[i] == '$' ) {
      newlen++;
    } else if ( str2[i] == '&' ) {
      newlen++;
    } else if ( str2[i] == '|' ) {
      newlen++;
    } else if ( str2[i] == ';' ) {
      newlen++;
    } else if ( str2[i] == '>' ) {
      newlen++;
    } else if ( str2[i] == '<' ) {
      newlen++;
    } else if ( isspace( str2[i] ) ) {
      newlen++;
    }
  }

  newstr = (char*)_allocsafe( lenstr1 + newlen + 1 );
  strcpy( newstr, tstr1 );
  o = lenstr1;

  for ( i = 0; i < lenstr2; i++ ) {
    switch ( quotemode ) {
    case 1:
      if ( str2[i] == '\'' ) {
	newstr[o++] = '\'';
	newstr[o++] = '\\';
	newstr[o++] = '\'';
	newstr[o++] = '\'';
      } else {
	newstr[o++] = str2[i];
      }
      break;
    case 2:
      if ( str2[i] == '"' ) {
	newstr[o++] = '\\';
	newstr[o++] = '"';
      } else if ( str2[i] == '\\' ) {
	newstr[o++] = '\\';
	newstr[o++] = '\\';
      } else if ( str2[i] == '$' ) {
	newstr[o++] = '\\';
	newstr[o++] = '$';
      } else {
	newstr[o++] = str2[i];
      }
      break;
    default:
      if ( str2[i] == '"' ) {
	newstr[o++] = '\\';
	newstr[o++] = '"';
      } else if ( str2[i] == '\'' ) {
	newstr[o++] = '\\';
	newstr[o++] = '\'';
      } else if ( str2[i] == '\\' ) {
	newstr[o++] = '\\';
	newstr[o++] = '\\';
      } else if ( str2[i] == '$' ) {
	newstr[o++] = '\\';
	newstr[o++] = '$';
      } else if ( str2[i] == '&' ) {
	newstr[o++] = '\\';
	newstr[o++] = '&';
      } else if ( str2[i] == '|' ) {
	newstr[o++] = '\\';
	newstr[o++] = '|';
      } else if ( str2[i] == ';' ) {
	newstr[o++] = '\\';
	newstr[o++] = ';';
      } else if ( str2[i] == '<' ) {
	newstr[o++] = '\\';
	newstr[o++] = '<';
      } else if ( str2[i] == '>' ) {
	newstr[o++] = '\\';
	newstr[o++] = '>';
      } else if ( isspace( str2[i] ) ) {
	newstr[o++] = '\\';
	newstr[o++] = str2[i];
      } else {
	newstr[o++] = str2[i];
      }
      break;
    }
  }
  newstr[o] = '\0';

  _freesafe( tstr1 );

  return newstr;
}

/*
 * this function will remove a backslash at the end
 * if this backslash
 * 1.exists
 * 2.is not inside single quotes (backslash doesn't matter there)
 * 3.the backslash is not backslashed (\\)
 */
char *AGUIX_fixBackslashed( const char *str1 )
{
  char *newstr;
  int lenstr1, bscount, i;
  int quotemode;

  if ( str1 == NULL ) return NULL;

  lenstr1 = strlen( str1 );
  if ( lenstr1 < 1 ) return dupstring( str1 );

  bscount = 0;
  for ( i = lenstr1 - 1; i >= 0; i-- ) {
    if ( str1[ i ] == '\\' ) {
      bscount++;
    } else {
      break;
    }
  }

  // even backslash count is okay because two
  // protects a backslash

  quotemode = AGUIX_getQuoteMode( str1 );
  if ( quotemode < 0 ) return NULL;

  newstr = dupstring( str1 );
  if ( quotemode != 1 ) {
    // backslash is no special character in single quotes
    if ( ( bscount % 2 ) == 1 ) {
      // clear last backslash
      newstr[ lenstr1 - 1 ] = '\0';
    }
  }
  return newstr;
}
