/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: vtkProgrammableSource.h,v $

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkProgrammableSource - generate source dataset via a user-specified function
// .SECTION Description
// vtkProgrammableSource is a source object that is programmable by the
// user. To use this object, you must specify a function that creates the
// output.  It is possible to generate an output dataset of any (concrete) 
// type; it is up to the function to properly initialize and define the 
// output. Typically, you use one of the methods to get a concrete output 
// type (e.g., GetPolyDataOutput() or GetStructuredPointsOutput()), and 
// then manipulate the output in the user-specified function.
//
// Example use of this include writing a function to read a data file or
// interface to another system. (You might want to do this in favor of
// deriving a new class.) Another important use of this class is that it
// allows users of interpreters (e.g., Tcl or Java) the ability to write
// source objects without having to recompile C++ code or generate new
// libraries.
// .SECTION See Also
// vtkProgrammableFilter vtkProgrammableAttributeDataFilter
// vtkProgrammableDataObjectSource

#ifndef __vtkProgrammableSource_h
#define __vtkProgrammableSource_h

#include "vtkSource.h"

class vtkPolyData;
class vtkStructuredPoints;
class vtkStructuredGrid;
class vtkUnstructuredGrid;
class vtkRectilinearGrid;

class VTK_GRAPHICS_EXPORT vtkProgrammableSource : public vtkSource
{
public:
  static vtkProgrammableSource *New();
  vtkTypeRevisionMacro(vtkProgrammableSource,vtkSource);

  // Description:
  // Specify the function to use to generate the source data. Note
  // that the function takes a single (void *) argument.
  void SetExecuteMethod(void (*f)(void *), void *arg);

  // Description:
  // Set the arg delete method. This is used to free user memory.
  void SetExecuteMethodArgDelete(void (*f)(void *));

  // Description:
  // Get the output as a concrete type. This method is typically used by the
  // writer of the source function to get the output as a particular type
  // (i.e., it essentially does type casting). It is the users responsibility
  // to know the correct type of the output data.
  vtkPolyData *GetPolyDataOutput();

  // Description:
  // Get the output as a concrete type.
  vtkStructuredPoints *GetStructuredPointsOutput();

  // Description:
  // Get the output as a concrete type.
  vtkStructuredGrid *GetStructuredGridOutput();

  // Description:
  // Get the output as a concrete type.
  vtkUnstructuredGrid *GetUnstructuredGridOutput();

  // Description:
  // Get the output as a concrete type.
  vtkRectilinearGrid *GetRectilinearGridOutput();

  void UpdateInformation();
  void UpdateData(vtkDataObject *output);

protected:
  vtkProgrammableSource();
  ~vtkProgrammableSource();

  void Execute();

  void (*ExecuteMethod)(void *); //function to invoke
  void (*ExecuteMethodArgDelete)(void *);
  void *ExecuteMethodArg;  

  vtkTimeStamp ExecuteTime;
private:
  vtkProgrammableSource(const vtkProgrammableSource&);  // Not implemented.
  void operator=(const vtkProgrammableSource&);  // Not implemented.
};

#endif

