/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef RENDERINGSPIN_H
#define RENDERINGSPIN_H

#include <visu_rendering.h>
#include <visu_data.h>
#include <visu_object.h>
#include <visu_tools.h>

/**
 * SpinDrawingPolicy:
 * @policyAlwaysSpin: Arrows are drawn whatever the modulus value.
 * @policyHideNullSpin: Spin with a null modulus are hidden.
 * @policyAtomicNullSpin: Follow atomic rendering for null modulus.
 * @policyNbModes: a flag to count the number of modes.
 *
 * Different policy to render the spin when the modulus is null. This
 * policy is applied for all #VisuElement.
 */
typedef enum
  {
    policyAlwaysSpin,    
    policyHideNullSpin,  
    policyAtomicNullSpin,
    policyNbModes
  } SpinDrawingPolicy;
/**
 * SpinModulusPolicy:
 * @policyNoneModulus: arrows have all the same size, whatever the
 * modulus value.
 * @policyPerTypeModulus: arrows are scaled per node type.
 * @policyGlobalModulus: arrows are scaled globaly.
 * @policyNbModulusModes: a flag to count the number of modes.
 *
 * Different policy to render the spin depending on the modulus.
 */
typedef enum
  {
    policyNoneModulus,    
    policyPerTypeModulus,  
    policyGlobalModulus,
    policyNbModulusModes
  } SpinModulusPolicy;

/**
 * VisuSpinShapeId:
 * @VISU_RENDERING_SPIN_ARROW_SMOOTH: the shape is smooth and rounded ;
 * @VISU_RENDERING_SPIN_ARROW_SHARP: the shape is built on squares ;
 * @VISU_RENDERING_SPIN_ELLIPSOID: the shape is an ellipsoid ;
 * @VISU_RENDERING_SPIN_TORUS: the shape is a torus (direction of the arrow is
 * normal to the torus plane).
 * @nbSpinShapes: private.
 *
 * An identifier for the different shapes to draw elements.
 */
typedef enum
  {
    VISU_RENDERING_SPIN_ARROW_SMOOTH,
    VISU_RENDERING_SPIN_ARROW_SHARP,
    VISU_RENDERING_SPIN_ELLIPSOID,
    VISU_RENDERING_SPIN_TORUS,
    /*< private >*/
    nbSpinShapes
  } VisuSpinShapeId;

/**
 * SpinFileType:
 * @FILE_KIND_POSITION: define a position file ;
 * @FILE_KIND_SPIN: define a spin description file.
 *
 * These are keys for the storing of spin files in a VisuData object. 
 */
typedef enum
  {
    FILE_KIND_POSITION,
    FILE_KIND_SPIN
  } SpinFileType;

/**
 * SPINVALUES_ID:
 *
 * This flag should be used when creating a new spin rendering method
 * as the flag for the spin values as node properties (see
 * visu_node_property_newPointer()).
 */
#define SPINVALUES_ID     "spinRendering_values"
/**
 * SPINMAXMODULUS_ID:
 *
 * This flag should be used when creating a new spin rendering method
 * as the flag for the max modulus values as a #VisuNode property (see
 * visu_node_property_newPointer()).
 */
#define SPINMAXMODULUS_ID "spinRendering_maxModulus"

/**
 * VISU_RENDERING_SPIN_NAME:
 *
 * Public name of the spin rendering mode.
 */
#define VISU_RENDERING_SPIN_NAME "Spin visualisation"

/**
 * VISU_RENDERING_SPIN_TYPE:
 *
 * return the type of #VisuRenderingSpin.
 */
#define VISU_RENDERING_SPIN_TYPE	     (visu_rendering_spin_get_type ())
/**
 * VISU_RENDERING_SPIN:
 * @obj: a #GObject to cast.
 *
 * Cast the given @obj into #VisuRenderingSpin type.
 */
#define VISU_RENDERING_SPIN(obj)	     (G_TYPE_CHECK_INSTANCE_CAST(obj, VISU_RENDERING_SPIN_TYPE, VisuRenderingSpin))
/**
 * VISU_RENDERING_SPIN_CLASS:
 * @klass: a #GObjectClass to cast.
 *
 * Cast the given @klass into #VisuRenderingSpinClass.
 */
#define VISU_RENDERING_SPIN_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST(klass, VISU_RENDERING_SPIN_TYPE, VisuRenderingSpinClass))
/**
 * IS_VISU_RENDERING_SPIN_TYPE:
 * @obj: a #GObject to test.
 *
 * Test if the given @ogj is of the type of #VisuRenderingSpin object.
 */
#define IS_VISU_RENDERING_SPIN_TYPE(obj)    (G_TYPE_CHECK_INSTANCE_TYPE(obj, VISU_RENDERING_SPIN_TYPE))
/**
 * IS_VISU_RENDERING_SPIN_CLASS:
 * @klass: a #GObjectClass to test.
 *
 * Test if the given @klass is of the type of #VisuRenderingSpinClass class.
 */
#define IS_VISU_RENDERING_SPIN_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE(klass, VISU_RENDERING_SPIN_TYPE))
/**
 * VISU_RENDERING_SPIN_GET_CLASS:
 * @obj: a #GObject to get the class of.
 *
 * It returns the class of the given @obj.
 */
#define VISU_RENDERING_SPIN_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS(obj, VISU_RENDERING_SPIN_TYPE, VisuRenderingSpinClass))

/**
 * visu_rendering_spin_get_type:
 *
 * This method returns the type of #VisuRenderingSpin, use
 * VISU_RENDERING_SPIN_TYPE instead.
 *
 * Returns: the type of #VisuRenderingSpin.
 */
GType visu_rendering_spin_get_type(void);

/**
 * VisuRenderingSpinPriv:
 *
 * An opaque structure.
 */
typedef struct _VisuRenderingSpinPriv VisuRenderingSpinPriv;
typedef struct _VisuRenderingSpinClass VisuRenderingSpinClass;
/**
 * VisuRenderingSpin:
 *
 * An opaque structure.
 */
struct _VisuRenderingSpin
{
  VisuRendering method;

  VisuRenderingSpinPriv *priv;
};
typedef struct _VisuRenderingSpin VisuRenderingSpin;

VisuRenderingSpin* visu_rendering_spin_new();

/***********************/
/* Different resources */
/***********************/
const char* visu_renderingSpin_getShapeNameI18n(VisuSpinShapeId n);
SpinDrawingPolicy visu_renderingSpin_getHidingPolicyFromName(const char *name);
const char* visu_renderingSpin_getHidingPolicyName(SpinDrawingPolicy n);
const char* visu_renderingSpin_getHidingPolicyNameI18n(SpinDrawingPolicy n);

/**
 * SpinElementResources:
 * @spin_elementHatLength: the length of the pointing element ;
 * @spin_elementTailLength: the length of the tail ;
 * @spin_elementHatRadius: the raidus of the pointing element ;
 * @spin_elementTailRadius: the radius of the tail ;
 * @spin_elementHatColor: if TRUE, the pointing part use the color of the element ;
 * @spin_elementTailColor: if TRUE, the tail uses the color of the element ;
 * @spin_elementAAxis: the size of the A axis (elipsoid shape) ;
 * @spin_elementBAxis: the size of the B axis (elipsoid shape) ;
 * @spin_elementElipsoidColor: if TRUE, the elipsoid uses the color of the element ;
 * @spin_elementShape: an id to defined the shape (rounded arrow, elipsoid...) ;
 * @spin_nbElementResources: number of resources per element.
 * 
 * These are resources defined for each element. They can be accessed with
 * rspin_getElementResource() or rspin_getElementResource_boolean() and other
 * methods of the same kind.
 */
typedef enum
  {
    spin_elementHatLength,
    spin_elementTailLength,
    spin_elementHatRadius,
    spin_elementTailRadius,
    spin_elementHatColor,
    spin_elementTailColor,
    spin_elementAAxis,
    spin_elementBAxis,
    spin_elementElipsoidColor,
    spin_elementShape,
    spin_nbElementResources
  } SpinElementResources;

/**
 * rspin_getElementResource:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the resource (see #SpinElementResources) ;
 * @type: a location to store the type.
 *
 * This is a generic method to access resources per element. Use rspin_setElementResource_boolean()
 * is favored if the type of the value is known (boolean in this exemple).
 * If the returned value is not NULL the argument @type contains the type of the returned value.
 *
 * Return value: a pointer to the location where the value for the given global 
 * resource is stored.
 */
gpointer rspin_getElementResource(VisuElement *ele, SpinElementResources property, GType *type);
/**
 * rspin_setElementResource_boolean:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to set ;
 * @value: its value.
 *
 * This method is used to change element resources that are boolean.
 *
 * Returns: TRUE if the value was changed.
 */
gboolean rspin_setElementResource_boolean(VisuElement *ele, SpinElementResources property,
					  gboolean value);
/**
 * rspin_setElementResource_uint:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to set ;
 * @value: its value.
 *
 * This method is used to change element resources that are unsigned int.
 *
 * Returns: TRUE if the value was changed.
 */
gboolean rspin_setElementResource_uint(VisuElement *ele, SpinElementResources property,
				       guint value);
/**
 * rspin_setElementResource_float:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to set ;
 * @value: its value.
 *
 * This method is used to change element resources that are floating point.
 *
 * Returns: TRUE if the value was changed.
 */
gboolean rspin_setElementResource_float(VisuElement *ele, SpinElementResources property,
					gfloat value);
/**
 * rspin_getElementResource_boolean:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to get.
 *
 * This is the specific method to retrieve value of boolean element resources.
 *
 * Returns: the boolean value.
 */
gboolean rspin_getElementResource_boolean(VisuElement *ele, SpinElementResources property);
/**
 * rspin_getElementResource_uint:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to get.
 *
 * This is the specific method to retrieve value of unsigned int element resources.
 *
 * Returns: the unsigned int value.
 */
guint rspin_getElementResource_uint(VisuElement *ele, SpinElementResources property);
/**
 * rspin_getElementResource_float:
 * @ele: a pointer to a #VisuElement object ;
 * @property: the id of the property to get.
 *
 * This is the specific method to retrieve value of floating point element resources.
 *
 * Returns: the floating point value.
 */
gfloat rspin_getElementResource_float(VisuElement *ele, SpinElementResources property);

#endif
