/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses mél :
	BILLARD, non joignable par mél ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant à visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est régi par la licence CeCILL soumise au droit français et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffusée par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accéder à cet en-tête signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accepté les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "scalarFields.h"

#include <string.h>

#include <visu_tools.h>
#include <coreTools/toolFileFormat.h>
#include <coreTools/toolMatrix.h>
#include <openGLFunctions/text.h>

/**
 * SECTION:scalarFields
 * @short_description:Gives capabilities to load a scalar field.
 *
 * <para>A scalar field is represented by the given of datas on a
 * regular grid meshing the bounding box. Scalar field can be read
 * from several kind of files by adding load methods using
 * scalarFieldAdd_loadMethod(). The basic implementation gives access
 * to ASCII encoded files following a simple format.</para>
 *
 * <para>In coordination with #Plane and #Shade, scalar field can be
 * represented as coloured map calling scalarFieldDraw_map(). The
 * current implementation of interpolation is very limited since basic
 * linear approximation is used.</para>
 *
 * <para>If a structure file also contains a scalar field, when
 * loaded, it should add a #VisuData property called
 * SCALAR_FIELD_DEFINED_IN_STRUCT_FILE using
 * g_object_set_data(). Then V_Sim will be able to handle the
 * structure file as a density file also.</para>
 */

#define MESH_FLAG             "meshType"
#define MESH_FLAG_UNIFORM     "uniform"
#define MESH_FLAG_NON_UNIFORM "nonuniform"

/*
 * ScalarField_struct:
 * @filename: the path to the file from which the scalar field
 *            has been read ;
 * @commentary: a commentary read from the file (must be in UTF-8) ;
 * @box: description of the associated bounding box ;
 * @nElements: number of points in each direction ;
 * @data: the values ;
 * @min: the minimum value ;
 * @max: the maximum value ;
 * @options: a GList of #Option values.
 *
 * The structure used to store a scalar field.
 */
struct ScalarField_struct
{
  /* The name of the file, where the data were read from. */
  gchar *filename;

  /* An associated commentary. */
  gchar *commentary;

  /* Description of the box. */
  float box[6];
  float fromXYZtoReducedCoord[3][3];

  /* Number of elements in each directions [x, y, z]. */
  int nElements[3];

  /* Mesh. */
  double *meshx;
  double *meshy;
  double *meshz;

  /* Datas. */
  double ***data;

  /* Minimum and maximum values. */
  double min, max;

  /* Set to TRUE if the data can replicate by periodicity. */
  gboolean periodic;

  /* mesh type: uniform or non uniform */
  ScalarField_meshflag mesh_type;

  /* A GList to store some options (key, values) associated
     to the data. */
  GList *options;
};

/* Local variables. */
static GList *loadMethods;

/* Local methods. */
static gboolean scalarFieldLoad_fromAscii(const gchar *filename, GList **fieldList, GError **error,
					  GHashTable *table);
static gint compareLoadPriority(gconstpointer a, gconstpointer b);


ScalarField* scalarFieldNew(const gchar *filename)
{
  ScalarField *field;

  g_return_val_if_fail(filename && filename[0], (ScalarField*)0);

  field               = g_malloc(sizeof(ScalarField));
  field->nElements[0] = 0;
  field->nElements[1] = 0;
  field->nElements[2] = 0;
  field->filename     = g_strdup(filename);
  field->commentary   = (gchar*)0;
  field->meshx        = (double*)0;
  field->meshy        = (double*)0;
  field->meshz        = (double*)0;
  field->data         = (double***)0;
  field->min          = G_MAXFLOAT;
  field->max          = -G_MAXFLOAT;
  field->periodic     = FALSE;
  field->mesh_type    = uniform;
  field->options      = (GList*)0;

  return field;
}
void scalarFieldFree(ScalarField *field)
{
  int i, j;
  GList *tmplst;

  g_return_if_fail(field);

  if (field->filename)
    g_free(field->filename);

  if (field->commentary)
    g_free(field->commentary);

  if (field->meshx)
    g_free(field->meshx);

  if (field->meshy)
    g_free(field->meshy);

  if (field->meshz)
    g_free(field->meshz);

  if (field->data)
    {
      for (i = 0; i < field->nElements[0]; i++)
	{
	  for (j = 0; j < field->nElements[1]; j++)
	    g_free(field->data[i][j]);
	  g_free(field->data[i]);
	}
      g_free(field->data);
    }

  if (field->options)
    {
      tmplst = field->options;
      while(tmplst)
        {
          tool_option_free(tmplst->data);
          tmplst = g_list_next(tmplst);
        }
      g_list_free(field->options);
    }
}


gboolean scalarFieldLoad_fromFile(const gchar *filename, GList **fieldList,
				  GError **error, GHashTable *table)
{
  gboolean validFormat;
  GList *tmpLst;

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(*fieldList == (GList*)0, FALSE);
  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);

  /* Try all supported format. */
  validFormat = FALSE;
  tmpLst = loadMethods;
  while(tmpLst && !validFormat)
    {
      DBG_fprintf(stderr, "Scalar Fields : try to open the file as a '%s'.\n",
		  ((ScalarFieldLoadStruct*)tmpLst->data)->name);
      validFormat = ((ScalarFieldLoadStruct*)tmpLst->data)->load(filename, fieldList, error, table);
      if (!validFormat && *error)
        {
          g_error_free(*error);
          *error = (GError*)0;
        }
      tmpLst = g_list_next(tmpLst);
    }
  
  if (!validFormat)
    g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_UNKNOWN_FORMAT, 
		_("unknown density/potential format.\n")); 
  return validFormat;
}


static gboolean scalarFieldLoad_fromAscii(const gchar *filename, GList **fieldList,
					  GError **error, GHashTable *table _U_)
{
  FILE *in;
  char rep[MAX_LINE_LENGTH], flag[MAX_LINE_LENGTH];
  char format[MAX_LINE_LENGTH], period[MAX_LINE_LENGTH];
  char *feed;
  gchar *comment;
  int res, i, j, k;
  int size[3];
  double box[6];
  ScalarField *field;
  gboolean periodic;
  ScalarField_meshflag meshtype;

  g_return_val_if_fail(filename, FALSE);
  g_return_val_if_fail(*fieldList == (GList*)0, FALSE);
  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);

  DBG_fprintf(stderr, "ScalarField : try to read '%s' as a ASCII scalar"
	      " field data file.\n", filename);

  in = fopen(filename, "r");
  if (!in)
    {
      g_set_error(error, G_FILE_ERROR, G_FILE_ERROR_ACCES,
		  _("impossible to open the file.\n"));
      return FALSE;
    }

  /* 1st line (comment) */
  if (!fgets(rep, MAX_LINE_LENGTH, in))
    {
      fclose(in);
      return FALSE;
    }
  rep[strlen(rep)-1] = 0; /* skipping \n */
  comment = g_locale_to_utf8(rep, -1, NULL, NULL, NULL);
  if (!comment)
    comment = g_strdup("");

  if (!fgets(rep, MAX_LINE_LENGTH, in) ||
      sscanf(rep, "%d %d %d", size, size + 1, size + 2) != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  if (!fgets(rep, MAX_LINE_LENGTH, in) ||
      sscanf(rep, "%lf %lf %lf", box, box + 1, box + 2) != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }
  if (!fgets(rep, MAX_LINE_LENGTH, in) ||
      sscanf(rep, "%lf %lf %lf", box + 3, box + 4, box + 5) != 3)
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  if (!fgets(rep, MAX_LINE_LENGTH, in) ||
      sscanf(rep, "%s %s", format, period) < 1 ||
      (strcmp(format, "xyz") && strcmp(format, "zyx")))
    {
      /* Not a valid ASCII format. */
      g_free(comment);
      fclose(in);
      return FALSE;
    }
  periodic = !strcmp(period, "periodic");

  /* OK, from now on, the format is supposed to be ASCII. */
  field = scalarFieldNew(filename);
  if (!field)
    {
      g_warning("impossible to create a ScalarField object.");
      g_free(comment);
      fclose(in);
      return FALSE;
    }

  /* by default the meshtype is set to uniform to keep working previous version.*/
  meshtype = uniform;
  feed = fgets(rep, MAX_LINE_LENGTH, in);
  while (feed && rep[0] == '#')
    {
      if (strncmp(rep + 2, MESH_FLAG, strlen(MESH_FLAG)) == 0)
	{
	  DBG_fprintf(stderr, "ScalarField: found flag '%s'.\n", MESH_FLAG);
	  res = sscanf(rep + 2 + strlen(MESH_FLAG) + 1, "%s", flag);
	  if (res == 1 && strcmp(flag, MESH_FLAG_UNIFORM) == 0)
	    meshtype = uniform;
	  else if (res == 1 && strcmp(flag, MESH_FLAG_NON_UNIFORM) == 0)
	    meshtype = nonuniform;
	  else if (res == 1)
	    {
              g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			  _("wrong '%s' value for flag '%s'.\n"), flag, MESH_FLAG);
              fclose(in);
              return TRUE;
	    }
	}
      feed = fgets(rep, MAX_LINE_LENGTH, in);
    }

  scalarFieldSet_periodic(field, periodic);
  scalarFieldSet_commentary(field, comment);
  scalarFieldSet_meshtype(field, meshtype);
  scalarFieldSet_gridSize(field, size);
  scalarFieldSet_box(field, box);
  *fieldList = g_list_append(*fieldList, (gpointer)field);

  if (meshtype == nonuniform)
    {
      DBG_fprintf(stderr, "ScalarField : Start to read meshx.\n");

      for ( i = 0; i < size[0]; i++ )
        {
	  if (!feed)
	    {
	      g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			  _("not enough meshx values.\n"));
	      fclose(in);
	      return TRUE;
	    }
          res = sscanf(rep, "%lf", &field->meshx[i]);
	  do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
          if (res != 1)
            {
              g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			  _("impossible to read meshx values.\n"));
              fclose(in);
              return TRUE;
            }
        }

      DBG_fprintf(stderr, "ScalarField : Start to read meshy.\n");

      for ( j = 0; j < size[1]; j++ )
        {
	  if (!feed)
	    {
	      g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			  _("not enough meshy values.\n"));
	      fclose(in);
	      return TRUE;
	    }
          res = sscanf(rep, "%lf", &field->meshy[j]);
	  do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
          if (res != 1)
            {
              g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			  _("impossible to read meshy values.\n"));
              fclose(in);
              return TRUE;
            }
        }

      DBG_fprintf(stderr, "ScalarField : Start to read meshz.\n");

      for ( k = 0; k < size[2]; k++ )
        {
	  if (!feed)
	    {
	      g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			  _("not enough meshz values.\n"));
	      fclose(in);
	      return TRUE;
	    }
          res = sscanf(rep, "%lf", &field->meshz[k]);
	  do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
          if (res != 1)
            {
              g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			  _("impossible to read meshz values.\n"));
              fclose(in);
              return TRUE;
            }
        }
    }

  DBG_fprintf(stderr, "ScalarField : Start to read data.\n");

  field->min = G_MAXFLOAT;
  field->max = G_MINFLOAT;
  if(!strcmp(format, "xyz"))
    {
      for ( k = 0; k < size[2]; k++ ) 
	for ( j = 0; j < size[1]; j++ ) 
	  for ( i = 0; i < size[0]; i++ ) 
	    {
	      if (!feed)
		{
		  g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			      _("not enough density values.\n"));
		  fclose(in);
		  return TRUE;
		}

	      res = sscanf(rep, "%lf", &field->data[i][j][k]);

	      do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
	      if (res != 1)
		{
		  g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			      _("impossible to read density values.\n"));
		  fclose(in);
		  return TRUE;
		}
	      field->min = MIN(field->data[i][j][k], field->min);
	      field->max = MAX(field->data[i][j][k], field->max);
	    }
    }
  else
    {
      for ( i = 0; i < size[0]; i++ ) 
	for ( j = 0; j < size[1]; j++ ) 
	  for ( k = 0; k < size[2]; k++ ) 
	    {
	      if (!feed)
		{
		  g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			      _("not enough density values.\n"));
		  fclose(in);
		  return TRUE;
		}
	      res = sscanf(rep, "%lf", &field->data[i][j][k]);
	      do feed = fgets(rep, MAX_LINE_LENGTH, in); while (rep[0] == '#' && feed);
	      if (res != 1)
		{
		  g_set_error(error, TOOL_FILE_FORMAT_ERROR, TOOL_FILE_FORMAT_ERROR_FORMAT,
			      _("impossible to read density values.\n"));
		  fclose(in);
		  return TRUE;
		}
	      field->min = MIN(field->data[i][j][k], field->min);
	      field->max = MAX(field->data[i][j][k], field->max);
	    }
    }
  
  fclose(in);
  return TRUE;
}

gchar* scalarFieldGet_commentary(ScalarField *field)
{
  g_return_val_if_fail(field, (gchar*)0);
  return field->commentary;
}
void scalarFieldSet_commentary(ScalarField *field, gchar* comment)
{
  g_return_if_fail(field);
  
  field->commentary = g_strdup(comment);
}

gboolean scalarFieldGet_periodic(ScalarField *field)
{
  g_return_val_if_fail(field, FALSE);
  return field->periodic;
}
void scalarFieldSet_periodic(ScalarField *field, gboolean periodic)
{
  g_return_if_fail(field);
  
  field->periodic = periodic;
}

ScalarField_meshflag scalarFieldGet_meshtype(ScalarField *field)
{
  g_return_val_if_fail(field, uniform);
  return field->mesh_type;
}

void scalarFieldSet_meshtype(ScalarField *field, ScalarField_meshflag meshtype)
{
  g_return_if_fail(field);

  field->mesh_type = meshtype;
}

gchar* scalarFieldGet_filename(ScalarField *field)
{
  g_return_val_if_fail(field, (gchar*)0);
  return field->filename;
}

void scalarFieldSet_gridSize(ScalarField *field, int grid[3])
{
  int i, j;
  
  g_return_if_fail(field);
  
  if (field->nElements[0] == grid[0] &&
      field->nElements[1] == grid[1] &&
      field->nElements[2] == grid[2])
    return;
  
  DBG_fprintf(stderr, "ScalarField: changing size from (%d ; %d ; %d)"
                      " to (%d ; %d ; %d).\n", field->nElements[0], field->nElements[1],
                      field->nElements[2], grid[0], grid[1], grid[2]);

  if (field->mesh_type == nonuniform)
    {
      DBG_fprintf(stderr, " |�free the previous mesh allocation.\n");
      /* If mesh was already allocated, we free it. */
      if (field->meshx)
        g_free(field->meshx);

      if (field->meshy)
        g_free(field->meshy);

      if (field->meshz)
        g_free(field->meshz);
    }

  /* If data was already allocated, we free it. */
  if (field->data)
    {
      DBG_fprintf(stderr, " |�free the previous data allocation.\n");
      for (i = 0; i < field->nElements[0]; i++)
	      {
          for (j = 0; j < field->nElements[1]; j++)
	          g_free(field->data[i][j]);
      	  g_free(field->data[i]);
      	}
      g_free(field->data);
    }

  /* We change the size and reallocate mesh and data. */
  field->nElements[0] = grid[0];
  field->nElements[1] = grid[1];
  field->nElements[2] = grid[2];

  if (field->mesh_type == nonuniform)
    {
      DBG_fprintf(stderr, "ScalarField: allocating meshx array.\n");
      field->meshx = g_malloc(sizeof(double) * grid[0]);

      DBG_fprintf(stderr, "ScalarField: allocating meshy array.\n");
      field->meshy = g_malloc(sizeof(double) * grid[1]);

      DBG_fprintf(stderr, "ScalarField: allocating meshz array.\n");
      field->meshz = g_malloc(sizeof(double) * grid[2]);
    }

  DBG_fprintf(stderr, "ScalarField: allocating data array.\n");
  field->data = g_malloc(sizeof(double **) * grid[0]);
  for(i = 0; i < grid[0]; i++)
    {
      field->data[i] = g_malloc(sizeof(double *) * grid[1]);
      for(j = 0; j < grid[1]; j++)
        field->data[i][j] = g_malloc(sizeof(double) * grid[2]);
    }
  DBG_fprintf(stderr, " | allocation done.\n");
}

void scalarFieldGet_minMax(ScalarField *field, double minmax[2])
{
  g_return_if_fail(field);
  
  minmax[0] = field->min;
  minmax[1] = field->max;
}
void scalarFieldSet_data(ScalarField *field, double *data, gboolean xyzOrder)
{
  int i, j, k, ii;
  
  g_return_if_fail(field && data);
  
  field->min = G_MAXFLOAT;
  field->max = -G_MAXFLOAT;
  ii = 0;
  if (xyzOrder)
    for (k = 0 ; k < field->nElements[2] ; k++)
      for (j = 0 ; j < field->nElements[1] ; j++)
	for (i = 0 ; i < field->nElements[0] ; i++)
	  {
	    field->data[i][j][k] = data[ii];
	    field->min = MIN(data[ii], field->min);
	    field->max = MAX(data[ii], field->max);
	    ii += 1;
	  }
  else
    for (i = 0 ; i < field->nElements[0] ; i++)
      for (j = 0 ; j < field->nElements[1] ; j++)
	for (k = 0 ; k < field->nElements[2] ; k++)
	  {
	    field->data[i][j][k] = data[ii];
	    field->min = MIN(data[ii], field->min);
	    field->max = MAX(data[ii], field->max);
	    ii += 1;
	  }
}

gboolean scalarFieldGet_value(ScalarField *field, float xyz[3],
			      double *value, float extension[3])
{
  float redXyz[3], factor[3], pos;
  int l, m, n, ijk[3], dijk[3], nMax;
  int nval1, nval2;
  double *mesh;
  ScalarField_meshflag meshtype;

  /* Taken from ABINIT */
  float x1,x2,x3;

  g_return_val_if_fail(field, FALSE);

  meshtype = scalarFieldGet_meshtype(field);

  /* First, we transform the coordinates into reduced coordinates. */
  tool_matrix_productVector(redXyz, field->fromXYZtoReducedCoord, xyz);

  /* We compute i, j, k. */
  for (l = 0; l < 3; l++)
    {
      /* If we are periodic and inside the extension, we put back in
	 the box. */
      if (field->periodic && redXyz[l] > -extension[l] &&
	  redXyz[l] < 1. + extension[l])
	redXyz[l] = fModulo(redXyz[l], 1);
      if (field->periodic)
	nMax = field->nElements[l];
      else
	nMax = field->nElements[l] - 1;

      switch (meshtype)
        {
        case uniform:
          pos = (float)nMax * redXyz[l];
          ijk[l] = (int)pos;
          factor[l] = pos - (float)ijk[l];
          break;
        case nonuniform:
	  mesh = (double*)0;
	  switch (l)
	    {
	    case 0:
	      mesh = scalarFieldGet_meshx(field);
	      break;
	    case 1:
	      mesh = scalarFieldGet_meshy(field);
	      break;
	    case 2:
	      mesh = scalarFieldGet_meshz(field);
	      break;
	    }
          nval1 = 0;
          nval2 = nMax-1;
	  n = 0;
          for (m = 0; m < nMax/2; m++)
            {
              n = (int)((nval2-nval1)/2);
              if (n == 0) {
                n = nval1;
                break;    }
              else
                n = nval1+n;
              if (redXyz[l] > mesh[n])
                nval1 = n;
              else
                nval2 = n;
            }
          ijk[l] = n;
          factor[l] = (redXyz[l]-mesh[n])/(mesh[n+1]-mesh[n]);
          break;
        default:
          g_warning("Wrong value for 'meshtype'.");
          return FALSE;
        }
	
      if (ijk[l] < 0 || redXyz[l] < 0.)
	return FALSE;
      if (ijk[l] >= nMax)
	return FALSE;
    }

  /* lower left is ijk. */
  /* upper right is dijk. */
  dijk[0] = (ijk[0] + 1) % field->nElements[0];
  dijk[1] = (ijk[1] + 1) % field->nElements[1];
  dijk[2] = (ijk[2] + 1) % field->nElements[2];
  
  /* weight is factor. */
  x1 = factor[0];
  x2 = factor[1];
  x3 = factor[2];

  /* calculation of the density value */
  *value  = 0.f;
  *value += field->data[ ijk[0]][ ijk[1]][ ijk[2]] *
    (1.f - x1) * (1.f - x2) * (1.f - x3);
  *value += field->data[dijk[0]][ ijk[1]][ ijk[2]] * x1 * (1.f - x2) * (1.f - x3);
  *value += field->data[ ijk[0]][dijk[1]][ ijk[2]] * (1.f - x1) * x2 * (1.f - x3);
  *value += field->data[ ijk[0]][ ijk[1]][dijk[2]] * (1.f - x1) * (1.f - x2) * x3;
  *value += field->data[dijk[0]][dijk[1]][ ijk[2]] * x1 * x2 * (1.f - x3);
  *value += field->data[ ijk[0]][dijk[1]][dijk[2]] * (1.f  -x1) * x2 * x3;
  *value += field->data[dijk[0]][ ijk[1]][dijk[2]] * x1 * (1.f - x2) * x3;
  *value += field->data[dijk[0]][dijk[1]][dijk[2]] * x1 * x2 * x3;

  return TRUE;
}
void scalarFieldSet_boxFull(ScalarField *field, double box[3][3])
{
  double box_[6];

  tool_matrix_reducePrimitiveVectors(box_, box);
  scalarFieldSet_box(field, box_);
}
void scalarFieldSet_box(ScalarField *field, double box[6])
{
  int i;

  g_return_if_fail(field);

  DBG_fprintf(stderr, "ScalarField : set the bounding box.\n");

  /* Change the box. */
  for (i = 0; i < 6; i++)
    field->box[i] = box[i];

  /* Update the transformation matrix. */
  field->fromXYZtoReducedCoord[0][0] =
    1 / field->box[VISU_DATA_BOX_DXX];
  field->fromXYZtoReducedCoord[0][1] =
    - field->box[VISU_DATA_BOX_DYX] /
    field->box[VISU_DATA_BOX_DXX] /
    field->box[VISU_DATA_BOX_DYY];
  field->fromXYZtoReducedCoord[0][2] =
    - (field->box[VISU_DATA_BOX_DZX] /
       field->box[VISU_DATA_BOX_DXX] -
       field->box[VISU_DATA_BOX_DYX] *
       field->box[VISU_DATA_BOX_DZY] / 
       field->box[VISU_DATA_BOX_DXX] / 
       field->box[VISU_DATA_BOX_DYY] ) /
    field->box[VISU_DATA_BOX_DZZ];
  field->fromXYZtoReducedCoord[1][0] = 0.;
  field->fromXYZtoReducedCoord[1][1] =
    1 / field->box[VISU_DATA_BOX_DYY];
  field->fromXYZtoReducedCoord[1][2] =
    - field->box[VISU_DATA_BOX_DZY] /
    field->box[VISU_DATA_BOX_DYY] /
    field->box[VISU_DATA_BOX_DZZ];
  field->fromXYZtoReducedCoord[2][0] = 0.;
  field->fromXYZtoReducedCoord[2][1] = 0.;
  field->fromXYZtoReducedCoord[2][2] = 1 /
    field->box[VISU_DATA_BOX_DZZ];
}
void scalarFieldGet_box(ScalarField *field, double box[6])
{
  int i;

  g_return_if_fail(field);

  for (i = 0; i < 6; i++)
    box[i] = (double)field->box[i];
}
void scalarFieldGet_gridSize(ScalarField *field, int grid[3])
{
  int i;

  g_return_if_fail(field);

  for (i = 0; i < 3; i++)
    grid[i] = field->nElements[i];
}
double* scalarFieldGet_meshx(ScalarField *field)
{
  g_return_val_if_fail(field, (double*)0);
  return field->meshx;
}
double* scalarFieldGet_meshy(ScalarField *field)
{
  g_return_val_if_fail(field, (double*)0);
  return field->meshy;
}
double* scalarFieldGet_meshz(ScalarField *field)
{
  g_return_val_if_fail(field, (double*)0);
  return field->meshz;
}
double*** scalarFieldGet_data(ScalarField *field)
{
  g_return_val_if_fail(field, (double***)0);

  return field->data;
}

void scalarFieldSet_fitToBox(VisuData *data, ScalarField *field)
{
  double box[6];
  int i;

  g_return_if_fail(data);

  DBG_fprintf(stderr, "ScalarField: change the current box to fit to %p.\n",
	      (gpointer) data);
  for (i = 0; i < 6; i++)
    box[i] = (double)visu_data_getBoxGeometry(data, i);
  scalarFieldSet_box(field, box);
}  
GList* scalarFieldGet_allOptions(ScalarField *field)
{
  g_return_val_if_fail(field, (GList*)0);
  
  return g_list_copy(field->options);
}
void scalarFieldAdd_option(ScalarField *field, ToolOption *option)
{
  g_return_if_fail(field && option);
  
  field->options = g_list_append(field->options, (gpointer)option);
}


/* Load method handling. */
void scalarFieldInit(void)
{
  char *type[] = {"*.pot", "*.dat", (char*)0};
  char *descr = _("Potential/density files");
  ToolFileFormat *fmt;
  
  loadMethods = (GList*)0;
  
  fmt = tool_file_format_new(descr, type);
  scalarFieldAdd_loadMethod("Plain ascii", scalarFieldLoad_fromAscii,
			    fmt, G_PRIORITY_LOW);
}

void scalarFieldAdd_loadMethod(const gchar *name, ScalarFieldLoadMethod method,
			       ToolFileFormat *format, int priority)
{
  ScalarFieldLoadStruct *meth;

  g_return_if_fail(name && method && format);

  meth = g_malloc(sizeof(ScalarFieldLoadStruct));
  meth->name = g_strdup(name);
  meth->fmt = format;
  meth->load = method;
  meth->priority = priority;

  loadMethods = g_list_prepend(loadMethods, meth);
  loadMethods = g_list_sort(loadMethods, compareLoadPriority);
}

static gint compareLoadPriority(gconstpointer a, gconstpointer b)
{
  if (((ScalarFieldLoadStruct*)a)->priority <
      ((ScalarFieldLoadStruct*)b)->priority)
    return (gint)-1;
  else if (((ScalarFieldLoadStruct*)a)->priority >
	   ((ScalarFieldLoadStruct*)b)->priority)
    return (gint)+1;
  else
    return (gint)0;
}
GList* scalarFieldGet_allLoadMethods(void)
{
  return loadMethods;
}
