/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef RENDERINGATOMIC_H
#define RENDERINGATOMIC_H

#include <visu_rendering.h>
#include <visu_data.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <coreTools/toolFileFormat.h>

G_BEGIN_DECLS

/* This method draws spheres to represent atoms.
   It introduces a new resource : the radius of
   the sphere. It can read data file either in d3
   format or in ascii format. */

/**
 * renderingAtomic_shapeId:
 * @shapeSphere: draw sphere ;
 * @shapeCube: draw cube ;
 * @shapeElipsoid: draw elipsoid ;
 * @shapePoint: draw square dot ;
 * @shapeTorus: draw torus ;
 * @nbAtomicShapes: number of shapes.
 *
 * This enum is used as identifier for shapes managed by the
 * attomic rendering method.
 */
typedef enum 
  {
    shapeSphere,
    shapeCube,
    shapeElipsoid,
    shapePoint,
    shapeTorus,
    nbAtomicShapes
  } renderingAtomic_shapeId;

/**
 * initAtomic:
 *
 *Create the structure and initialise its values.
 */
void renderingAtomicInit();

#define VISU_RENDERING_ATOMIC "Atom visualisation"

/**
 * renderingAtomicLoad:
 * @data: a #VisuData object ;
 * @format: a pointer on a format (can be NULL if format is to be guessed) ;
 * @nSet: an integer ;
 * @error: a pointer to store a possible error.
 *
 * Try to load the file declared in the @data. It tries all formats added with
 * renderingAtomicAdd_loadMethod(). The @nSet argument is used to load
 * a specific set of nodes if the input format supports it. If @nSet
 * is 0, then the default set of nodes is loaded.
 *
 * Returns: FALSE if the file can't be loaded.
 */
gboolean renderingAtomicLoad(VisuData *data, FileFormat *format,
			     int nSet, GError **error);



/***************/
/* OpenGL Part */
/***************/

/**
 * renderingAtomic_createShape:
 * @visuData: a #VisuData object ;
 * @ele: a visu element.
 *
 * This function is used to create the OpenGL objects that
 * correspond to the @ele #VisuElement in the atomic rendering
 * method. It creates an OpenGL list whose number is stored internaly
 * and returned. For the atomic method, elements are
 * represented by spheres and similar shapes.
 *
 * Returns: an id representing an OpenGL list where the shape has been
 *          created.
 */
int renderingAtomic_createShape(VisuData *visuData, VisuElement* ele);
/**
 * renderingAtomic_positionShape:
 * @visuData: a #VisuData object ;
 * @node: a #VisuNode object ;
 * @ele: a #VisuElement.
 *
 * This method translates the given node to the right position. The @ele
 * parameter is the #VisuElement of the given node.
 */
void renderingAtomic_positionShape(VisuData *visuData, VisuNode *node, VisuElement* ele);

/***********************/
/* Different resources */
/***********************/
/**
 * renderingAtomicGet_radius:
 * @ele: a #VisuElement object.
 *
 * In the rendering atomic method, shapes are characterized by
 * a radius. This method gets it for the specified element. If this
 * element has no radius defined yet, the default value is
 * associated and returned.
 *
 * Returns: the radius of the specified element. A negative value if something
 *          goies wrong.
 */
float renderingAtomicGet_radius(VisuElement *ele);
/**
 * renderingAtomicGet_radiusDefault:
 *
 * This method gets the default radius of the rendering atomic method.
 *
 * Returns: the default value for radius resource.
 */
float renderingAtomicGet_radiusDefault();
/**
 * renderingAtomicSet_radius:
 * @ele: a #VisuElement object ;
 * @value: a positive floating point value.
 *
 * This change the radius value of element @ele to @value.
 *
 * Returns: 1 if a call to renderingAtomic_createShape() is required, 0 if not.
 */
int renderingAtomicSet_radius(VisuElement* ele, float value);

/**
 * renderingAtomicGet_shape:
 * @ele: a #VisuElement object.
 *
 * In the rendering atomic method, shapes are multiple.
 * This method gets it for the specified element. Shapes are
 * characterized by their id, corresponding to an integer value.
 * Use the enum #renderingAtomic_shapeId to associate an integer value
 * to a specific shape.
 *
 * Returns: the shape id of the element @ele.
 */
int renderingAtomicGet_shape(VisuElement *ele);
/**
 * renderingAtomicGet_shapeDefault:
 *
 * This method gets the default shape.
 *
 * Returns: the default shape id.
 */
int renderingAtomicGet_shapeDefault();
/**
 * renderingAtomicGet_shapeName:
 * @shape: an integer.
 *
 * This method does the corresponding between a shape id and
 * its name (a string value).
 *
 * Returns: the name associated to a shape.
 */
const char* renderingAtomicGet_shapeName(int shape);
/**
 * renderingAtomicGet_shapeNameDefault:
 *
 * This method is used to retrieve the default name for shapes.
 *
 * Returns: the name associated to the default shape.
 */
const char* renderingAtomicGet_shapeNameDefault();
/**
 * renderingAtomicSet_shape:
 * @ele: a #VisuElement object ;
 * @shape: an integer.
 *
 * This changes the shape of the element @ele to the shape defined by its id.
 *
 * Returns: 1 if a call to renderingAtomic_createShape() is required, 0 if not.
 */
int renderingAtomicSet_shape(VisuElement *ele, int shape);
/**
 * renderingAtomicSet_shapeFromName:
 * @ele: a #VisuElement object ;
 * @shape: a string.
 *
 * This method is equivalent to renderingAtomicSet_shape() but the shape is
 * defined by its name.
 *
 * Returns: 1 if a call to renderingAtomic_createShape() is required, 0 if not.
 */
int renderingAtomicSet_shapeFromName(VisuElement *ele, char* shape);
/**
 * renderingAtomicGet_allShapes:
 *
 * This methods retrieve the whole list of shape names used by V_Sim for example
 * in the resources file. These names are not translated. If internationalized
 * names are required, use renderingAtomicGet_allShapesI18n() instead.
 *
 * Returns: a pointer to a list of shape names (should not be modified or freed).
 */
const char** renderingAtomicGet_allShapes();
/**
 * renderingAtomicGet_allShapesI18n:
 *
 * This methods retrieve the whole list of shape names, translated strings.
 *
 * Returns: a pointer to a list of shape names (should not be modified or freed).
 */
const char** renderingAtomicGet_allShapesI18n();
/**
 * renderingAtomicSet_elipsoidParameters:
 * @ele: a #VisuElement object ;
 * @ratio: a float ;
 * @phi: a float ;
 * @theta: a float.
 *
 * Change the parameters for the elipsoid shape for the given @ele.
 * These parameters include a @ratio which is the ratio of the
 * long axis on the short one. Thus @ratio is always equal or greater than 1.
 * Arguments @theta and @phi are the direction of the long axis.
 *
 * Returns: TRUE if renderingAtomic_createShape() should be called.
 */
gboolean renderingAtomicSet_elipsoidParameters(VisuElement *ele, float ratio,
					       float phi, float theta);
/**
 * renderingAtomicGet_elipsoidRatio:
 * @ele: a #VisuElement object.
 *
 * Retrieve the ratio parameter of the elipsoid shape for the element @ele.
 *
 * Returns: the ratio of the elipsoid.
 */
float renderingAtomicGet_elipsoidRatio(VisuElement *ele);
/**
 * renderingAtomicGet_elipsoidPhi:
 * @ele: a #VisuElement object.
 *
 * Retrieve the phi angle parameter of the elipsoid shape for the element @ele.
 *
 * Returns: the phi angle of the elipsoid.
 */
float renderingAtomicGet_elipsoidPhi(VisuElement *ele);
/**
 * renderingAtomicGet_elipsoidTheta:
 * @ele: a #VisuElement object.
 *
 * Retrieve the theta angle parameter of the elipsoid shape for the element @ele.
 *
 * Returns: the theta angle of the elipsoid.
 */
float renderingAtomicGet_elipsoidTheta(VisuElement *ele);
/**
 * renderingAtomicSet_elipsoidRatio:
 * @ele: a #VisuElement object.
 * @ratio: a float ;
 *
 * Set the ratio parameter of the elipsoid shape for the element @ele.
 *
 * Returns: TRUE if renderingAtomic_createShape() should be called.
 */
gboolean renderingAtomicSet_elipsoidRatio(VisuElement *ele, float ratio);
/**
 * renderingAtomicSet_elipsoidPhi:
 * @ele: a #VisuElement object.
 * @phi: a float ;
 *
 * Set the phi angle parameter of the elipsoid shape for the element @ele.
 *
 * Returns: TRUE if renderingAtomic_createShape() should be called.
 */
gboolean renderingAtomicSet_elipsoidPhi(VisuElement *ele, float phi);
/**
 * renderingAtomicSet_elipsoidTheta:
 * @ele: a #VisuElement object.
 * @theta: a float.
 *
 * Set the theta angle parameter of the elipsoid shape for the element @ele.
 *
 * Returns: TRUE if renderingAtomic_createShape() should be called.
 */
gboolean renderingAtomicSet_elipsoidTheta(VisuElement *ele, float theta);

/**
 * renderingAtomicAdd_loadMethod:
 * @meth: a new load method.
 *
 * This routine is used to register a new load method with its description
 * (file formats, name...). The list of all known load methods is automatically
 * resorted after a call to this method to reflect the priorities.
 */
void renderingAtomicAdd_loadMethod(RenderingFormatLoad* meth);

G_END_DECLS


#endif
