/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef LIGHT_H
#define LIGHT_H

#include <glib.h>

/**
 * lightEnvironnement_struct:
 *
 * An opaque structure to allow an OpenGL context to manage and use several lights.
 */
struct lightEnvironnement_struct;
/**
 * LightEnvironnement:
 *
 * A short way to access #lightEnvironnement_struct objects.
 */
typedef struct lightEnvironnement_struct LightEnvironnement;

/**
 * lightEnvironnement_new:
 *
 * Create a new #LightEnvironnement object. It contains no light when created.
 * Use lightEnvironnementAdd_light() to add new lights and
 * lightEnvironnementDelete_light() to remove others.
 *
 * Returns: a newly created #LightEnvironnement. Use lightEnvironnementFree()
 *          to free such an object.
 */
LightEnvironnement* lightEnvironnement_new();
/**
 * lightEnvironnementFree:
 * @env: a #LightEnvironnement object.
 *
 * Free memory occupied by the given environnement.
 */
void lightEnvironnementFree(LightEnvironnement *env);

/**
 * Material:
 * @material_amb: the ambient identifier ;
 * @material_dif: the diffuse identifier ;
 * @material_shi: the shiningness identifier ;
 * @material_spe: the specular identifier ;
 * @material_emi: the emissivity identifier ;
 * @nb_material: number of used material identifiers.
 *
 * This enum is used to address the OpenGL parameters for light rendering.
 */
typedef enum
  {
    material_amb,
    material_dif,
    material_shi,
    material_spe,
    material_emi,
    nb_material
  } Material;

/**
 * light_struct:
 * @enabled: if the light is used or not ;
 * @ambient: the ambient color of the light ;
 * @diffuse: the diffuse color of the light ;
 * @specular: the specular color of the light ;
 * @position: the position in space of the light ;
 * @multiplier: a value that multiply all color values (should be in [0;1]).
 *
 * This structure is convenient to store lights as defined by OpenGL.
 */
struct light_struct
{  
  gboolean enabled;
  float ambient[4];
  float diffuse[4];
  float specular[4];
  float position[4];
  float multiplier;
};
/**
 * Light:
 *
 * A short name for #light_struct objects.
 */
typedef struct light_struct Light;

/**
 * lightEnvironnementAdd_light:
 * @env: a #LightEnvironnement object ;
 * @light: a #Light object.
 *
 * This method adds the given @light to the list of known lights declared
 * in the given environnement. The light is not copied and should not be freed
 * when stored in the environnement.
 *
 * Returns: TRUE if lightEnvironnementApply() should be called.
 */
gboolean lightEnvironnementAdd_light(LightEnvironnement *env, Light *light);
/**
 * lightEnvironnementDelete_light:
 * @env: a #LightEnvironnement object ;
 * @light: a #Light object.
 *
 * This method removes the given @light from the list of known lights  declared
 * in the given environnement. The @light argument is first removed and then freed
 * by a call to g_free().
 *
 * Returns: TRUE if lightEnvironnementApply() should be called.
 */
gboolean lightEnvironnementDelete_light(LightEnvironnement *env, Light *light);
/**
 * lightEnvironnementGet_lightList:
 * @env: a #LightEnvironnement object.
 *
 * Retrieve the list of known #Light used by the given environnement.
 *
 * Returns: a list of #Light objects. Should not be freed.
 */
GList* lightEnvironnementGet_lightList(LightEnvironnement *env);
/**
 * lightEnvironnementEmpty_lightList:
 * @env: a #LightEnvironnement object.
 *
 * Empty the list of stored lights. All stored lights objects are freed.
 *
 * Returns: TRUE if the lightEnvironnementApply() should be called.
 */
gboolean lightEnvironnementEmpty_lightList(LightEnvironnement *env);
/**
 * lightEnvironnementApply:
 * @env: a #LightEnvironnement object.
 *
 * Apply all stored informations about lights to the current OpenGL context.
 *
 * Returns: TRUE if the "OpenGLAskForReDraw" signal should be
 *          emitted after a call to this method.
 */
gboolean lightEnvironnementApply(LightEnvironnement *env);

/**
 * light_newDefault:
 *
 * Create a new light with default value (white color and position in
 * the front, right, top position of the screen).
 *
 * Returns: the newly created #Light. Use g_free() to deallocate this light.
 */
Light* light_newDefault();



#endif
