/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include <gtk/gtksignal.h>
#include <gtk/gtkcombobox.h>
#include <gtk/gtkcelllayout.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrendererpixbuf.h>

#include "gtk_shadeComboBoxWidget.h"
#include <visu_tools.h>

/**
 * SECTION:shadecombobox
 * @short_description: Defines a specialised #GtkComboBox to choose
 * pre-built shades.
 * @see_also: #StippleComboBox, #Shade
 * @include: coreTools/toolShade.h
 *
 * <para>This widget looks like a #GtkComboBox and it displays a list
 * of preset colour shades. These patterns are defined by colour
 * parameters, see #Shade.</para>
 * <para>This widget can emit a #ShadeComboBox::shade-selected
 * signal that is a wrapper around the #GtkComboBox::changed signal,
 * but it is emitted only when a new shade is selected and this shade
 * is passed to the callback.</para>
 *
 * Since: 3.3
 */

enum {
  SHADE_SELECTED_SIGNAL,
  LAST_SIGNAL
};

/* This enum is used to access the column of the GtkListStore
   that contains the informations of stroed shades. */
enum
  {
    /* This has a pointer to a 48x16 image to represent the shade. */
    COLUMN_SHADE_PIXBUF,
    /* This is a pointer to a label that describes the shade. */
    COLUMN_SHADE_LABEL,
    /* This a pointer to the #Shade as defined in toolShade.h */
    COLUMN_SHADE_POINTER_TO,

    N_COLUMN_SHADE
  };

/* Store a tree model to remember shades. */
#define SHADE_BOX_WIDTH  48
#define SHADE_BOX_HEIGHT 16
#define SHADE_BOX_BITS   8

static void shadeComboBox_dispose (GObject *obj);
static void shadeComboBox_finalize(GObject *obj);

static guint shadeComboBox_signals[LAST_SIGNAL] = { 0 };

struct _ShadeComboBox
{
  GtkComboBox comboShade;
  Shade* previouslySelectedShade;

  gboolean hasAlphaChannel;

  /* Memory gestion. */
  gboolean dispose_has_run;
};

struct _ShadeComboBoxClass
{
  GtkComboBoxClass parent_class;

  void (*shadeComboBox) (ShadeComboBox *shadeCombo);

  /* This listStore contains all the shades
     known by widgets of this class. It is used
     as TreeModel for the combobox in the widget. */
  GtkListStore *listStoredShades;

  gulong shadeAddedSignalId;
};

/* Local callbacks. */
static void shadeComboBox_changed(GtkWidget *widget, ShadeComboBox *shadeComboBox);

/* Local methods. */
static void addShadeToModel(GtkTreeIter *iter, ShadeComboBoxClass* klass,
			    Shade* shade);
static void buildWidgets(ShadeComboBox *shadeComboBox);

G_DEFINE_TYPE(ShadeComboBox, shadeComboBox, GTK_TYPE_COMBO_BOX)

static void shadeComboBox_class_init(ShadeComboBoxClass *klass)
{
  GtkTreeIter iter;
  GList *shadeLst;
  
  DBG_fprintf(stderr, "Gtk ShadeComboBox : creating the class of the widget.\n");
  DBG_fprintf(stderr, "                     - adding new signals ;\n");
  /**
   * ShadeComboBox::shade-selected:
   * @combo: the #ShadeComboBox that emits the signal ;
   * @shade: the newly selected #Shade.
   *
   * This signal is emitted when a new valid colour shade is selected.
   *
   * Since: 3.3
   */
  shadeComboBox_signals[SHADE_SELECTED_SIGNAL] =
    g_signal_new ("shade-selected",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		  G_STRUCT_OFFSET (ShadeComboBoxClass, shadeComboBox),
		  NULL, 
		  NULL,                
		  g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, G_TYPE_POINTER);

  DBG_fprintf(stderr, "                     - initializing the listStore of shades.\n");
  /* Init the listStore of shades. */
  klass->listStoredShades = gtk_list_store_new(N_COLUMN_SHADE,
					       GDK_TYPE_PIXBUF,
					       G_TYPE_STRING,
					       G_TYPE_POINTER);
  shadeLst = toolShadeBuild_presetList();
  while (shadeLst)
    {
      addShadeToModel(&iter, klass, (Shade*)shadeLst->data);
      shadeLst = g_list_next(shadeLst);
    }

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose = shadeComboBox_dispose;
  G_OBJECT_CLASS(klass)->finalize = shadeComboBox_finalize;
}

static void shadeComboBox_dispose(GObject *obj)
{
  DBG_fprintf(stderr, "Gtk ShadeComboBox : dispose object %p.\n", (gpointer)obj);

  if (SHADE_COMBOX(obj)->dispose_has_run)
    return;

  SHADE_COMBOX(obj)->dispose_has_run = TRUE;
  /* Chain up to the parent class */
  G_OBJECT_CLASS(shadeComboBox_parent_class)->dispose(obj);
}
static void shadeComboBox_finalize(GObject *obj)
{
  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Gtk ShadeComboBox : finalize object %p.\n", (gpointer)obj);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(shadeComboBox_parent_class)->finalize(obj);

  DBG_fprintf(stderr, " | freeing ... OK.\n");
}


static void shadeComboBox_init(ShadeComboBox *shadeComboBox)
{
  DBG_fprintf(stderr, "Gtk ShadeComboBox : initializing new object (%p).\n",
	      (gpointer)shadeComboBox);

  shadeComboBox->hasAlphaChannel = TRUE;
  shadeComboBox->dispose_has_run = FALSE;
  shadeComboBox->previouslySelectedShade = (Shade*)0;
}

static void buildWidgets(ShadeComboBox *shadeComboBox)
{
  GObjectClass *klass;
  GtkCellRenderer *renderer;

  klass = G_OBJECT_GET_CLASS(shadeComboBox);
  gtk_combo_box_set_model(GTK_COMBO_BOX(shadeComboBox),
			  GTK_TREE_MODEL(SHADE_COMBOX_CLASS(klass)->listStoredShades));

  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(shadeComboBox), renderer, FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(shadeComboBox), renderer,
				"pixbuf", COLUMN_SHADE_PIXBUF);
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(shadeComboBox), renderer, FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(shadeComboBox), renderer,
				"text", COLUMN_SHADE_LABEL);

  gtk_combo_box_set_active(GTK_COMBO_BOX(shadeComboBox), -1);
  g_signal_connect(G_OBJECT(shadeComboBox), "changed",
		   G_CALLBACK(shadeComboBox_changed), (gpointer)shadeComboBox);
}

GtkWidget* shadeComboBox_new(gboolean hasAlphaChannel)
{
  ShadeComboBox *shadeComboBox;

  DBG_fprintf(stderr, "Gtk ShadeComboBox : creating new object with alpha : %d.\n",
	      hasAlphaChannel);

  shadeComboBox = SHADE_COMBOX(g_object_new(shadeComboBox_get_type (), NULL));
  shadeComboBox->hasAlphaChannel = hasAlphaChannel;

  DBG_fprintf(stderr, "Gtk ShadeComboBox : build widgets.\n");

  buildWidgets(shadeComboBox);

  return GTK_WIDGET(shadeComboBox);
}

static void shadeComboBox_changed(GtkWidget *widget _U_, ShadeComboBox *shadeComboBox)
{
  int selected;
  GtkTreeIter iter;
  GObjectClass *klass;
  Shade *shade;

  selected = gtk_combo_box_get_active(GTK_COMBO_BOX(shadeComboBox));
  DBG_fprintf(stderr, "Gtk ShadeComboBox : internal combobox changed signal -> %d.\n", selected);
  if (selected < 0)
    {
      shadeComboBox->previouslySelectedShade = (Shade*)0;
/*       DBG_fprintf(stderr, "Gtk ShadeComboBox : emitting 'shade-selected' signal.\n"); */
/*       g_signal_emit(G_OBJECT(shadeComboBox), */
/* 		    shadeComboBox_signals[SHADE_SELECTED_SIGNAL], 0, (gpointer)0, NULL); */
      return;
    }

  gtk_combo_box_get_active_iter(GTK_COMBO_BOX(shadeComboBox), &iter);
  klass = G_OBJECT_GET_CLASS(shadeComboBox);
  gtk_tree_model_get(GTK_TREE_MODEL(SHADE_COMBOX_CLASS(klass)->listStoredShades), &iter,
		     COLUMN_SHADE_POINTER_TO, &shade,
		     -1);
  if (shade != shadeComboBox->previouslySelectedShade)
    {
      shadeComboBox->previouslySelectedShade = shade;
      DBG_fprintf(stderr, "Gtk ShadeComboBox : emitting 'shade-selected' signal.\n");
      g_signal_emit(G_OBJECT(shadeComboBox),
		    shadeComboBox_signals[SHADE_SELECTED_SIGNAL], 0, (gpointer)shade, NULL);
    }
  else
    DBG_fprintf(stderr, "Gtk ShadeComboBox : aborting 'shade-selected' signal.\n");
}

GdkPixbuf* shadeComboBoxBuild_shadeStamp(Shade *shade, GdkPixbuf *pixbuf)
{
  GdkPixbuf *pixbufShadeBox;
  int rowstride, x, y;
  guchar *pixels, *p;
  float grey;
  float **rgbVals;

  if (!pixbuf)
    pixbufShadeBox = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE,
				    SHADE_BOX_BITS,
				    SHADE_BOX_WIDTH,
				    SHADE_BOX_HEIGHT);
  else
    pixbufShadeBox = pixbuf;
  rowstride = gdk_pixbuf_get_rowstride(pixbufShadeBox);
  pixels = gdk_pixbuf_get_pixels(pixbufShadeBox);
  rgbVals  = g_malloc(sizeof(float*) * SHADE_BOX_WIDTH);
  rgbVals[0] = g_malloc(sizeof(float) * SHADE_BOX_WIDTH * 5);
  for (x = 0; x < SHADE_BOX_WIDTH; x++)
    {
      rgbVals[x] = rgbVals[0] + x * 5;
      rgbVals[x][0] = (float)x / (float)(SHADE_BOX_WIDTH - 1);
      shadeGet_valueTransformedInRGB(shade, &rgbVals[x][1], rgbVals[x][0]);
    }
  for (y = 0; y < SHADE_BOX_HEIGHT; y++)
    for (x = 0; x < SHADE_BOX_WIDTH; x++)
      {
	p = pixels + y * rowstride + x * 3;
	if (x < SHADE_BOX_WIDTH / 2)
	  {
	    if (y < SHADE_BOX_HEIGHT / 2)
	      grey = 0.75;
	    else
	      grey = 0.5;
	  }
	else
	  {
	    if (y < SHADE_BOX_HEIGHT / 2)
	      grey = 0.5;
	    else
	      grey = 0.75;
	  }
	p[0] = (guchar)((rgbVals[x][1] * rgbVals[x][4] +
			 (1. - rgbVals[x][4]) * grey) * 255.);
	p[1] = (guchar)((rgbVals[x][2] * rgbVals[x][4] +
			 (1. - rgbVals[x][4]) * grey) * 255.);
	p[2] = (guchar)((rgbVals[x][3] * rgbVals[x][4] +
			 (1. - rgbVals[x][4]) * grey) * 255.);
      }
  g_free(rgbVals[0]);
  g_free(rgbVals);
  return pixbufShadeBox;
}

static void addShadeToModel(GtkTreeIter *iter, ShadeComboBoxClass* klass,
			    Shade* shade)
{
  g_return_if_fail(iter && klass && shade);

  DBG_fprintf(stderr, "Gtk ShadeComboBox : appending a new shade '%s'.\n",
	      shadeGet_label(shade));
  gtk_list_store_append(klass->listStoredShades, iter);
  gtk_list_store_set(klass->listStoredShades, iter,
		     COLUMN_SHADE_PIXBUF      , shadeComboBoxBuild_shadeStamp(shade, (GdkPixbuf*)0),
		     COLUMN_SHADE_LABEL       , shadeGet_label(shade),
		     COLUMN_SHADE_POINTER_TO  , (gpointer)shade,
		     -1);
  DBG_fprintf(stderr, "Gtk ShadeComboBox : appending a new shade '%s' at position '%s'.\n",
	      shadeGet_label(shade), gtk_tree_model_get_string_from_iter
	      (GTK_TREE_MODEL(SHADE_COMBOX_CLASS(klass)->listStoredShades), iter));
}

gboolean shadeComboBoxSet_selectionByShade(ShadeComboBox* shadeComboBox, Shade *shade)
{
  GtkTreeIter iter;
  gboolean validIter;
  GObjectClass *klass;
  GtkListStore *model;
  Shade *tmpShade;

  g_return_val_if_fail(shade && IS_SHADE_COMBOX(shadeComboBox), FALSE);

  DBG_fprintf(stderr, "Gtk ShadeComboBox : select a new shade %p.\n", (gpointer)shade);
  klass = G_OBJECT_GET_CLASS(shadeComboBox);
  model = GTK_LIST_STORE(SHADE_COMBOX_CLASS(klass)->listStoredShades);
  validIter = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(model), &iter);
  while (validIter)
    {
      gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
			 COLUMN_SHADE_POINTER_TO, &tmpShade,
			 -1);
      if (tmpShade == shade)
	{
	  gtk_combo_box_set_active_iter(GTK_COMBO_BOX(shadeComboBox), &iter);
	  return TRUE;
	}
      validIter = gtk_tree_model_iter_next(GTK_TREE_MODEL(model), &iter);
    }
  return FALSE;
}
GdkPixbuf* shadeComboBoxGet_selectedPixbuf(ShadeComboBox *shadeComboBox)
{
  gboolean validIter;
  GtkTreeIter iter;
  GdkPixbuf *pixbuf;
  GObjectClass *klass;

  g_return_val_if_fail(IS_SHADE_COMBOX(shadeComboBox), (GdkPixbuf*)0);

  validIter = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(shadeComboBox), &iter);
  if (!validIter)
    return (GdkPixbuf*)0;

  pixbuf = (GdkPixbuf*)0;
  klass = G_OBJECT_GET_CLASS(shadeComboBox);
  gtk_tree_model_get(GTK_TREE_MODEL(SHADE_COMBOX_CLASS(klass)->listStoredShades), &iter,
		     COLUMN_SHADE_PIXBUF, &pixbuf,
		     -1);
  return pixbuf;
}
Shade* shadeComboBoxGet_selectedShade(ShadeComboBox *shadeComboBox)
{
  gboolean validIter;
  GtkTreeIter iter;
  Shade *shade;
  GObjectClass *klass;

  g_return_val_if_fail(IS_SHADE_COMBOX(shadeComboBox), (Shade*)0);

  validIter = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(shadeComboBox), &iter);
  if (!validIter)
    return (Shade*)0;

  shade = (Shade*)0;
  klass = G_OBJECT_GET_CLASS(shadeComboBox);
  gtk_tree_model_get(GTK_TREE_MODEL(SHADE_COMBOX_CLASS(klass)->listStoredShades), &iter,
		     COLUMN_SHADE_POINTER_TO, &shade,
		     -1);
  return shade;
}

GdkPixbuf* shadeComboBoxGet_pixbufFromShade(ShadeComboBox *shadeComboBox, Shade *shade)
{
  GtkTreeIter iter;
  gboolean validIter;
  GdkPixbuf *pixbuf;
  Shade *cl;
  GtkListStore *model;

  g_return_val_if_fail(IS_SHADE_COMBOX(shadeComboBox) && shade, (GdkPixbuf*)0);

  model = SHADE_COMBOX_CLASS(G_OBJECT_GET_CLASS(shadeComboBox))->listStoredShades;
  validIter = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(model), &iter);
  while (validIter)
    {
      pixbuf = (GdkPixbuf*)0;
      cl = (Shade*)0;
      gtk_tree_model_get(GTK_TREE_MODEL(model), &iter,
			 COLUMN_SHADE_PIXBUF, &pixbuf,
			 COLUMN_SHADE_POINTER_TO, &cl,
			 -1);
      if (shade == cl)
	return pixbuf;
      validIter = gtk_tree_model_iter_next(GTK_TREE_MODEL(model), &iter);
    }
  return (GdkPixbuf*)0;
}
