#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
/*#include <linux/netdevice.h>*/
/*#include <linux/if_userlink.h>*/
#include "if_userlink.h"

#define	_PATH_PROC_DEVICES	"/proc/devices"
#define	LEN_PROCLINE	256

int
open_userlink(char *devname)
{
    FILE *fp;
    int major, n, fd;
    char line[LEN_PROCLINE], name[IFNAMSIZ];

    if ((fp = fopen(_PATH_PROC_DEVICES, "r")) == NULL) return(-1);
    while(fgets(line, LEN_PROCLINE, fp)) {
	/* Searching major number of userlink driver */
	if (sscanf(line, "%d %s", &major, name) == 2)
	    if (!strcmp(name, "userlink")) break;
	major = -1;
    }
    if (major < 0) {
	/* When userlink driver is not loaded, call dummy ioctl
	   to invoke kerneld daemon */
	int sd;
	struct sockaddr_in *sin;
	struct ifreq ifr;

	bzero(&ifr, sizeof(ifr));
	sd = socket(AF_INET, SOCK_DGRAM, 0);
	sin = (struct sockaddr_in *)&(ifr.ifr_addr);
	sin->sin_family = AF_INET;
	strcpy(ifr.ifr_name, "userlink");
	ioctl(sd, SIOCGIFFLAGS, &ifr);
	fseek(fp, 0L, SEEK_SET);
	/* Searching major number of userlink driver, again */
	while(fgets(line, LEN_PROCLINE, fp)) {
	    if (sscanf(line, "%d %s", &major, name) == 2)
		if (!strcmp(name, "userlink")) break;
	    major = -1;
	}
	close(sd);
    }
    fclose(fp);
    if (major < 0) return(-1);
    strcpy(name, "/tmp/ulXXXXXX"); /* special file name of char dev */
    if (mktemp(name) == NULL) return(-1);
    for (n = 0; n < 10; n ++) {
	if (mknod(name, (S_IFCHR|S_IREAD|S_IWRITE), major << 8 | n) != 0)
	    return(-1);
	errno = 0;
	fd = open(name, O_RDWR);
	unlink(name);
	if (fd >= 0) {
	    sprintf(devname, "ul%d", n);
	    return(fd);
	}
    }
    return(-1);
}

int
set_userlink_version(char *devname, int version)
{
    int                  sk;
    struct ul_ifru_data uld;
    struct ifreq        ifr;

    if ((sk = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	perror("socket");
	return(-1);
    }
    uld.command = UL_IOCSET;
    uld.version = version;
    uld.max_mtu = 0;
    strncpy(ifr.ifr_name, devname, IFNAMSIZ);
    ifr.ifr_data = (char*) &uld;

    if (ioctl(sk, SIOCDEVPRIVATE, &ifr) < 0) {
	perror("SIOCDEVPRIVATE");
	close(sk);
	return(-1);
    }
    close(sk);
    return(0);
}

int
get_userlink_version(char *devname)
{
    int                  sk;
    struct ul_ifru_data uld;
    struct ifreq        ifr;

    if ((sk = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	perror("socket");
	return(-1);
    }
    uld.command = UL_IOCGET;
    uld.version = 0;
    uld.max_mtu = 0;
    strncpy(ifr.ifr_name, devname, IFNAMSIZ);
    ifr.ifr_data = (char*) &uld;
    if (ioctl(sk, SIOCDEVPRIVATE, &ifr) < 0) {
	perror("SIOCDEVPRIVATE");
	close(sk);
	return(-1);
    }
    close(sk);
    return(uld.version);
}

#include <arpa/inet.h>
#include <netinet/ip.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <net/route.h>
#include <sys/time.h>
#include <signal.h>

void DumpBuffer(u_char *buff, int n, int flag)
{
    int i;

    if (flag > 1) {
	struct iphdr *ip;
	struct in_addr ia;

	ip = (struct iphdr *)buff;
	printf("\nihl=%d,ver=%d,tos=%X,len=%d,id=%X,off=%d,"
	       "ttl=%d,proto=%X,check=%X\n",
	       ip->ihl, ip->version, ip->tos, ntohs(ip->tot_len),
	       ntohs(ip->id), ntohs(ip->frag_off), ip->ttl,
	       ip->protocol, ntohs(ip->check));
	ia.s_addr = ip->saddr;
	printf("saddr=%s,", inet_ntoa(ia));
	ia.s_addr = ip->daddr;
	printf("daddr=%s", inet_ntoa(ia));
	buff += sizeof(*ip);
    }
    for (i = 0; i < n; i ++) {
	if (!(i % 16)) printf("\n");
	printf("%02X ", buff[i]);
    }
    printf("\n");
}

void RouteSet(int cmd, u_int32_t dest, u_int32_t gway, u_int32_t mask)
{
    struct rtentry rt;
    int s;

    if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
	perror("socket\n");

    bzero(&rt, sizeof(rt));
    ((struct sockaddr_in *)&rt.rt_dst)->sin_addr.s_addr = dest;
    ((struct sockaddr_in *)&rt.rt_gateway)->sin_addr.s_addr = gway;
    rt.rt_dst.sa_family = rt.rt_gateway.sa_family = AF_INET;
    if (cmd == 1) {
	if (mask == INADDR_NONE) rt.rt_flags = RTF_HOST;
	else {
	    ((struct sockaddr_in *)&rt.rt_dst)->sin_addr.s_addr &=
		mask;
	    rt.rt_flags = RTF_GATEWAY;
	}
	rt.rt_flags |= RTF_UP;
	if (ioctl(s, SIOCADDRT, &rt) < 0) perror("SIOCADDRT");
    } else {
	rt.rt_flags = RTF_GATEWAY;
	if (ioctl(s, SIOCDELRT, &rt) < 0 && errno != ESRCH)
	    perror("SIOCDELRT");
    }
    close(s);
}

static int
setif(char *dev, int sd, u_int32_t laddr, u_int32_t raddr, u_int32_t mask, int mtu)
{
    static struct ifreq ifrq;
    struct sockaddr_in *sin;

    bzero(&ifrq, sizeof(ifrq));
    strncpy(ifrq.ifr_name, dev, IFNAMSIZ);
    sin = (struct sockaddr_in *)&(ifrq.ifr_addr);
    sin->sin_family = AF_INET;

    /* ifaddr */
    sin->sin_addr.s_addr = laddr;
    if(ioctl(sd, SIOCSIFADDR, &ifrq) < 0)
	perror("SIOCSIFADDR");
    /* pointopoint */
    sin->sin_addr.s_addr = raddr;
    if(ioctl(sd, SIOCSIFDSTADDR, &ifrq) < 0)
	perror("SIOCSIFDSTADDR");
    /* netmask */
    sin->sin_addr.s_addr = mask;
    if(ioctl(sd, SIOCSIFNETMASK, &ifrq) < 0)
	perror("SIOCSIFNETMASK");
    if (mtu > 0) {
	/* mtu */
	ifrq.ifr_mtu = mtu;
	if(ioctl(sd, SIOCSIFMTU, &ifrq) < 0)
	    perror("SIOCSIFMTU");
    }
    /* flag */
    if(ioctl(sd, SIOCGIFFLAGS, &ifrq) < 0) {
	perror("SIOCGFLAGS");
	return(-1);
    }
    if (laddr == 0 && raddr == 0)
	ifrq.ifr_flags &= ~IFF_UP;
    else
	ifrq.ifr_flags |= IFF_UP;
    if(ioctl(sd, SIOCSIFFLAGS, &ifrq) < 0) {
	perror("SIOCSFLAGS");
	return(-1);
    }
    return(0);
}

int IfDown(char *dev)
{
    int s;

    if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	perror("socket");
	return(-1);
    }

    setif(dev, s, 0, 0, 0, 0);
    close(s);
    return(0);
}

int IfUp(char *dev, u_int32_t laddr, u_int32_t raddr, u_int32_t nmask, int mtu)
{
    int s;
    u_int32_t mask, addr;

    if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	perror("socket");
	return(-1);
    }

    addr = ntohl(laddr);
    if (IN_CLASSA(addr))
	mask = IN_CLASSA_NET;
    else if (IN_CLASSB(addr))
	mask = IN_CLASSB_NET;
    else
	mask = IN_CLASSC_NET;
    if (nmask && (ntohl(nmask) & mask) == mask)
	mask = ntohl(nmask);
    nmask = htonl(mask);
    if (setif(dev, s, laddr, raddr, nmask, mtu) < 0) {
	close(s);
	return(-1);
    }
    nmask = INADDR_NONE;
    RouteSet(1, raddr, laddr, nmask);
    close(s);
    return(0);
}

static int
Read1(int fd, char *buf, int max)
{
    return(read(fd, buf, max));
}

static int
Read2(int fd, char *buf, int max)
{
    return(read(fd, buf, max));
}

static int
Write1(int fd, char *buf, int max)
{
    return(write(fd, buf, max));
}

static int
Write2(int fd, char *buf, int max)
{
    return(write(fd, buf, max));
}

int
main(int argc, char *argv[])
{
    u_int32_t addr1=0, addr2=0;
    int fd1, fd2, n, dump=0, ver=UL_V_NONE, mfd;
    u_char name1[16], name2[16], buff[9000], *p;
    fd_set readFds, orgReadFds;

    for (n = 1; n < argc; n ++) {
	p = argv[n];
	if (*p ++ == '-') switch(*p) {
	case 'd':
	    dump = 1;
	    break;
	case 'D':
	    dump = 2;
	    break;
	case 'b':
	    ver = UL_V_BASE;
	    break;
	case 'n':
	    ver = UL_V_NONE;
	    break;
	} else {
	    if (addr1 > 0)
		addr2 = inet_addr(argv[n]);
	    else
		addr1 = inet_addr(argv[n]);
	}
    }
    if (addr2 == 0) addr2 = inet_addr("127.0.0.2");
    if (addr1 == 0) addr1 = inet_addr("127.0.0.3");
    if ((fd1 = open_userlink(name1)) < 0) {
	perror("OpenDevice");
	exit(1);
    }
    set_userlink_version(name1, ver);
    IfUp(name1, addr2, addr1, 0, 1500);

    if ((fd2 = open_userlink(name2)) < 0) {
	perror("OpenDevice");
	exit(1);
    }
    IfUp(name2, addr1, addr2, 0, 1500);
    set_userlink_version(name2, ver);
    printf("%s <-> %s\n", name1, name2);

    FD_ZERO(&orgReadFds);
    FD_SET(0, &orgReadFds);
    FD_SET(fd1, &orgReadFds);
    FD_SET(fd2, &orgReadFds);
    mfd = ((fd1 > fd2) ? fd1: fd2) + 1;
    while (1) {
	readFds = orgReadFds;
	if (select(mfd, &readFds, NULL, NULL, NULL) < 0
	    && errno == EINTR) continue;
	if (FD_ISSET(fd1, &readFds)) {
	    n = Read1(fd1, buff, 1500);
	    Write2(fd2, buff, n);
	    if (dump) {
		fprintf(stderr, "1->2 size = %d", n);
		DumpBuffer(buff, n, dump);
	    }
	}
	if (FD_ISSET(fd2, &readFds)) {
	    n = Read2(fd2, buff, 1500);
	    Write1(fd1, buff, n);
	    if (dump) {
		fprintf(stderr, "2->1 size = %d", n);
		DumpBuffer(buff, n, dump);
	    }
	}
	if (FD_ISSET(0, &readFds)) {
	    buff[0] = 0;
	    n = read(0, buff, 1500);
	    if (buff[0] == 0) break;
/*	    write(0, buff, n);*/
	}
    }
    IfDown(name2);
    IfDown(name1);
    return(0);
}
