/* ========================================================================== */
/* === UMFPACK_get_numeric ================================================== */
/* ========================================================================== */

/* -------------------------------------------------------------------------- */
/* UMFPACK Version 3.2 (Jan. 1, 2002), Copyright (c) 2002 by Timothy A.       */
/* Davis, University of Florida, davis@cise.ufl.edu.  All Rights Reserved.    */
/* See README, umfpack.h, or type "umfpack_details" in Matlab for License.    */
/* -------------------------------------------------------------------------- */

/*
    User-callable.  Gets the LU factors and the permutation vectors held in the
    Numeric object.  L is returned in sparse row form with sorted rows, U is
    returned in sparse column form with sorted columns, and P and Q are
    returned as permutation vectors.  See umfpack_get_numeric.h for a more
    detailed description.

    Returns TRUE if successful, FALSE if the Numeric object is invalid or
    if out of memory.

    Dynamic memory usage:  calls UMF_malloc twice, for a total space of
    2*n integers, and then frees all of it via UMF_free when done.

*/

#include "umf_internal.h"
#include "umf_valid_numeric.h"
#include "umf_malloc.h"
#include "umf_free.h"

#ifndef NDEBUG
PRIVATE Int init_count ;
#endif

PRIVATE void get_L
(
    Int Lp [ ],
    Int Li [ ],
    double Lx [ ],
    NumericType *Numeric,
    Int Pattern [ ],
    Int Wi [ ]
) ;

PRIVATE void get_U
(
    Int Up [ ],
    Int Ui [ ],
    double Ux [ ],
    NumericType *Numeric,
    Int Pattern [ ],
    Int Wi [ ]
) ;

/* ========================================================================== */

GLOBAL Int UMFPACK_get_numeric
(
    Int Lp [ ],
    Int Li [ ],
    double Lx [ ],
    Int Up [ ],
    Int Ui [ ],
    double Ux [ ],
    Int P [ ],
    Int Q [ ],
    void *NumericHandle
)
{

    /* ---------------------------------------------------------------------- */
    /* local variables */
    /* ---------------------------------------------------------------------- */

    NumericType *Numeric ;
    Int getL, getU, *Rperm, *Cperm, k, n, *Wi, *Pattern ;

#ifndef NDEBUG
    init_count = UMF_malloc_count ;
#endif

    Wi = (Int *) NULL ;
    Pattern = (Int *) NULL ;

    /* ---------------------------------------------------------------------- */
    /* check input parameters */
    /* ---------------------------------------------------------------------- */

    Numeric = (NumericType *) NumericHandle ;
    if (!UMF_valid_numeric (Numeric))
    {
	return (UMFPACK_ERROR_invalid_Numeric_object) ;
    }

    n = Numeric->n ;

    /* ---------------------------------------------------------------------- */
    /* allocate workspace */
    /* ---------------------------------------------------------------------- */

    getL = Lp && Li && Lx ;
    getU = Up && Ui && Ux ;

    if (getL || getU)
    {
	Wi = (Int *) UMF_malloc (n, sizeof (Int)) ;
	Pattern = (Int *) UMF_malloc (n, sizeof (Int)) ;
	if (!Wi || !Pattern)
	{
	    (void) UMF_free ((void *) Wi) ;
	    (void) UMF_free ((void *) Pattern) ;
	    ASSERT (UMF_malloc_count == init_count) ;
	    return (UMFPACK_ERROR_out_of_memory) ;
	}
    }
    ASSERT (UMF_malloc_count == init_count + 2) ;

    /* ---------------------------------------------------------------------- */
    /* get contents of Numeric */
    /* ---------------------------------------------------------------------- */

    if (P)
    {
	Rperm = Numeric->Rperm ;
	for (k = 0 ; k < n ; k++)
	{
	    P [k] = Rperm [k] ;
	}
    }

    if (Q)
    {
	Cperm = Numeric->Cperm ;
	for (k = 0 ; k < n ; k++)
	{
	    Q [k] = Cperm [k] ;
	}
    }

    if (getL)
    {
	get_L (Lp, Li, Lx, Numeric, Pattern, Wi) ;
    }

    if (getU)
    {
	get_U (Up, Ui, Ux, Numeric, Pattern, Wi) ;
    }

    /* ---------------------------------------------------------------------- */
    /* free the workspace */
    /* ---------------------------------------------------------------------- */

    (void) UMF_free ((void *) Wi) ;
    (void) UMF_free ((void *) Pattern) ;
    ASSERT (UMF_malloc_count == init_count) ;

    return (UMFPACK_OK) ;
}


/* ========================================================================== */
/* === get_L ================================================================ */
/* ========================================================================== */

/*
    The matrix L is stored in the following arrays in the Numeric object:

	Int Lpos [0..n-1]
	Int Lip [0..n-1], index into Numeric->Memory
	Int Lilen [0..n-1]
	Unit *(Numeric->Memory), pointer to memory space holding row indices
		and numerical values

    Let L_k denote the pattern of entries in column k of L (excluding the
    diagonal).

    An Lchain is a sequence of columns of L whose nonzero patterns are related.
    The start of an Lchain is denoted by a negative value of Lip [k].

    To obtain L_k:

    (1)	If column k starts an Lchain, then L_k is stored in its entirety.
	|Lip [k]| is an index into Numeric->Memory for the integer row indices
	in L_k.  The number of entries in the column is |L_k| = Lilen [k].
	This defines the pattern of the "leading" column of this chain.
	Lpos [k] is not used for the first column in the chain.  Column zero
	is always a leading column.

    (2) If column k does not start an Lchain, then L_k is represented as a
	superset of L_k-1.  Define Lnew_k such that (L_k-1 - {k} union Lnew_k)
	= L_k, where Lnew_k and (L_k-1)-{k} are disjoint.  Lnew_k are the
	entries in L_k that are not in L_k-1.  Lpos [k] holds the position of
	pivot row index k in the prior pattern L_k-1 (if it is present), so
	that the set subtraction (L_k-1)-{k} can be computed quickly, when
	computing the pattern of L_k from L_k-1.  The number of new entries in
	L_k is stored in Lilen [k] = |Lnew_k|.

	Note that this means we must have the pattern L_k-1 to compute L_k.

    In both cases (1) and (2), we obtain the pattern L_k.

    The numerical values are stored in Numeric->Memory, starting at the index
    ABS (Lip [k]) + Lilen [k].  It is stored in the same order as the entries
    in L_k, after L_k is obtained from cases (1) or (2), above.

    The advantage of using this "packed" data structure is that it can
    dramatically reduce the amount of storage needed for the pattern of L.
    The disadvantage is that it can be difficult for the user to access,
    and it does not match the sparse matrix data structure used in MATLAB.
    Thus, this routine is provided to create a conventional sparse matrix
    data structure for L, in sparse-row form.  A row-form of L appears to
    MATLAB to be a column-oriented from of the transpose of L.  If you would
    like a column-form of L, then use UMFPACK_transpose (an example of this
    is in umfpackmex.c).

*/
/* ========================================================================== */

PRIVATE void get_L
(
    Int Lp [ ],		/* of size n+1 */
    Int Li [ ],		/* of size lnz, where lnz = Lp [n] */
    double Lx [ ],	/* of size lnz */
    NumericType *Numeric,
    Int Pattern [ ],	/* workspace of size n */
    Int Wi [ ]		/* workspace of size n */
)
{
    /* ---------------------------------------------------------------------- */
    /* local variables */
    /* ---------------------------------------------------------------------- */

    Int deg, *ip, j, row, n, *Lpos, *Lilen, *Lip, p, llen,
	lnz2, lp, newLchain, k, pos ;
    double *xp, value ;

    /* ---------------------------------------------------------------------- */
    /* get parameters */
    /* ---------------------------------------------------------------------- */

    DEBUG4 (("get_L start:\n")) ;
    n = Numeric->n ;
    Lpos = Numeric->Lpos ;
    Lilen = Numeric->Lilen ;
    Lip = Numeric->Lip ;

    /* ---------------------------------------------------------------------- */
    /* count the nonzeros in each row of L */
    /* ---------------------------------------------------------------------- */

    for (row = 0 ; row < n ; row++)
    {
	/* include the diagonal entry in the row counts */
	Wi [row] = 1 ;
    }

    /* count the nonzero entries in L */
    for (k = 0 ; k < n ; k++)
    {

	/* ------------------------------------------------------------------ */
	/* make column of L in Pattern [0..deg-1] */
	/* ------------------------------------------------------------------ */

	lp = Lip [k] ;
	newLchain = (lp < 0) ;
	if (newLchain)
	{
	    lp = -lp ;
	    deg = 0 ;
	    DEBUG4 (("start of chain for column of L\n")) ;
	}

	/* remove pivot row */
	pos = Lpos [k] ;
	if (pos != EMPTY)
	{
	    DEBUG4 (("  k "ID" removing row "ID" at position "ID"\n",
	    k, Pattern [pos], pos)) ;
	    ASSERT (!newLchain) ;
	    ASSERT (deg > 0) ;
	    ASSERT (pos >= 0 && pos < deg) ;
	    ASSERT (Pattern [pos] == k) ;
	    Pattern [pos] = Pattern [--deg] ;
	}

	/* concatenate the pattern */
	ip = (Int *) (Numeric->Memory + lp) ;
	llen = Lilen [k] ;
	for (j = 0 ; j < llen ; j++)
	{
	    row = *ip++ ;
	    DEBUG4 (("  row "ID"  k "ID"\n", row, k)) ;
	    ASSERT (row > k) ;
	    Pattern [deg++] = row ;
	}

	xp = (double *) (Numeric->Memory + lp + UNITS (Int, llen)) ;

	for (j = 0 ; j < deg ; j++)
	{
	    DEBUG4 (("  row "ID"  k "ID" value %g\n", Pattern [j], k, *xp)) ;
	    row = Pattern [j] ;
	    value = *xp++ ;
	    if (value != 0.0)
	    {
		Wi [row]++ ;
	    }
	}
    }

    /* ---------------------------------------------------------------------- */
    /* construct the final row form of L */
    /* ---------------------------------------------------------------------- */

    /* create the row pointers */
    lnz2 = 0 ;
    for (row = 0 ; row < n ; row++)
    {
	Lp [row] = lnz2 ;
	lnz2 += Wi [row] ;
	Wi [row] = Lp [row] ;
    }
    Lp [n] = lnz2 ;
    ASSERT (Numeric->lnz + n == lnz2) ;

    /* add entries from the rows of L */
    for (k = 0 ; k < n ; k++)
    {

	/* ------------------------------------------------------------------ */
	/* make column of L in Pattern [0..deg-1] */
	/* ------------------------------------------------------------------ */

	lp = Lip [k] ;
	newLchain = (lp < 0) ;
	if (newLchain)
	{
	    lp = -lp ;
	    deg = 0 ;
	    DEBUG4 (("start of chain for column of L\n")) ;
	}

	/* remove pivot row */
	pos = Lpos [k] ;
	if (pos != EMPTY)
	{
	    DEBUG4 (("  k "ID" removing row "ID" at position "ID"\n",
	    k, Pattern [pos], pos)) ;
	    ASSERT (!newLchain) ;
	    ASSERT (deg > 0) ;
	    ASSERT (pos >= 0 && pos < deg) ;
	    ASSERT (Pattern [pos] == k) ;
	    Pattern [pos] = Pattern [--deg] ;
	}

	/* concatenate the pattern */
	ip = (Int *) (Numeric->Memory + lp) ;
	llen = Lilen [k] ;
	for (j = 0 ; j < llen ; j++)
	{
	    row = *ip++ ;
	    DEBUG4 (("  row "ID"  k "ID"\n", row, k)) ;
	    ASSERT (row > k) ;
	    Pattern [deg++] = row ;
	}

	xp = (double *) (Numeric->Memory + lp + UNITS (Int, llen)) ;

	for (j = 0 ; j < deg ; j++)
	{
	    DEBUG4 (("  row "ID"  k "ID" value %g\n", Pattern [j], k, *xp)) ;
	    row = Pattern [j] ;
	    value = *xp++ ;
	    if (value != 0.0)
	    {
		p = Wi [row]++ ;
		Li [p] = k ;
		Lx [p] = value ;
	    }
	}
    }

    /* add all of the diagonal entries (L is unit diagonal) */
    for (row = 0 ; row < n ; row++)
    {
	p = Wi [row]++ ;
	Li [p] = row ;
	Lx [p] = 1.0 ;
	ASSERT (Wi [row] == Lp [row+1]) ;
    }

#ifndef NDEBUG
    DEBUG6 (("L matrix (stored by rows):")) ;
    UMF_dump_col_matrix (Lx, Li, Lp, n, Numeric->lnz + n) ;
#endif

    DEBUG4 (("get_L done:\n")) ;
}


/* ========================================================================== */
/* === get_U ================================================================ */
/* ========================================================================== */

/*
    The matrix U is stored in the following arrays in the Numeric object:

	Int Upos [0..n-1]
	Int Uip [0..n], index into Numeric->Memory
	Int Uilen [0..n]
	Unit *(Numeric->Memory), pointer to memory space holding column indices
		and numerical values

    Let U_k denote the pattern of entries in row k of U (excluding the
    diagonal).

    A Uchain is a sequence of columns of U whose nonzero patterns are related.
    The start of a Uchain is denoted by a negative value of Uip [k].

    To obtain U_k-1:

    (1) If row k is the start of a Uchain then Uip [k] is negative and |Uip [k]|
	is an index into Numeric->Memory for the integer column indices in
	U_k-1.  The number of entries in the row is |U_k-1| = Uilen [k].  This
	defines the pattern of the "trailing" row of this chain that ends at
	row k-1.


    (2) If row k is not the start of a Uchain, then U_k-1 is a subset of U_k.
	The indices in U_k are arranged so that last Uilen [k] entries of
	U_k are those indices not in U_k-1.  Next, the pivot column index k is
	added if it appears in row U_k-1 (it never appears in U_k).  Upos [k]
	holds the position of pivot column index k in the pattern U_k-1 (if it
	is present), so that the set union (U_k-1)+{k} can be computed quickly,
	when computing the pattern of U_k-1 from U_k.

	Note that this means we must have the pattern U_k to compute L_k-1.

    In both cases (1) and (2), we obtain the pattern U_k.

    The numerical values are stored in Numeric->Memory.  If k is the start of a
    Uchain, then the offset is |Uip [k]| plus the size of the space needed to
    store the pattern U_k-1.  Otherwise, Uip [k] is the offset itself of the
    numerical values, since in this case no pattern is stored.
    The numerical values are stored in the same order as the entries in U_k,
    after U_k is obtained from cases (1) or (2), above.

    The advantage of using this "packed" data structure is that it can
    dramatically reduce the amount of storage needed for the pattern of U.
    The disadvantage is that it can be difficult for the user to access,
    and it does not match the sparse matrix data structure used in MATLAB.
    Thus, this routine is provided to create a conventional sparse matrix
    data structure for U, in sparse-column form.

*/
/* ========================================================================== */

PRIVATE void get_U
(
    Int Up [ ],		/* of size n+1 */
    Int Ui [ ],		/* of size unz, where unz = Up [n] */
    double Ux [ ],	/* of size unz */
    NumericType *Numeric,
    Int Pattern [ ],	/* workspace of size n */
    Int Wi [ ]		/* workspace of size n */
)
{
    /* ---------------------------------------------------------------------- */
    /* local variables */
    /* ---------------------------------------------------------------------- */

    Int deg, j, *ip, col, *Upos, *Uilen, *Uip, n, ulen,
	unz2, p, k, up, newUchain, pos ;
    double *xp, *D, value ;

    /* ---------------------------------------------------------------------- */
    /* get parameters */
    /* ---------------------------------------------------------------------- */

    DEBUG4 (("get_U start:\n")) ;
    n = Numeric->n ;
    Upos = Numeric->Upos ;
    Uilen = Numeric->Uilen ;
    Uip = Numeric->Uip ;
    D = Numeric->D ;

    /* ---------------------------------------------------------------------- */
    /* count the nonzeros in each column of U */
    /* ---------------------------------------------------------------------- */

    for (col = 0 ; col < n ; col++)
    {
	/* include the diagonal entry in the column counts */
	Wi [col] = 1 ;
    }

    deg = 0 ;

    for (k = n-1 ; k >= 0 ; k--)
    {

	/* ------------------------------------------------------------------ */
	/* use row k of U */
	/* ------------------------------------------------------------------ */

	up = Uip [k] ;
	ulen = Uilen [k] ;
	newUchain = (up < 0) ;
	if (newUchain)
	{
	    up = -up ;
	    xp = (double *) (Numeric->Memory + up + UNITS (Int, ulen)) ;
	}
	else
	{
	    xp = (double *) (Numeric->Memory + up) ;
	}

	for (j = 0 ; j < deg ; j++)
	{
	    DEBUG4 (("  k "ID" col "ID" value %g\n", k, Pattern [j], *xp)) ;
	    col = Pattern [j] ;
	    ASSERT (col >= 0 && col < n) ;
	    value = *xp++ ;
	    if (value != 0.0)
	    {
		Wi [col]++ ;
	    }
	}

	/* ------------------------------------------------------------------ */
	/* make row k-1 of U in Pattern [0..deg-1] */
	/* ------------------------------------------------------------------ */

	if (newUchain)
	{
	    /* next row is a new Uchain */
	    deg = ulen ;
	    DEBUG4 (("end of chain for row of U "ID" deg "ID"\n", k-1, deg)) ;
	    ip = (Int *) (Numeric->Memory + up) ;
	    for (j = 0 ; j < deg ; j++)
	    {
		col = *ip++ ;
		DEBUG4 (("  k "ID" col "ID"\n", k-1, col)) ;
		ASSERT (k <= col) ;
		Pattern [j] = col ;
	    }
	}
	else
	{
	    deg -= ulen ;
	    DEBUG4 (("middle of chain for row of U "ID" deg "ID"\n", k-1, deg));
	    ASSERT (deg >= 0) ;
	    pos = Upos [k] ;
	    if (pos != EMPTY)
	    {
		/* add the pivot column */
		DEBUG4 (("k "ID" add pivot entry at position "ID"\n", k, pos)) ;
		ASSERT (pos >= 0 && pos <= deg) ;
		Pattern [deg++] = Pattern [pos] ;
		Pattern [pos] = k ;
	    }
	}
    }

    /* ---------------------------------------------------------------------- */
    /* construct the final column form of U */
    /* ---------------------------------------------------------------------- */

    /* create the column pointers */
    unz2 = 0 ;
    for (col = 0 ; col < n ; col++)
    {
	Up [col] = unz2 ;
	unz2 += Wi [col] ;
    }
    Up [n] = unz2 ;
    DEBUG1 (("unz "ID"\n", unz2)) ;
    ASSERT (Numeric->unz + n == unz2) ;

    for (col = 0 ; col < n ; col++)
    {
	Wi [col] = Up [col+1] ;
    }

    /* add all of the diagonal entries (sparse part only) */
    for (col = 0 ; col < n ; col++)
    {
	p = --(Wi [col]) ;
	Ui [p] = col ;
	Ux [p] = D [col] ;
    }

    /* add all the entries from the sparse rows of U */

    deg = 0 ;

    for (k = n-1 ; k >= 0 ; k--)
    {

	/* ------------------------------------------------------------------ */
	/* use row k of U */
	/* ------------------------------------------------------------------ */

	up = Uip [k] ;
	ulen = Uilen [k] ;
	newUchain = (up < 0) ;
	if (newUchain)
	{
	    up = -up ;
	    xp = (double *) (Numeric->Memory + up + UNITS (Int, ulen)) ;
	}
	else
	{
	    xp = (double *) (Numeric->Memory + up) ;
	}

	xp += deg ;
	for (j = deg-1 ; j >= 0 ; j--)
	{
	    DEBUG4 (("  k "ID" col "ID" value %g\n", k, Pattern [j], *xp)) ;
	    col = Pattern [j] ;
	    ASSERT (col >= 0 && col < n) ;
	    value = *(--xp) ;
	    if (value != 0.0)
	    {
		p = --(Wi [col]) ;
		Ui [p] = k ;
		Ux [p] = value ;
	    }
	}

	/* ------------------------------------------------------------------ */
	/* make row k-1 of U in Pattern [0..deg-1] */
	/* ------------------------------------------------------------------ */

	if (newUchain)
	{
	    /* next row is a new Uchain */
	    deg = ulen ;
	    DEBUG4 (("end of chain for row of U "ID" deg "ID"\n", k-1, deg)) ;
	    ip = (Int *) (Numeric->Memory + up) ;
	    for (j = 0 ; j < deg ; j++)
	    {
		col = *ip++ ;
		DEBUG4 (("  k "ID" col "ID"\n", k-1, col)) ;
		ASSERT (k <= col) ;
		Pattern [j] = col ;
	    }
	}
	else
	{
	    deg -= ulen ;
	    DEBUG4 (("middle of chain for row of U "ID" deg "ID"\n", k-1, deg));
	    ASSERT (deg >= 0) ;
	    pos = Upos [k] ;
	    if (pos != EMPTY)
	    {
		/* add the pivot column */
		DEBUG4 (("k "ID" add pivot entry at position "ID"\n", k, pos)) ;
		ASSERT (pos >= 0 && pos <= deg) ;
		Pattern [deg++] = Pattern [pos] ;
		Pattern [pos] = k ;
	    }
	}
    }

#ifndef NDEBUG
    DEBUG6 (("U matrix:")) ;
    UMF_dump_col_matrix (Ux, Ui, Up, n, Numeric->unz + n) ;
#endif

}

