/****************************************************************************
*
* NAME: smsPitchScale.cp
* VERSION: 1.01
* HOME URL: http://www.dspdimension.com
* KNOWN BUGS: none
*
* SYNOPSIS: Routine for doing pitch scaling while maintaining
* duration using the Short Time Fourier Transform.
*
* DESCRIPTION: The routine takes a pitchScale factor value which is between 0.5
* (one octave down) and 2. (one octave up). A value of exactly 1 does not change
* the pitch. numSampsToProcess tells the routine how many samples in indata[0...
* numSampsToProcess-1] should be pitch scaled and moved to outdata[0 ...
* numSampsToProcess-1]. The two buffers can be identical (ie. it can process the
* data in-place). fftFrameLength defines the FFT frame size used for the
* processing. Typical values are 1024, 2048 and 4096. It may be any value <=
* MAX_FFT_FRAME_LENGTH but it MUST be a power of 2. osamp is the STFT
* oversampling factor which also determines the overlap between adjacent STFT
* frames. It should at least be 4 for moderate scaling ratios. A value of 32 is
* recommended for best quality. sampleRate takes the sample rate for the signal 
* in unit Hz, ie. 44100 for 44.1 kHz audio. The data passed to the routine in 
* indata[] should be in the range [-1.0, 1.0), which is also the output range 
* for the data. 
*
* COPYRIGHT 1999 Stephan M. Sprenger <sms@dspdimension.com>
*
* 						The Wide Open License (WOL)
*
* Permission to use, copy, modify, distribute and sell this software and its
* documentation for any purpose is hereby granted without fee, provided that
* the above copyright notice and this license appear in all source copies. 
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY OF
* ANY KIND. See http://www.dspguru.com/wol.htm for more information.
*
*****************************************************************************/

#include <string.h>
#include "config.h"
#include <math.h>

#ifdef EXPLICIT_S
#include <srfftw.h>
#else
#include <rfftw.h>
#endif //EXPLICIT_S

#include "pitchscale.h"

void pitch_scale(sbuffers *buffers, float pitchScale, long fftFrameLength, long osamp, long numSampsToProcess, float sampleRate, float *indata, float *outdata, int adding, float gain) {
/*
	Routine smsPitchScale(). See top of file for explanation
	Purpose: doing pitch scaling while maintaining duration using the Short
	Time Fourier Transform.
	Author: (c)1999 Stephan M. Sprenger <sms@dspdimension.com>
*/
	double magn, phase, tmp, real, imag;
	double freqPerBin, expct, fadeZoneLen;
	long i,k, qpd, index, inFifoLatency, stepSize, fftFrameSize2;
	fftw_real in[MAX_FRAME_LENGTH*2], out[MAX_FRAME_LENGTH*2];

	float *gInFIFO = buffers->gInFIFO;
	float *gOutFIFO = buffers->gOutFIFO;
	float *gLastPhase = buffers->gLastPhase;
	float *gSumPhase = buffers->gSumPhase;
	float *gOutputAccum = buffers->gOutputAccum;
	float *gAnaFreq = buffers->gAnaFreq;
	float *gAnaMagn = buffers->gAnaMagn;
	float *gSynFreq = buffers->gSynFreq;
	float *gSynMagn = buffers->gSynMagn;
	float *gWindow = buffers->gWindow;
	long gRover = buffers->gRover;

	/* set up some handy variables */
	fadeZoneLen = fftFrameLength/2;
	fftFrameSize2 = fftFrameLength/2;
	stepSize = fftFrameLength/osamp;
	freqPerBin = sampleRate/(double)fftFrameLength;
	expct = 2.*M_PI*(double)stepSize/(double)fftFrameLength;
	inFifoLatency = fftFrameLength-stepSize;
	if (gRover == false) gRover = inFifoLatency;

	/* main processing loop */
	for (i = 0; i < numSampsToProcess; i++){

		/* As long as we have not yet collected enough data just read in */
		gInFIFO[gRover] = indata[i];
		if (adding) {
			outdata[i] += (gOutFIFO[gRover-inFifoLatency] * gain);
		} else {
			outdata[i] = gOutFIFO[gRover-inFifoLatency];
		}
		gRover++;

		/* As long as we have not yet collected enough data just read in */
		/* now we have enough data for processing */
		if (gRover >= fftFrameLength) {
			gRover = inFifoLatency;

			/* do windowing and store */
			for (k = 0; k < fftFrameLength; k++) {
				in[k] = gInFIFO[k] * gWindow[k];
			}

		/* As long as we have not yet collected enough data just read in */

			/* ***************** ANALYSIS ******************* */
			/* do transform */
			rfftw_one(aplan, in, out);

			/* this is the analysis step */
			for (k = 0; k <= fftFrameSize2; k++) {

				real = out[k];
				imag = out[(2*fftFrameLength) - k];

				/* compute magnitude and phase */
				magn = 2.0f*sqrt(real*real + imag*imag);
				phase = atan2(imag,real);

				/* compute phase difference */
				tmp = phase - gLastPhase[k];
				gLastPhase[k] = phase;

				/* subtract expected phase difference */
				tmp -= (double)k*expct;

				/* map delta phase into +/- Pi interval */
				qpd = tmp/M_PI;
				if (qpd >= 0) qpd += qpd&1;
				else qpd -= qpd&1;
				tmp -= M_PI*(double)qpd;

				/* get deviation from bin frequency from the +/- Pi interval */
				tmp = osamp*tmp/(2.0f*M_PI);

				/* compute the k-th partials' true frequency */
				tmp = (double)k*freqPerBin + tmp*freqPerBin;

				/* store magnitude and true frequency in analysis arrays */
				gAnaMagn[k] = magn;
				gAnaFreq[k] = tmp;

			}

			/* ***************** PROCESSING ******************* */
			/* this does the actual pitch scaling */
			memset(gSynMagn, 0, fftFrameLength*sizeof(float));
			memset(gSynFreq, 0, fftFrameLength*sizeof(float));
			for (k = 0; k <= fftFrameSize2; k++) {
				index = k/pitchScale;
				if (index <= fftFrameSize2) {
					/* new bin overrides existing if magnitude is higher */ 
					if (gAnaMagn[index] > gSynMagn[k]) {
						gSynMagn[k] = gAnaMagn[index];
						gSynFreq[k] = gAnaFreq[index] * pitchScale;
					}

					/* fill empty bins with nearest neighbour */

					if ((gSynFreq[k] == 0.) && (k > 0)) {
						gSynFreq[k] = gSynFreq[k-1];
						gSynMagn[k] = gSynMagn[k-1];
					}
				}
			}


			/* ***************** SYNTHESIS ******************* */
			/* this is the synthesis step */
			for (k = 0; k <= fftFrameSize2; k++) {

				/* get magnitude and true frequency from synthesis arrays */
				magn = gSynMagn[k];
				tmp = gSynFreq[k];

				/* subtract bin mid frequency */
				tmp -= (double)k*freqPerBin;

				/* get bin deviation from freq deviation */
				tmp /= freqPerBin;

				/* take osamp into account */
				tmp = 2.*M_PI*tmp/osamp;

				/* add the overlap phase advance back in */
				tmp += (double)k*expct;

				/* accumulate delta phase to get bin phase */
				gSumPhase[k] += tmp;
				phase = gSumPhase[k];

				in[k] = magn*cos(phase);
				in[fftFrameLength - k] = magn*sin(phase);
			} 

			/* do inverse transform */
			rfftw_one(splan, in, out);

			/* do windowing and add to output accumulator */ 
			for(k=0; k < fftFrameLength; k++) {
				gOutputAccum[k] += 2.0f*gWindow[k]*out[k]/(fftFrameSize2*osamp);
			}
			for (k = 0; k < stepSize; k++) gOutFIFO[k] = gOutputAccum[k];

			/* shift accumulator */
			memmove(gOutputAccum, gOutputAccum+stepSize, fftFrameLength*sizeof(float));

			/* move input FIFO */
			for (k = 0; k < inFifoLatency; k++) gInFIFO[k] = gInFIFO[k+stepSize];
		}
	}

	buffers->gRover = gRover;
}
