#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#include "overdrive-data.h"
#include "ladspa-util.h"

#define OVERDRIVE_LIMIT                0
#define OVERDRIVE_DRV                  1
#define OVERDRIVE_LOCOLOUR             2
#define OVERDRIVE_HICOLOUR             3
#define OVERDRIVE_INPUT                4
#define OVERDRIVE_OUTPUT               5
#define OVERDRIVE_S_LIMIT              0
#define OVERDRIVE_S_DRIVE              1
#define OVERDRIVE_S_LOCOLOUR           2
#define OVERDRIVE_S_HICOLOUR           3
#define OVERDRIVE_S_INPUT1             4
#define OVERDRIVE_S_INPUT2             5
#define OVERDRIVE_S_OUTPUT1            6
#define OVERDRIVE_S_OUTPUT2            7

LADSPA_Descriptor *overdriveDescriptor = NULL;

typedef struct {
	LADSPA_Data *limit;
	LADSPA_Data *drv;
	LADSPA_Data *locolour;
	LADSPA_Data *hicolour;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data run_adding_gain;
} Overdrive;

LADSPA_Descriptor *overdrive_sDescriptor = NULL;

typedef struct {
	LADSPA_Data *limit;
	LADSPA_Data *drive;
	LADSPA_Data *locolour;
	LADSPA_Data *hicolour;
	LADSPA_Data *input1;
	LADSPA_Data *input2;
	LADSPA_Data *output1;
	LADSPA_Data *output2;
	LADSPA_Data run_adding_gain;
} Overdrive_s;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return overdriveDescriptor;
	case 1:
		return overdrive_sDescriptor;
	default:
		return NULL;
	}
}

void cleanupOverdrive(LADSPA_Handle instance) {
	free(instance);
}

void connectPortOverdrive(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Overdrive *plugin;

	plugin = (Overdrive *)instance;
	switch (port) {
	case OVERDRIVE_LIMIT:
		plugin->limit = data;
		break;
	case OVERDRIVE_DRV:
		plugin->drv = data;
		break;
	case OVERDRIVE_LOCOLOUR:
		plugin->locolour = data;
		break;
	case OVERDRIVE_HICOLOUR:
		plugin->hicolour = data;
		break;
	case OVERDRIVE_INPUT:
		plugin->input = data;
		break;
	case OVERDRIVE_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateOverdrive(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Overdrive *plugin_data = (Overdrive *)malloc(sizeof(Overdrive));
	plugin_data->run_adding_gain = 1.0f;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runOverdrive(LADSPA_Handle instance, unsigned long sample_count) {
	Overdrive *plugin_data = (Overdrive *)instance;

	/* Amps limit (in dB relative to nominal 0) (float value) */
	LADSPA_Data limit = *(plugin_data->limit);

	/* Drive level (float value) */
	LADSPA_Data drv = *(plugin_data->drv);

	/* Low density colouration (float value) */
	LADSPA_Data locolour = *(plugin_data->locolour);

	/* High density colouration (float value) */
	LADSPA_Data hicolour = *(plugin_data->hicolour);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;
	LADSPA_Data gain_fix = 1.0 / pow(10, limit/20);
	
	for (pos = 0; pos < sample_count; pos++) {
	        LADSPA_Data amp = *(input++) * gain_fix;
	        int table_pos, sign;
	
	        sign = amp < 0 ? -1 : 1;
	        amp *= sign;
	        table_pos = LIMIT(amp * 16384, 0, 16383);
	
	        buffer_write(*(output++), ((1.0f-drv) * amp + drv * drive[table_pos] + 
	               locolour * low[table_pos] + hicolour * high[table_pos]) * sign);
	}
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainOverdrive(LADSPA_Handle instance, LADSPA_Data gain) {
	((Overdrive *)instance)->run_adding_gain = gain;
}

void runAddingOverdrive(LADSPA_Handle instance, unsigned long sample_count) {
	Overdrive *plugin_data = (Overdrive *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Amps limit (in dB relative to nominal 0) (float value) */
	LADSPA_Data limit = *(plugin_data->limit);

	/* Drive level (float value) */
	LADSPA_Data drv = *(plugin_data->drv);

	/* Low density colouration (float value) */
	LADSPA_Data locolour = *(plugin_data->locolour);

	/* High density colouration (float value) */
	LADSPA_Data hicolour = *(plugin_data->hicolour);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;
	LADSPA_Data gain_fix = 1.0 / pow(10, limit/20);
	
	for (pos = 0; pos < sample_count; pos++) {
	        LADSPA_Data amp = *(input++) * gain_fix;
	        int table_pos, sign;
	
	        sign = amp < 0 ? -1 : 1;
	        amp *= sign;
	        table_pos = LIMIT(amp * 16384, 0, 16383);
	
	        buffer_write(*(output++), ((1.0f-drv) * amp + drv * drive[table_pos] + 
	               locolour * low[table_pos] + hicolour * high[table_pos]) * sign);
	}
}

void cleanupOverdrive_s(LADSPA_Handle instance) {
	free(instance);
}

void connectPortOverdrive_s(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Overdrive_s *plugin;

	plugin = (Overdrive_s *)instance;
	switch (port) {
	case OVERDRIVE_S_LIMIT:
		plugin->limit = data;
		break;
	case OVERDRIVE_S_DRIVE:
		plugin->drive = data;
		break;
	case OVERDRIVE_S_LOCOLOUR:
		plugin->locolour = data;
		break;
	case OVERDRIVE_S_HICOLOUR:
		plugin->hicolour = data;
		break;
	case OVERDRIVE_S_INPUT1:
		plugin->input1 = data;
		break;
	case OVERDRIVE_S_INPUT2:
		plugin->input2 = data;
		break;
	case OVERDRIVE_S_OUTPUT1:
		plugin->output1 = data;
		break;
	case OVERDRIVE_S_OUTPUT2:
		plugin->output2 = data;
		break;
	}
}

LADSPA_Handle instantiateOverdrive_s(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Overdrive_s *plugin_data = (Overdrive_s *)malloc(sizeof(Overdrive_s));
	plugin_data->run_adding_gain = 1.0f;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runOverdrive_s(LADSPA_Handle instance, unsigned long sample_count) {
	Overdrive_s *plugin_data = (Overdrive_s *)instance;

	/* Amps limit (in dB relative to nominal 0) (float value) */
	LADSPA_Data limit = *(plugin_data->limit);

	/* Drive level (float value) */
	LADSPA_Data drive = *(plugin_data->drive);

	/* Low density colouration (float value) */
	LADSPA_Data locolour = *(plugin_data->locolour);

	/* High density colouration (float value) */
	LADSPA_Data hicolour = *(plugin_data->hicolour);

	/* Input L (array of floats of length sample_count) */
	LADSPA_Data *input1 = plugin_data->input1;

	/* Input R (array of floats of length sample_count) */
	LADSPA_Data *input2 = plugin_data->input2;

	/* Output L (array of floats of length sample_count) */
	LADSPA_Data *output1 = plugin_data->output1;

	/* Output R (array of floats of length sample_count) */
	LADSPA_Data *output2 = plugin_data->output2;

	unsigned long pos;
	LADSPA_Data gain_fix = pow(10, limit/20);
	
	for (pos = 0; pos < sample_count; pos++) {
	        /* Amplitude of signal */
	        LADSPA_Data amp1 = *(input1++) / gain_fix;
	        LADSPA_Data amp2 = *(input2++) / gain_fix;
	
	        buffer_write(*(output1++), ((((1.0 - drive) * amp1) + (((1.0 / (1.0 + exp(-4.0 * amp1))) - 0.5) * 2.0 * drive) + (0.1 * locolour * sin(amp1 * 23) * amp1) + (0.1 * hicolour * cos(amp1 * 997) * amp1))) * gain_fix);
	        buffer_write(*(output2++), ((((1.0 - drive) * amp2) + (((1.0 / (1.0 + exp(-4.0 * amp2))) - 0.5) * 2.0 * drive) + (0.1 * locolour * sin(amp2 * 23) * amp2) + (0.1 * hicolour * cos(amp2 * 997) * amp2))) * gain_fix);
	}
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainOverdrive_s(LADSPA_Handle instance, LADSPA_Data gain) {
	((Overdrive_s *)instance)->run_adding_gain = gain;
}

void runAddingOverdrive_s(LADSPA_Handle instance, unsigned long sample_count) {
	Overdrive_s *plugin_data = (Overdrive_s *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Amps limit (in dB relative to nominal 0) (float value) */
	LADSPA_Data limit = *(plugin_data->limit);

	/* Drive level (float value) */
	LADSPA_Data drive = *(plugin_data->drive);

	/* Low density colouration (float value) */
	LADSPA_Data locolour = *(plugin_data->locolour);

	/* High density colouration (float value) */
	LADSPA_Data hicolour = *(plugin_data->hicolour);

	/* Input L (array of floats of length sample_count) */
	LADSPA_Data *input1 = plugin_data->input1;

	/* Input R (array of floats of length sample_count) */
	LADSPA_Data *input2 = plugin_data->input2;

	/* Output L (array of floats of length sample_count) */
	LADSPA_Data *output1 = plugin_data->output1;

	/* Output R (array of floats of length sample_count) */
	LADSPA_Data *output2 = plugin_data->output2;

	unsigned long pos;
	LADSPA_Data gain_fix = pow(10, limit/20);
	
	for (pos = 0; pos < sample_count; pos++) {
	        /* Amplitude of signal */
	        LADSPA_Data amp1 = *(input1++) / gain_fix;
	        LADSPA_Data amp2 = *(input2++) / gain_fix;
	
	        buffer_write(*(output1++), ((((1.0 - drive) * amp1) + (((1.0 / (1.0 + exp(-4.0 * amp1))) - 0.5) * 2.0 * drive) + (0.1 * locolour * sin(amp1 * 23) * amp1) + (0.1 * hicolour * cos(amp1 * 997) * amp1))) * gain_fix);
	        buffer_write(*(output2++), ((((1.0 - drive) * amp2) + (((1.0 / (1.0 + exp(-4.0 * amp2))) - 0.5) * 2.0 * drive) + (0.1 * locolour * sin(amp2 * 23) * amp2) + (0.1 * hicolour * cos(amp2 * 997) * amp2))) * gain_fix);
	}
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	overdriveDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (overdriveDescriptor) {
		overdriveDescriptor->UniqueID = 1182;
		overdriveDescriptor->Label = strdup("overdrive");
		overdriveDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		overdriveDescriptor->Name =
		 strdup("Mono Overdrive");
		overdriveDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		overdriveDescriptor->Copyright =
		 strdup("GPL");
		overdriveDescriptor->PortCount = 6;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(6,
		 sizeof(LADSPA_PortDescriptor));
		overdriveDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(6,
		 sizeof(LADSPA_PortRangeHint));
		overdriveDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(6, sizeof(char*));
		overdriveDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Amps limit (in dB relative to nominal 0) */
		port_descriptors[OVERDRIVE_LIMIT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[OVERDRIVE_LIMIT] =
		 strdup("Amps limit (in dB relative to nominal 0)");
		port_range_hints[OVERDRIVE_LIMIT].HintDescriptor = 0;

		/* Parameters for Drive level */
		port_descriptors[OVERDRIVE_DRV] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[OVERDRIVE_DRV] =
		 strdup("Drive level");
		port_range_hints[OVERDRIVE_DRV].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_DRV].LowerBound = 0;
		port_range_hints[OVERDRIVE_DRV].UpperBound = +1;

		/* Parameters for Low density colouration */
		port_descriptors[OVERDRIVE_LOCOLOUR] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[OVERDRIVE_LOCOLOUR] =
		 strdup("Low density colouration");
		port_range_hints[OVERDRIVE_LOCOLOUR].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_LOCOLOUR].LowerBound = 0;
		port_range_hints[OVERDRIVE_LOCOLOUR].UpperBound = +1;

		/* Parameters for High density colouration */
		port_descriptors[OVERDRIVE_HICOLOUR] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[OVERDRIVE_HICOLOUR] =
		 strdup("High density colouration");
		port_range_hints[OVERDRIVE_HICOLOUR].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_HICOLOUR].LowerBound = 0;
		port_range_hints[OVERDRIVE_HICOLOUR].UpperBound = +1;

		/* Parameters for Input */
		port_descriptors[OVERDRIVE_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[OVERDRIVE_INPUT] =
		 strdup("Input");
		port_range_hints[OVERDRIVE_INPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_INPUT].LowerBound = -1;
		port_range_hints[OVERDRIVE_INPUT].UpperBound = +1;

		/* Parameters for Output */
		port_descriptors[OVERDRIVE_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[OVERDRIVE_OUTPUT] =
		 strdup("Output");
		port_range_hints[OVERDRIVE_OUTPUT].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_OUTPUT].LowerBound = -1;
		port_range_hints[OVERDRIVE_OUTPUT].UpperBound = +1;

		overdriveDescriptor->activate = NULL;
		overdriveDescriptor->cleanup = cleanupOverdrive;
		overdriveDescriptor->connect_port = connectPortOverdrive;
		overdriveDescriptor->deactivate = NULL;
		overdriveDescriptor->instantiate = instantiateOverdrive;
		overdriveDescriptor->run = runOverdrive;
		overdriveDescriptor->run_adding = runAddingOverdrive;
		overdriveDescriptor->set_run_adding_gain = setRunAddingGainOverdrive;
	}

	overdrive_sDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (overdrive_sDescriptor) {
		overdrive_sDescriptor->UniqueID = 1183;
		overdrive_sDescriptor->Label = strdup("overdrive_s");
		overdrive_sDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		overdrive_sDescriptor->Name =
		 strdup("Stereo Overdrive");
		overdrive_sDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		overdrive_sDescriptor->Copyright =
		 strdup("GPL");
		overdrive_sDescriptor->PortCount = 8;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(8,
		 sizeof(LADSPA_PortDescriptor));
		overdrive_sDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(8,
		 sizeof(LADSPA_PortRangeHint));
		overdrive_sDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(8, sizeof(char*));
		overdrive_sDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Amps limit (in dB relative to nominal 0) */
		port_descriptors[OVERDRIVE_S_LIMIT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[OVERDRIVE_S_LIMIT] =
		 strdup("Amps limit (in dB relative to nominal 0)");
		port_range_hints[OVERDRIVE_S_LIMIT].HintDescriptor = 0;

		/* Parameters for Drive level */
		port_descriptors[OVERDRIVE_S_DRIVE] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[OVERDRIVE_S_DRIVE] =
		 strdup("Drive level");
		port_range_hints[OVERDRIVE_S_DRIVE].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_S_DRIVE].LowerBound = 0;
		port_range_hints[OVERDRIVE_S_DRIVE].UpperBound = 1;

		/* Parameters for Low density colouration */
		port_descriptors[OVERDRIVE_S_LOCOLOUR] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[OVERDRIVE_S_LOCOLOUR] =
		 strdup("Low density colouration");
		port_range_hints[OVERDRIVE_S_LOCOLOUR].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW;
		port_range_hints[OVERDRIVE_S_LOCOLOUR].LowerBound = 0;

		/* Parameters for High density colouration */
		port_descriptors[OVERDRIVE_S_HICOLOUR] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[OVERDRIVE_S_HICOLOUR] =
		 strdup("High density colouration");
		port_range_hints[OVERDRIVE_S_HICOLOUR].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW;
		port_range_hints[OVERDRIVE_S_HICOLOUR].LowerBound = 0;

		/* Parameters for Input L */
		port_descriptors[OVERDRIVE_S_INPUT1] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[OVERDRIVE_S_INPUT1] =
		 strdup("Input L");
		port_range_hints[OVERDRIVE_S_INPUT1].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_S_INPUT1].LowerBound = -1;
		port_range_hints[OVERDRIVE_S_INPUT1].UpperBound = +1;

		/* Parameters for Input R */
		port_descriptors[OVERDRIVE_S_INPUT2] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[OVERDRIVE_S_INPUT2] =
		 strdup("Input R");
		port_range_hints[OVERDRIVE_S_INPUT2].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_S_INPUT2].LowerBound = -1;
		port_range_hints[OVERDRIVE_S_INPUT2].UpperBound = +1;

		/* Parameters for Output L */
		port_descriptors[OVERDRIVE_S_OUTPUT1] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[OVERDRIVE_S_OUTPUT1] =
		 strdup("Output L");
		port_range_hints[OVERDRIVE_S_OUTPUT1].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_S_OUTPUT1].LowerBound = -1;
		port_range_hints[OVERDRIVE_S_OUTPUT1].UpperBound = +1;

		/* Parameters for Output R */
		port_descriptors[OVERDRIVE_S_OUTPUT2] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[OVERDRIVE_S_OUTPUT2] =
		 strdup("Output R");
		port_range_hints[OVERDRIVE_S_OUTPUT2].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[OVERDRIVE_S_OUTPUT2].LowerBound = -1;
		port_range_hints[OVERDRIVE_S_OUTPUT2].UpperBound = +1;

		overdrive_sDescriptor->activate = NULL;
		overdrive_sDescriptor->cleanup = cleanupOverdrive_s;
		overdrive_sDescriptor->connect_port = connectPortOverdrive_s;
		overdrive_sDescriptor->deactivate = NULL;
		overdrive_sDescriptor->instantiate = instantiateOverdrive_s;
		overdrive_sDescriptor->run = runOverdrive_s;
		overdrive_sDescriptor->run_adding = runAddingOverdrive_s;
		overdrive_sDescriptor->set_run_adding_gain = setRunAddingGainOverdrive_s;
	}
}

void _fini() {
	int i;

	if (overdriveDescriptor) {
		free((char *)overdriveDescriptor->Label);
		free((char *)overdriveDescriptor->Name);
		free((char *)overdriveDescriptor->Maker);
		free((char *)overdriveDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)overdriveDescriptor->PortDescriptors);
		for (i = 0; i < overdriveDescriptor->PortCount; i++)
			free((char *)(overdriveDescriptor->PortNames[i]));
		free((char **)overdriveDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)overdriveDescriptor->PortRangeHints);
		free(overdriveDescriptor);
	}
	if (overdrive_sDescriptor) {
		free((char *)overdrive_sDescriptor->Label);
		free((char *)overdrive_sDescriptor->Name);
		free((char *)overdrive_sDescriptor->Maker);
		free((char *)overdrive_sDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)overdrive_sDescriptor->PortDescriptors);
		for (i = 0; i < overdrive_sDescriptor->PortCount; i++)
			free((char *)(overdrive_sDescriptor->PortNames[i]));
		free((char **)overdrive_sDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)overdrive_sDescriptor->PortRangeHints);
		free(overdrive_sDescriptor);
	}

}
