#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ladspa.h"

#define AMP_GAIN                       0
#define AMP_INPUT                      1
#define AMP_OUTPUT                     2

LADSPA_Descriptor *ampDescriptor = NULL;

typedef struct {
	LADSPA_Data *gain;
	LADSPA_Data *input;
	LADSPA_Data *output;
	LADSPA_Data run_adding_gain;
} Amp;

const LADSPA_Descriptor *ladspa_descriptor(unsigned long index) {
	switch (index) {
	case 0:
		return ampDescriptor;
	default:
		return NULL;
	}
}

void cleanupAmp(LADSPA_Handle instance) {
	free(instance);
}

void connectPortAmp(
 LADSPA_Handle instance,
 unsigned long port,
 LADSPA_Data *data) {
	Amp *plugin;

	plugin = (Amp *)instance;
	switch (port) {
	case AMP_GAIN:
		plugin->gain = data;
		break;
	case AMP_INPUT:
		plugin->input = data;
		break;
	case AMP_OUTPUT:
		plugin->output = data;
		break;
	}
}

LADSPA_Handle instantiateAmp(
 const LADSPA_Descriptor *descriptor,
 unsigned long s_rate) {
	Amp *plugin_data = (Amp *)malloc(sizeof(Amp));
	plugin_data->run_adding_gain = 1.0f;

	return (LADSPA_Handle)plugin_data;
}

#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b = v)
#define RUN_ADDING    0
#define RUN_REPLACING 1

void runAmp(LADSPA_Handle instance, unsigned long sample_count) {
	Amp *plugin_data = (Amp *)instance;

	/* Amps gain (in dB) (float value) */
	LADSPA_Data gain = *(plugin_data->gain);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;
	float coef = pow(10, gain/20);

	for (pos = 0; pos < sample_count; pos++) {
	  buffer_write(*(output++), *(input++) * coef);
	}
}
#undef buffer_write
#undef RUN_ADDING
#undef RUN_REPLACING

#define buffer_write(b, v) (b += (v) * run_adding_gain)
#define RUN_ADDING    1
#define RUN_REPLACING 0

void setRunAddingGainAmp(LADSPA_Handle instance, LADSPA_Data gain) {
	((Amp *)instance)->run_adding_gain = gain;
}

void runAddingAmp(LADSPA_Handle instance, unsigned long sample_count) {
	Amp *plugin_data = (Amp *)instance;
	LADSPA_Data run_adding_gain = plugin_data->run_adding_gain;

	/* Amps gain (in dB) (float value) */
	LADSPA_Data gain = *(plugin_data->gain);

	/* Input (array of floats of length sample_count) */
	LADSPA_Data *input = plugin_data->input;

	/* Output (array of floats of length sample_count) */
	LADSPA_Data *output = plugin_data->output;

	unsigned long pos;
	float coef = pow(10, gain/20);

	for (pos = 0; pos < sample_count; pos++) {
	  buffer_write(*(output++), *(input++) * coef);
	}
}

void _init() {
	char **port_names;
	LADSPA_PortDescriptor *port_descriptors;
	LADSPA_PortRangeHint *port_range_hints;

	ampDescriptor =
	 (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

	if (ampDescriptor) {
		ampDescriptor->UniqueID = 1181;
		ampDescriptor->Label = strdup("amp");
		ampDescriptor->Properties =
		 LADSPA_PROPERTY_HARD_RT_CAPABLE;
		ampDescriptor->Name =
		 strdup("Simple amplifier");
		ampDescriptor->Maker =
		 strdup("Steve Harris <steve@plugin.org.uk>");
		ampDescriptor->Copyright =
		 strdup("GPL");
		ampDescriptor->PortCount = 3;

		port_descriptors = (LADSPA_PortDescriptor *)calloc(3,
		 sizeof(LADSPA_PortDescriptor));
		ampDescriptor->PortDescriptors =
		 (const LADSPA_PortDescriptor *)port_descriptors;

		port_range_hints = (LADSPA_PortRangeHint *)calloc(3,
		 sizeof(LADSPA_PortRangeHint));
		ampDescriptor->PortRangeHints =
		 (const LADSPA_PortRangeHint *)port_range_hints;

		port_names = (char **)calloc(3, sizeof(char*));
		ampDescriptor->PortNames =
		 (const char **)port_names;

		/* Parameters for Amps gain (in dB) */
		port_descriptors[AMP_GAIN] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
		port_names[AMP_GAIN] =
		 strdup("Amps gain (in dB)");
		port_range_hints[AMP_GAIN].HintDescriptor =
		 LADSPA_HINT_BOUNDED_BELOW | LADSPA_HINT_BOUNDED_ABOVE;
		port_range_hints[AMP_GAIN].LowerBound = -70;
		port_range_hints[AMP_GAIN].UpperBound = +70;

		/* Parameters for Input */
		port_descriptors[AMP_INPUT] =
		 LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
		port_names[AMP_INPUT] =
		 strdup("Input");
		port_range_hints[AMP_INPUT].HintDescriptor = 0;

		/* Parameters for Output */
		port_descriptors[AMP_OUTPUT] =
		 LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
		port_names[AMP_OUTPUT] =
		 strdup("Output");
		port_range_hints[AMP_OUTPUT].HintDescriptor = 0;

		ampDescriptor->activate = NULL;
		ampDescriptor->cleanup = cleanupAmp;
		ampDescriptor->connect_port = connectPortAmp;
		ampDescriptor->deactivate = NULL;
		ampDescriptor->instantiate = instantiateAmp;
		ampDescriptor->run = runAmp;
		ampDescriptor->run_adding = runAddingAmp;
		ampDescriptor->set_run_adding_gain = setRunAddingGainAmp;
	}
}

void _fini() {
	int i;

	if (ampDescriptor) {
		free((char *)ampDescriptor->Label);
		free((char *)ampDescriptor->Name);
		free((char *)ampDescriptor->Maker);
		free((char *)ampDescriptor->Copyright);
		free((LADSPA_PortDescriptor *)ampDescriptor->PortDescriptors);
		for (i = 0; i < ampDescriptor->PortCount; i++)
			free((char *)(ampDescriptor->PortNames[i]));
		free((char **)ampDescriptor->PortNames);
		free((LADSPA_PortRangeHint *)ampDescriptor->PortRangeHints);
		free(ampDescriptor);
	}

}
