/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "utility.h"
#include "DialogFileChooser.h"
#include "Config.h"
#include "Encodings.h"
#include <libglademm/xml.h>
#include "ISubtitleEditor.h"
#include "SubtitleSystem.h"


class ColumnEncoding : public Gtk::TreeModel::ColumnRecord
{
public:
	ColumnEncoding()
	{
		add(use);
		add(name);
		add(charset);
	}
public:
	Gtk::TreeModelColumn<bool>					use;
	Gtk::TreeModelColumn<Glib::ustring>	name;
	Gtk::TreeModelColumn<Glib::ustring>	charset;
};


class EncodingsSelector : public Gtk::Table
{
public:
	EncodingsSelector(bool show_format=true)
	:Gtk::Table(2,2, false), 
		m_labelEncodings(_("Character Coding:"), 0.0,0.5), m_labelFormat(_("Format:"), 0.0,0.5)
	{
		set_col_spacings(3);

		attach(m_labelFormat, 0,1,0,1, (Gtk::SHRINK | Gtk::FILL));
		attach(m_comboFormat,1,2,0,1);
		
		attach(m_labelEncodings, 0,1,1,2,  (Gtk::SHRINK | Gtk::FILL));
		attach(m_comboEncodings,1,2,1,2);

		// s'il y a une modifications dans la configuration
		Signal::getInstance().config_encodings_changed.connect(
				sigc::mem_fun(*this, &EncodingsSelector::init_combo_encodgins));

		//
		std::list<Glib::ustring> formats = SubtitleSystem::getInstance().get_formats();

		for(std::list<Glib::ustring>::const_iterator it=formats.begin(); it != formats.end(); ++it)
			m_comboFormat.append_text(*it);

		m_comboFormat.set_active(0);
		//
		init_combo_encodgins();
		
		show_all();

		if(show_format == false)
		{
			m_labelFormat.hide();
			m_comboFormat.hide();
		}
	}

	void init_combo_encodgins()
	{
		m_comboEncodings.clear();
		// TODO : use clear_items for GTK+ 2.8
		//m_comboEncodings.clear_items();

		m_comboEncodings.append_text(_("Auto Detected"));

		Config &cfg = Config::getInstance();

		std::list<Glib::ustring> list_encodgins;
		if(cfg.get_value_string_list("encodings", "encodings", list_encodgins))
		{
			std::list<Glib::ustring>::const_iterator it;
			for(it = list_encodgins.begin(); it!=list_encodgins.end(); ++it)
			{
				EncodingInfo *info= Encodings::get_from_charset(*it);
				if(info)
				{
					gchar *name = g_strdup_printf("%s (%s)", info->name, info->charset);		
					m_comboEncodings.append_text(name);
					g_free(name);
				}
			}
		}

		//
		bool used_auto_detected = false;
		if(cfg.get_value_bool("encodings", "used-auto-detected", used_auto_detected))
		{
			if(used_auto_detected)
				m_comboEncodings.set_active(0);
			else
				m_comboEncodings.set_active(1);
		}
	}

	Glib::ustring getEncoding()
	{
		if(m_comboEncodings.get_active() == 0)
			return "";

		Glib::ustring text = m_comboEncodings.get_active_text();
		
		Glib::ustring::size_type a = text.find('(');
		Glib::ustring::size_type b = text.find(')', a);

		if(a == Glib::ustring::npos || b == Glib::ustring::npos)
			return "";
		
		return text.substr(a+1,b-a-1);
	}

	Glib::ustring getFormat()
	{
		return m_comboFormat.get_active_text();
	}
protected:
	Gtk::Label				m_labelEncodings;
	Gtk::ComboBoxText m_comboEncodings;

	Gtk::Label				m_labelFormat;
	Gtk::ComboBoxText	m_comboFormat;
};



 

/*
 *
 */
DialogFileChooser::DialogFileChooser(const Glib::ustring &title, Gtk::FileChooserAction action, bool use_format)
:Gtk::FileChooserDialog(title, action)
{
	bool show_format = true;

	if(action == Gtk::FILE_CHOOSER_ACTION_OPEN)
		show_format = false;

	if(use_format==false)
		show_format = false;

	add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
	set_default_response(Gtk::RESPONSE_OK);

	m_encodingsSelector = manage(new EncodingsSelector(show_format));
	set_extra_widget(*m_encodingsSelector);
}

/*
 *
 */
Glib::ustring DialogFileChooser::getEncoding()
{
	Glib::ustring text = m_encodingsSelector->getEncoding();
	// TODO : remove
	if(text == "Auto Detected")
		return "";
	
	return text;
}

	/*
	 * return format saving (SSA,ASS,SUBRIP,...)
	 */
Glib::ustring DialogFileChooser::getFormat()
{
	return m_encodingsSelector->getFormat();
}

/*
 * exemple :
 * name = "All Format Supported"
 * patterns = "*.ssa;*.ass;*.sub;*.txt"
 */
void DialogFileChooser::addFilter(const Glib::ustring &name, const Glib::ustring &patterns)
{
	Gtk::FileFilter filter;
	
	filter.set_name(name);
	
	std::istringstream iss(patterns);
	std::string word;
	while( std::getline(iss, word, ';') )
	{
		filter.add_pattern(word);
	}

	add_filter(filter);
}

