// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

 
/*
  menuops.cc
  This file contains code to set up the StarPlot menu bar and to deal with
  the toggle and radio button options in the StarPlot Options menu.
*/

#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <map>

#include "starplot.h"
#include "xpmdata.h"
#include "../version.h"


GtkItemFactoryEntry menu_items[] = {
  FILE_MENU, CHART_MENU, OPTIONS_TOGGLE_MENU, OPTIONS_LABEL_MENU,
  OPTIONS_DIAMETER_MENU, OPTIONS_COORD_MENU, STARS_MENU, HELP_MENU
};

GtkWidget *options_toggle_widgets[OPTIONS_TOGGLE_MENU_SIZE];
char options_toggle_names[OPTIONS_TOGGLE_MENU_SIZE][50];
GtkWidget *options_label_widgets[OPTIONS_LABEL_MENU_SIZE];
char options_label_names[OPTIONS_LABEL_MENU_SIZE][50];
GtkWidget *options_diameter_widgets[OPTIONS_DIAMETER_MENU_SIZE];
char options_diameter_names[OPTIONS_DIAMETER_MENU_SIZE][50];
GtkWidget *options_coord_widgets[OPTIONS_COORD_MENU_SIZE];
char options_coord_names[OPTIONS_COORD_MENU_SIZE][50];


/* set_menu_globals(): A boring function to set all the arrays of menu item
 *  widgets to be globally available. */

void set_menu_globals (GtkItemFactory *item_factory)
{
  unsigned int i;
  GtkWidget *item = 0;

  strcpy(options_toggle_names[TOGGLE_BARS], "/Options/Toggle Bars");
  strcpy(options_toggle_names[TOGGLE_GRID], "/Options/Toggle Grid");
  strcpy(options_toggle_names[TOGGLE_LEGEND], "/Options/Toggle Legend");

  strcpy(options_label_names[(int)LANDMARK_LABEL],
	 "/Options/Star Labels/Landmark Stars");
  strcpy(options_label_names[(int)STRING_LABEL],
	 "/Options/Star Labels/All Star Names");
  strcpy(options_label_names[(int)NUMBER_LABEL],
	 "/Options/Star Labels/Numerical Index");
  strcpy(options_label_names[(int)NO_LABEL],
	 "/Options/Star Labels/Labels Off");

  strcpy(options_diameter_names[(int)MAGNITUDE_DIAMETERS],
	 "/Options/Star Diameters/By Magnitude");
  strcpy(options_diameter_names[(int)MK_DIAMETERS],
	 "/Options/Star Diameters/By MK Class");

  strcpy(options_coord_names[CELESTIAL], 
	 "/Options/Coordinate System/Celestial");
  strcpy(options_coord_names[GALACTIC],
	 "/Options/Coordinate System/Galactic");

  /* The following code sets the check buttons in the Options menu to be
     toggled ON by default, matching the initial state of the program.

     According to the GTK documentation, I should not use the "active" field
     of GtkCheckMenuItem directly, so below I should replace each line
       (GTK_CHECK_MENU_ITEM (item))->active = TRUE;
     with the line
       gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (item), TRUE);
     However, when I do this, StarPlot segfaults.  Fixes welcomed.
  */
  for (i = 0; i < OPTIONS_TOGGLE_MENU_SIZE; i++) {
    item = gtk_item_factory_get_widget (item_factory, options_toggle_names[i]);
    (GTK_CHECK_MENU_ITEM (item))->active = TRUE;
    options_toggle_widgets[i] = item;
  }

  for (i = 0; i < OPTIONS_LABEL_MENU_SIZE; i++)
    options_label_widgets[i] =
      gtk_item_factory_get_widget (item_factory, options_label_names[i]);
  for (i = 0; i < OPTIONS_DIAMETER_MENU_SIZE; i++)
    options_diameter_widgets[i] =
      gtk_item_factory_get_widget (item_factory, options_diameter_names[i]);
  for (i = 0; i < OPTIONS_COORD_MENU_SIZE; i++)
    options_coord_widgets[i] =
      gtk_item_factory_get_widget (item_factory, options_coord_names[i]);

  return;
}


/* my_gtk_main_menu(): This function prepares the StarPlot menu bar. */

void my_gtk_main_menu (GtkWidget *window, GtkWidget **menubar)
{
  GtkItemFactory *item_factory;
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);
  
  accel_group = gtk_accel_group_new ();
       
  /* This function initializes the item factory.
     Param 1: The type of menu - can be GTK_TYPE_MENU_BAR, GTK_TYPE_MENU,
     or GTK_TYPE_OPTION_MENU.
     Param 2: The path of the menu.
     Param 3: A pointer to a gtk_accel_group.  The item factory sets up
     the accelerator table while generating menus.
  */

  item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", 
				       accel_group);

  /* This function generates the menu items. Pass the item factory,
     the number of items in the array, the array itself, and any
     callback data for the the menu items. */
  gtk_item_factory_create_items (item_factory, nmenu_items, menu_items, NULL);

  /* Attach the new accelerator group to the window. */
  gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);

  /* Set menu widgets to be globally available */
  set_menu_globals(item_factory);

  if (menubar)
    /* Finally, return the actual menu bar created by the item factory. */ 
    *menubar = gtk_item_factory_get_widget (item_factory, "<main>");
}


/* my_gtk_button_bar(): This function creates a button bar with any
   combination of OK, Defaults and Cancel buttons, and packs it into
   a vbox or hbox given as an argument.  It will NOT, however, set
   up the callback functions.  Use NULL for any of the first
   three arguments for which a button should not be created.
*/
void my_gtk_button_bar(GtkWidget **OK, GtkWidget **defaults,
		       GtkWidget **cancel, GtkWidget *insertionbox)
{
  GtkWidget *buttonbox = gtk_hbox_new (TRUE, 0);
  gtk_widget_show (buttonbox);
  gtk_box_pack_start (GTK_BOX (insertionbox), buttonbox, FALSE, FALSE, 0);
 
  if (OK) {
    *OK = gtk_button_new_with_label ("OK");
    gtk_widget_set_usize (*OK, BUTTONWIDTH, BUTTONHEIGHT);
    gtk_box_pack_start (GTK_BOX (buttonbox), *OK, TRUE, FALSE, 0);
    gtk_widget_show (*OK);
  }

  if (defaults) {
    *defaults = gtk_button_new_with_label ("Defaults");
    gtk_widget_set_usize (*defaults, BUTTONWIDTH, BUTTONHEIGHT);
    gtk_box_pack_start (GTK_BOX (buttonbox), *defaults, TRUE, FALSE, 0);
    gtk_widget_show (*defaults);
  }

  if (cancel) {
    *cancel = gtk_button_new_with_label ("Cancel");
    gtk_widget_set_usize (*cancel, BUTTONWIDTH, BUTTONHEIGHT);
    gtk_box_pack_start (GTK_BOX (buttonbox), *cancel, TRUE, FALSE, 0);
    gtk_widget_show (*cancel);
  }

  return;
}

/* my_gtk_buttons_connect_destroy(): Connects all the buttons supplied
   as arguments to the gtk_destroy_widget() function.  Should be called
   after any other signal connections.
*/
void my_gtk_buttons_connect_destroy(GtkWidget *OK, GtkWidget *defaults,
				    GtkWidget *cancel, GtkWidget *window)
{
  if (OK)
    gtk_signal_connect_object (GTK_OBJECT (OK), "clicked", 
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (window));
  if (defaults)
    gtk_signal_connect_object (GTK_OBJECT (defaults), "clicked", 
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (window));
  if (cancel)
    gtk_signal_connect_object (GTK_OBJECT (cancel), "clicked", 
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (window));
  return;
}




/* set_item(): This function is a general function which can be used to
 *  set the value of any item, and if desired, change the state of the
 *  applicable GtkMenuItem in a radio-button/check-button menu.
 *  "itemptr" is a pointer to the value to change; "value" is the new value
 *  desired; "menuptr" is the GtkWidget associated with the value; "redraw"
 *  is true if updating the StarPlot main display is desired, and "c"
 *  is the type of display update to perform. */

template<class T> void
set_item(T *itemptr, T value, GtkWidget *menuptr, bool redraw, Changetype c)
{
  // this keeps track of whether or not a particular item should be changed
  static map<T *, bool> enabled;

  // will only insert a "true" value for given item pointer if that item
  //  has not yet been used:
  pair<T *, bool> currenttype = pair<T *, bool>(itemptr, true);
  enabled.insert(currenttype);

  // test for whether itemptr is enabled, to ensure that (a) this function
  //  doesn't cause itself to be called again by playing with radio button
  //  toggle states; (b) this is not being called as a result of a _different_
  //  radio button which controls the same value being toggled ON, thus
  //  toggling this one OFF.

  if (enabled[itemptr]) {
    enabled[itemptr] = false;
    {
      *itemptr = value;
      if (menuptr) {
	if (GTK_OBJECT_TYPE(menuptr) == gtk_type_from_name("GtkRadioMenuItem"))
	  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuptr), TRUE);
	else if (gtk_type_is_a (GTK_OBJECT_TYPE(menuptr), 
				gtk_type_from_name("GtkCheckMenuItem")))
	  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuptr),
					  value ? TRUE : FALSE);
      }
      if (redraw)
	redraw_all (c);
    }
    enabled[itemptr] = true;
  }
}


/* The following trivial functions each carry out one of the options in
   the Options menu.
*/

void flip_toggle_item(bool *itemptr, int number, bool redraw)
{ set_item(itemptr, ! *itemptr, options_toggle_widgets[number], redraw); }

void set_toggle_item(bool *itemptr, bool value, int number, bool redraw)
{
  if (*itemptr == value) return;
  else flip_toggle_item(itemptr, number, redraw);
}

void toggle_bars()  
{ flip_toggle_item(&starrules.StarBars, TOGGLE_BARS, true); }
void toggle_grid() 
{ flip_toggle_item(&starrules.ChartGrid, TOGGLE_GRID, true); }
void toggle_legend() 
{ flip_toggle_item(&starrules.ChartLegend, TOGGLE_LEGEND, true); }

void set_label_item(Labeltype l, bool redraw)
{
  if (starrules.StarLabels == l) return;

  if (hr_viewer) {
    set_item(&starrules.StarLabels, l,
	     hr_options_label_widgets[l], false);
  }
  set_item(&starrules.StarLabels, l,
	   options_label_widgets[l], redraw);
}

void labels_all()       { set_label_item(STRING_LABEL, true); }
void labels_landmk()    { set_label_item(LANDMARK_LABEL, true); }
void labels_numerical() { set_label_item(NUMBER_LABEL, true); }
void labels_off()       { set_label_item(NO_LABEL, true); }

void set_diameter_item(Diametertype d, bool redraw)
{
  if (starrules.StarDiameters == d) return;

  if (hr_viewer) {
    set_item(&starrules.StarDiameters, d,
	     hr_options_diameter_widgets[d], false);
  }
  set_item(&starrules.StarDiameters, d,
	   options_diameter_widgets[d], redraw);
}

void mk_diameters()  { set_diameter_item(MK_DIAMETERS, true); }
void mag_diameters() { set_diameter_item(MAGNITUDE_DIAMETERS, true); }

void set_coord_item(bool coords, bool redraw)
{
  static bool enabled = true;
  if (starrules.CelestialCoords == coords) return;

  if (enabled) {      // wrapper to keep the checkmenuitem toggle from
    enabled = false;  // causing endless recursive calls of this function
    {
      starrules.CelestialCoords = coords;
      if (redraw) {
	if (coords == (bool)CELESTIAL)
	  starrules.ChartLocation = starrules.ChartLocation.toCelestial();
	else
	  starrules.ChartLocation = starrules.ChartLocation.toGalactic();
	redraw_all(COORDINATE_CHANGE);
      }
      gtk_check_menu_item_set_active 
	(GTK_CHECK_MENU_ITEM (options_coord_widgets[coords]), TRUE);
    }
    enabled = true;
  }
}

void cel_coords() { set_coord_item(CELESTIAL, true); }
void gal_coords() { set_coord_item(GALACTIC, true); }


/* These callbacks are for the button bar immediately underneath the menu. */

void button_zoom(GtkWidget *w, gpointer zoomfactor)
{
  double zf = *(double *)zoomfactor;

  // zoomfactor is > 1.0 to zoom in, < 1.0 to zoom out
  starrules.ChartRadius /= zf;
  if (starrules.ChartAutosetDimmestMagnitude && zf != 1.0)
    starrules.ChartDimmestMagnitude = automagnitude(starrules.ChartRadius);
  redraw_all(RADIUS_CHANGE);
  return;
}

void button_rotate(GtkWidget *w, gpointer angle)
{
  double a = *(double *)angle;
  starrules.ChartOrientation
    = SolidAngle(starrules.ChartOrientation.getPhi() + a,
		 starrules.ChartOrientation.getTheta());
  redraw_all(ORIENTATION_CHANGE);
  return;
}

void button_tilt(GtkWidget *w, gpointer angle)
{
  double a = *(double *)angle;
  starrules.ChartOrientation
    = SolidAngle(starrules.ChartOrientation.getPhi(),
		 starrules.ChartOrientation.getTheta() + a);
  redraw_all(ORIENTATION_CHANGE);
  return;
}

void button_magchange(GtkWidget *w, gpointer magchange)
{
  double m = *(double *)magchange;
  starrules.ChartDimmestMagnitude += m;
  redraw_all(FILTER_CHANGE);
  return;
}


/* Define the button bar: */

void my_gtk_push_buttons (GtkWidget *window, GtkWidget **buttonbar)
{
  GtkWidget *vbox, *buttons[9], *pixmapw;
  GtkTooltips *tooltips;
  GtkStyle *style = gtk_widget_get_style (window);
  GtkStyle *blackbackground = gtk_style_copy (style);
  GdkPixmap *pixmap, *mask;

  static double zoominfactor = 2.0, zoomoutfactor = 1.0 / zoominfactor;
  static double leftangle = 0.5 HOURS, rightangle = -leftangle;
  static double northangle = 5.0 DEGREES, southangle = -northangle;
  static double magup = -0.5, magdown = -magup;

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox);

  *buttonbar = gtk_event_box_new();
  gtk_container_add (GTK_CONTAINER (*buttonbar), vbox);
  gtk_widget_show (*buttonbar);

  blackbackground->bg[GTK_STATE_NORMAL] = blackbackground->black;
  gtk_widget_set_style (*buttonbar, blackbackground);

  for (unsigned int i = 0; i < 9; i++) {
    // this creates the button icons; they are defined in the static
    //  array `xpmdata' in the xpmdata.h header file.
    pixmap = gdk_pixmap_colormap_create_from_xpm_d 
      (0, gtk_widget_get_colormap(window), &mask,
       &style->bg[GTK_STATE_NORMAL], (gchar **)xpmdata[i]);
    pixmapw = gtk_pixmap_new (pixmap, mask);
    gdk_pixmap_unref(pixmap);
    gdk_pixmap_unref(mask);

    // create the buttons themselves
    gtk_widget_show (pixmapw);
    buttons[i] = gtk_button_new();
    gtk_container_add (GTK_CONTAINER(buttons[i]), pixmapw);
    gtk_widget_show (buttons[i]);
    gtk_box_pack_start (GTK_BOX (vbox), buttons[i], FALSE, TRUE, 0);
  }

  // connect the callback functions to the buttons
  gtk_signal_connect (GTK_OBJECT (buttons[0]), "clicked",
		      GTK_SIGNAL_FUNC (file_open), 0);
  gtk_signal_connect (GTK_OBJECT (buttons[1]), "clicked",
		      GTK_SIGNAL_FUNC (button_zoom), &zoominfactor);
  gtk_signal_connect (GTK_OBJECT (buttons[2]), "clicked",
		      GTK_SIGNAL_FUNC (button_zoom), &zoomoutfactor);
  gtk_signal_connect (GTK_OBJECT (buttons[3]), "clicked",
		      GTK_SIGNAL_FUNC (button_rotate), &leftangle);
  gtk_signal_connect (GTK_OBJECT (buttons[4]), "clicked",
		      GTK_SIGNAL_FUNC (button_rotate), &rightangle);
  gtk_signal_connect (GTK_OBJECT (buttons[5]), "clicked",
		      GTK_SIGNAL_FUNC (button_tilt), &northangle);
  gtk_signal_connect (GTK_OBJECT (buttons[6]), "clicked",
		      GTK_SIGNAL_FUNC (button_tilt), &southangle);
  gtk_signal_connect (GTK_OBJECT (buttons[7]), "clicked",
		      GTK_SIGNAL_FUNC (button_magchange), &magup);
  gtk_signal_connect (GTK_OBJECT (buttons[8]), "clicked",
		      GTK_SIGNAL_FUNC (button_magchange), &magdown);

  // and set some popup help tips
  tooltips = gtk_tooltips_new ();
  gtk_tooltips_set_tip (tooltips, buttons[0], "Open star database...", 0);
  gtk_tooltips_set_tip (tooltips, buttons[1], "Zoom in", 0);
  gtk_tooltips_set_tip (tooltips, buttons[2], "Zoom out", 0);
  gtk_tooltips_set_tip (tooltips, buttons[3],
			"Rotate chart clockwise about its axis", 0);
  gtk_tooltips_set_tip (tooltips, buttons[4], 
			"Rotate chart counterclockwise about its axis", 0);
  gtk_tooltips_set_tip (tooltips, buttons[5], 
			"Tilt chart north pole towards you", 0);
  gtk_tooltips_set_tip (tooltips, buttons[6], 
			"Tilt chart south pole towards you", 0);
  gtk_tooltips_set_tip (tooltips, buttons[7], 
			"Decrease magnitude limit (Show fewer stars)", 0);
  gtk_tooltips_set_tip (tooltips, buttons[8], 
			"Increase magnitude limit (Show more stars)", 0);
  return;
}


/* Silly little "about" dialog */
void abouthelp()
{
  GtkWidget *popup, *OK, *topbox, *label[11];
  
  popup = gtk_dialog_new ();
  gtk_window_set_policy (GTK_WINDOW (popup), FALSE, FALSE, TRUE);
  gtk_window_set_title (GTK_WINDOW (popup), "About StarPlot");
  
  topbox = gtk_vbox_new (TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (topbox), 10);
  gtk_widget_show (topbox);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (popup)->vbox), topbox,
		      TRUE, TRUE, 0);
  
  label[0] = gtk_label_new ("StarPlot version " STARPLOT_VERSION);
  label[1] = gtk_label_new ("");
  label[2] = gtk_label_new ("Copyright " COPYRIGHT_SYMBOL " 2000, 2001 "
  			    "Kevin B. McCarty");
  label[3] = gtk_label_new ("under the GNU General Public License");
  label[4] = gtk_label_new ("");
  label[5] = gtk_label_new ("For help, please select Help->Documentation");
  label[6] = gtk_label_new ("from the menu, or see the HTML documents in");
  label[7] = gtk_label_new (DOCDIR "/html");
  label[8] = gtk_label_new ("");
  label[9] = gtk_label_new ("For additional data files, see the StarPlot web"
  			    " page:");
  label[10] = gtk_label_new ("http://www.princeton.edu/~kmccarty/"
  			     "starplot.html");

  for (unsigned int i = 5; i < 11; i++)
    gtk_misc_set_alignment (GTK_MISC (label[i]), (float)0.0, (float)0.0);

  for (unsigned int i = 0; i < 11; i++) {
    gtk_box_pack_start (GTK_BOX (topbox), label[i], TRUE, TRUE, 0);
    gtk_widget_show (label[i]);
  }

  my_gtk_button_bar (&OK, NULL, NULL, GTK_DIALOG (popup)->action_area);
  my_gtk_buttons_connect_destroy (OK, NULL, NULL, popup);
  
  gtk_window_set_focus (GTK_WINDOW (popup), OK);
  gtk_widget_show (popup);
}


/* Note: help_select() function is defined in filedialogs.cc since it goes
   with the other file selection dialogs. */


/* helpdocs(): Read help documentation */
void helpdocs()
{
  pid_t pid;
  if (!help_browser) {
    help_select_and_open();
    return;
  }

  char * const command[3] = 
    { help_browser, "file://" DOCDIR "/html/index.html", NULL };

  pid = fork();

  if (pid < 0) /* problem */ return;
  else if (pid == 0) {
    /* avoid zombie processes by fork()ing twice */
    /* This code comes from _Advanced Programming in the UNIX Environment_,
     *  W. Richard Stevens, 1993 (program #8.5) */
    pid = fork();
    if (pid < 0) /* problem */ _exit(EXIT_FAILURE);
    else if (pid == 0) { /* second child: execvp() the help browser */
      if (execvp(command[0], command) < 0) _exit(EXIT_FAILURE);
    }
    else /* first child */ _exit(EXIT_SUCCESS);
  }

  /* parent waits for first child */
  if (waitpid(pid, NULL, 0) != pid) return;
  return;
}
