/*--------------------------------------------------------------------*//*:Ignore this sentence.
Copyright (C) 1999, 2001, 2005 SIL International. All rights reserved.

Distributable under the terms of either the Common Public License or the
GNU Lesser General Public License, as specified in the LICENSING.txt file.

File: XftFont.cpp
Responsibility: Keith Stribley
Last reviewed: Not yet.

Description:
    A Font is an object that represents a font-family + bold + italic setting, that contains
  Graphite tables.
----------------------------------------------------------------------------------------------*/
#include "Main.h"

#include "FontTableCache.h"
#include "XftGrFont.h"

#include  FT_TRUETYPE_TABLES_H

// TBD do this properly
#define FUDGE_FACTOR 72
#define fix26_6(x) (x >> 6) + (x & 32 ? (x > 0 ? 1 : 0) : (x < 0 ? -1 : 0))
#define float26_6(x) (x / 32.)

namespace gr
{


XftGrFont::XftGrFont(Display * disp, Screen * scrn)
  : Font(),
  m_hfont(NULL),
  m_ftFace(NULL),
  m_pTableCache(NULL),
  m_ascent(0),
  m_descent(0),
  m_emSquare(0),
  m_dpiX(FUDGE_FACTOR),// set properly later unless an error occurs
  m_dpiY(FUDGE_FACTOR),// set properly later unless an error occurs
  m_locked(false)
{
  m_fItalic = false;
  m_fBold = false;
  // colors are not used
  m_clrFore = kclrBlack;
  m_clrBack = kclrTransparent;
  //m_fpropDef.szFaceName[0] = '\0';
  m_pixHeight = 0;
  initializeDPI(disp, scrn);
}

XftGrFont::XftGrFont(XftFont * xftFont, Display * disp, Screen * scrn)
  : Font(),
  m_hfont(xftFont),
  m_ftFace(NULL),
  m_pTableCache(NULL),
  m_ascent(0),
  m_descent(0),
  m_emSquare(0),
  m_dpiX(FUDGE_FACTOR),// set properly later unless an error occurs
  m_dpiY(FUDGE_FACTOR),// set properly later unless an error occurs
  m_locked(false)
{
  Assert(m_hfont); // shouldn't be null but we play safe anyway
  initializeDPI(disp, scrn);
  initializeFromFace();
}

XftGrFont::XftGrFont(FT_Face face, Display * disp, Screen * scrn)
  : Font(),
  m_hfont(NULL),
  m_ftFace(face),
  m_pTableCache(NULL),
  m_ascent(0),
  m_descent(0),
  m_emSquare(0),
  m_dpiX(FUDGE_FACTOR),// set properly later unless an error occurs
  m_dpiY(FUDGE_FACTOR),// set properly later unless an error occurs
  m_locked(false)
{
  Assert(m_ftFace); // shouldn't be null but we play safe anyway
  initializeDPI(disp, scrn);
  initializeFromFace();
}

void
XftGrFont::initializeDPI(Display * disp, Screen * scrn)
{
  Display * lDisp = disp;
  if (disp == NULL)
  {
    disp = XOpenDisplay(NULL);  // get current display
  }
  // this may fail if the max number of X clients is exceeded
  if (disp != NULL) 
  {
    if (scrn == NULL)
    {
      scrn = DefaultScreenOfDisplay(disp);
    }
    if (scrn != NULL)
    {
      m_dpiX = int(WidthOfScreen(scrn) * 25.4 / WidthMMOfScreen(scrn));
      m_dpiY = int(HeightOfScreen(scrn) * 25.4 / HeightMMOfScreen(scrn));
    }
    if (lDisp == NULL)        // we opened a display so close it
    {
      XCloseDisplay(disp);
    }
  }
}

void
XftGrFont::initializeFromFace()
{
  m_fItalic = false;
  m_fBold = false;
  // colors are not used
  m_clrFore = kclrBlack;
  m_clrBack = kclrTransparent;
  FT_Face face = lockFace();
  Assert(face);
  if (face)
  {
    if (face->style_flags & FT_STYLE_FLAG_BOLD) m_fBold = true;
    if (face->style_flags & FT_STYLE_FLAG_ITALIC) m_fItalic = true;
    size_t i = 0;
    // may be able to remove the check on MAX_FACE_LENGTH
    while ((face->family_name[i] != 0) && i < kMaxFaceLength)
    {
      // assumes ascii only name
      m_faceName.push_back(face->family_name[i]);
      i++;
      Assert(i < kMaxFaceLength);
    }
    // append a null if there is room
    //if (i < kMaxFaceLength) m_fpropDef.szFaceName[i] = 0;
    Assert(face->size);
    m_pixHeight = float26_6(face->size->metrics.height);
    m_emSquare = face->size->metrics.y_ppem;
    m_ascent = float26_6(face->size->metrics.ascender);
    m_descent = float26_6(face->size->metrics.descender);
    if (m_descent < 0) m_descent = - m_descent;

    Assert(m_pixHeight > 0);

    unlockFace(); // must unlock face before GetFontFace since this may try to
    // lock it again by calling methods back on us

    m_pfface = FontFace::GetFontFace(this, m_faceName,
                                     m_fBold, m_fItalic);
    if (m_pfface) m_pfface->IncFontCount();
  }
  else
  {
    m_pixHeight = 0;
  }
}


XftGrFont::~XftGrFont()
{
  Assert(!m_locked);
  // if this is the last copy of the font sharing these tables
  // delete them
  if (m_pTableCache)
  {
    m_pTableCache->decrementFontCount();
    if (m_pTableCache->getFontCount() == 0)
    {
      delete m_pTableCache;
      m_pTableCache = NULL;
    }
  }
  m_glyphMetricHash.clear();
  // note the DecFontCount(); is done in the Font base class
}
  
Font * XftGrFont::copyThis()
{
  XftGrFont * copy = new XftGrFont(*this);
  return copy;
}

XftGrFont::XftGrFont(XftGrFont & font)
: Font(),
  m_hfont(font.m_hfont),
  m_ftFace(font.m_ftFace),
  m_ascent(font.m_ascent),
  m_descent(font.m_descent),
  m_emSquare(font.m_emSquare),
  m_dpiX(font.m_dpiX),
  m_dpiY(font.m_dpiY),
  m_locked(false)
{

  m_fItalic = font.m_fItalic;
  m_fBold = font.m_fBold;
  // colors are not used
  m_clrFore = font.m_clrFore;
  m_clrBack = font.m_clrBack;
  m_faceName.assign(font.m_faceName);
  m_pixHeight = font.m_pixHeight;  

  m_pTableCache = font.m_pTableCache;
  // use the same table cache between instances
  if (m_pTableCache)
    m_pTableCache->incrementFontCount();

  // I dont' see why we need to reget the face, but WinFont does
  //m_pfface = FontFace::GetFontFace(this, m_fpropDef.szFaceName,
  //                                 m_fpropDef.fBold, m_fpropDef.fItalic);
  m_pfface = font.m_pfface;                                   
  if (m_pfface) m_pfface->IncFontCount();

}


FT_Face XftGrFont::lockFace(void)
{
  Assert(!m_locked);
  // if Xft is being used, get the face, otherwise, return our local handle
  FT_Face face = m_ftFace;
  if (m_hfont)
  {
    face = XftLockFace(m_hfont);
  }
  m_locked = true;
  return face;
}

void XftGrFont::unlockFace(void)
{
  Assert(m_locked);
  if (m_hfont)
  {
    XftUnlockFace(m_hfont);
  }
  m_locked = false;
}

  //virtual FontErrorCode isValidForGraphite(int * pnVersion = NULL, int * pnSubVersion = NULL);

/*----------------------------------------------------------------------------------------------
  Read a table from the font.
----------------------------------------------------------------------------------------------*/  
const void * 
XftGrFont::getTable(fontTableId32 tableID, size_t * pcbSize)
{
  *pcbSize = 0;
  // use a cache to reduce the number of times tables have to be reloaded
  if (m_pTableCache == NULL)
  {
    // constructor automatically sets the font count to 1
    m_pTableCache = new FontTableCache();
  }
  TableId tid;
  for (int i = 0; i<ktiLast; i++)
  {
    tid = static_cast<TableId>(i);
    if (tableID == TtfUtil::TableIdTag(tid))
    {
      if (m_pTableCache->getTable(tid))
      {
        *pcbSize = m_pTableCache->getTableSize(tid);
        return m_pTableCache->getTable(tid);
      }
      break;
    }
  }
  Assert(tid < ktiLast);
  byte * prgbBuffer = NULL;
  FT_Face face = lockFace();
  fontTableId32 ftTableId = tableID;
  Assert(sizeof(ftTableId) == 4);// otherwise these masks go wrong!
  if(ftTableId) // MS tags differ in endidness from FT ones 
     ftTableId = ftTableId >> 24 | ftTableId << 24 |
     (ftTableId >> 8 & 0xff00) | (ftTableId << 8 & 0xff0000);
  FT_ULong cbTableSz = 0;
  // since length in is 0 this only retrieves the size without loading
  FT_Load_Sfnt_Table(face, ftTableId, 0, prgbBuffer, &cbTableSz);
  if (cbTableSz > 0)
  {
    prgbBuffer = m_pTableCache->allocateTable(tid, cbTableSz);
    if (prgbBuffer)
      FT_Load_Sfnt_Table(face, ftTableId, 0, prgbBuffer, &cbTableSz);
  }
  unlockFace();
  *pcbSize = cbTableSz; // allows error checking

  return prgbBuffer;
}



void XftGrFont::getFontMetrics(float * pAscent, float * pDescent,
    float * pEmSquare)
{
    if (m_ascent == 0) // is this ever 0 in a real font?
    {
      FT_Face face = lockFace();
      Assert(face);
      Assert(face->size);
      m_emSquare = face->size->metrics.x_ppem;
      m_ascent = float26_6(face->size->metrics.ascender);
      m_descent = float26_6(face->size->metrics.descender);
      // make the descent positive to be compatible with the rest of graphite
      if (m_descent < 0) m_descent = - m_descent;
      unlockFace();
    }
    if (pEmSquare) *pEmSquare = m_emSquare;
    if (pAscent) *pAscent = m_ascent;
    if (pDescent) *pDescent = m_descent;
}


void XftGrFont::getGlyphPoint(gid16 gid, unsigned int pointNum, gr::Point & xyReturn)
{
  // this isn't used very often, so don't bother caching

   FT_Face face = lockFace();
   Assert(face);
   FT_Outline *outline;

   FT_Load_Glyph(face, gid, 0);
   outline = &face->glyph->outline;
   xyReturn.x = fix26_6(outline->points[pointNum].x);
   xyReturn.y = fix26_6(outline->points[pointNum].y);
   //printf("GrGraphics::XYFromGlyphPoint %d %d\n",*pxRet,*pyRet);
   unlockFace();
}

void XftGrFont::getGlyphMetrics(gid16 glyphID, gr::Rect & boundingBox, gr::Point & advances)
{
  FT_Glyph_Metrics gm;
  GlyphMetricHash::iterator i = m_glyphMetricHash.end();
  // used the cached metrics if available, otherwise load the glyph
  if (m_glyphMetricHash.size() > 0)
    m_glyphMetricHash.find(glyphID);
  if (i != m_glyphMetricHash.end())
  {
    gm = i->second;
    //printf("Reusing Glyph %d\t",chw);
    advances.x = fix26_6(gm.horiAdvance);
    advances.y = 0;
    boundingBox.bottom = fix26_6(gm.horiBearingY);;
    boundingBox.top = boundingBox.bottom + fix26_6(gm.height);
    boundingBox.left = fix26_6(gm.horiBearingX);
    boundingBox.right = fix26_6(gm.width) + boundingBox.left;
  }
  else
  {
    //printf("Loading Glyph %d\t",chw);
    FT_Face face = lockFace();
    Assert(face);
    FT_Load_Glyph(face, glyphID, 0);
    gm = face->glyph->metrics;
    GlyphMetricHash::value_type pair(glyphID, gm);
    m_glyphMetricHash.insert(pair);
    advances.x = fix26_6(gm.horiAdvance);
    advances.y = 0;
    boundingBox.bottom = fix26_6(gm.horiBearingY);;
    boundingBox.top = boundingBox.bottom + fix26_6(gm.height);
    boundingBox.left = fix26_6(gm.horiBearingX);
    boundingBox.right = fix26_6(gm.width) + boundingBox.left;

    unlockFace();
  }
}


bool XftGrFont::FontHasGraphiteTables(XftFont * xftFont)
{
  bool isGraphiteFont = false;
  if (xftFont)
  {
    FT_Face face = XftLockFace(xftFont);
    isGraphiteFont = FontHasGraphiteTables(face);
    XftUnlockFace(xftFont);
  }
  return isGraphiteFont;
}

bool XftGrFont::FontHasGraphiteTables(FT_Face face)
{
  bool isGraphiteFont = false;
  if (face)
  {
    FT_Byte tmp[20];
    FT_ULong len = 0;
    // need Freetype 2.1.4 and up
    FT_Error error = FT_Load_Sfnt_Table(face, FT_MAKE_TAG('S','i','l','f'), 0, &tmp[0], &len);
    if(( 0 == error) && (len > 0))
    {
       isGraphiteFont =  true; 
    }
  }
  return isGraphiteFont;
}

} // namespace gr

