
/*
#    Sfront, a SAOL to C translator    
#    This file: Time synchronization routines for file playback.
#
# Copyright (c) 1999-2006, Regents of the University of California
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#  Redistributions of source code must retain the above copyright
#  notice, this list of conditions and the following disclaimer.
#
#  Redistributions in binary form must reproduce the above copyright
#  notice, this list of conditions and the following disclaimer in the
#  documentation and/or other materials provided with the distribution.
#
#  Neither the name of the University of California, Berkeley nor the
#  names of its contributors may be used to endorse or promote products
#  derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#    Maintainer: John Lazzaro, lazzaro@cs.berkeley.edu
*/


/******************************************************************/
/*                                                                */
/* Playback synchronization routines act to estimate the cpuload  */
/* for each k-cycle. Two functions are must be defined:           */
/*                                                                */
/* void ksyncinit() { }                                           */
/*                                                                */
/*   called before first ipass begins, lets sync routines         */
/*   initialize state.                                            */
/*                                                                */
/* float ksync()  { }                                             */
/*                                                                */
/*   returns the current estimate of cpuload, the SA standard     */
/*   name.                                                        */
/*                                                                */
/* Rules for writing your own playback sync.c file:               */
/*                                                                */
/* (1) all globals and functions must begin with sync_, all defs  */
/*     must begin with SYNC_.                                     */
/* (2) Define both ksync() and ksyncinit().                       */
/*                                                                */
/* Note that:                                                     */
/*  KUTIME: a long,  defines k-cycle length in microseconds       */
/*  KMTIME: a float, defines k-cycle length in milliseconds       */
/*  KTIME:  a float, defines k-cycle length in seconds            */
/******************************************************************/


#include <sys/time.h>

#define SYNC_ONESECOND 1000000L
#define SYNC_SELDELAY 1000L
#define SYNC_NORM (1.0F/KUTIME)
struct timeval sync_last, sync_this;

/***********************************************************/
/*         synchronizes on k-rate boundaries               */
/***********************************************************/

float ksync()

{
  float ret = 1.0F;

  gettimeofday(&sync_this, NULL);

  if ((sync_last.tv_usec += KUTIME) < SYNC_ONESECOND) /* non-rollover case */
    {
      if ((sync_last.tv_usec >= sync_this.tv_usec) &&
	  (sync_last.tv_sec == sync_this.tv_sec))
	ret -= SYNC_NORM*(sync_last.tv_usec - sync_this.tv_usec);

      sync_last = sync_this;
      ret = (ret > 0.0F) ? ret : 0.0F;
      return ret;
    }

  sync_last.tv_sec++;
  sync_last.tv_usec %= SYNC_ONESECOND;

  if (sync_last.tv_sec >= sync_this.tv_sec)
    {
      if (sync_last.tv_sec > sync_this.tv_sec)
	{
	  ret -= SYNC_NORM*(sync_last.tv_usec + 
			    SYNC_ONESECOND - sync_this.tv_usec);
	}
      else
	{
	  if (sync_last.tv_usec > sync_this.tv_usec)
	    ret -= SYNC_NORM*(sync_last.tv_usec - sync_this.tv_usec);
	}
    }

  sync_last = sync_this;
  ret = (ret > 0.0F) ? ret : 0.0F;
  return ret;
}

/***********************************************************/
/*         initializes k-rate boundaries sync              */
/***********************************************************/

void ksyncinit()

{
  gettimeofday(&sync_last, NULL);
}


