# Samizdat front page
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/helpers/syndication_helper'

class FrontpageController < Controller
  include SyndicationHelper

  def index
    @title = config['site']['name']

    page = (@request['page'] or 1).to_i
    features_page = (@request['features_page'] or 1).to_i

    full_front_page = (1 == page and 1 == features_page)
    render_features = (full_front_page or 1 == page)
    render_updates = (full_front_page or 1 == features_page)

    # provide rss links
    if full_front_page
      rss_features = 'frontpage/rss'
      rss_updates = 'frontpage/rss?feed=updates'
      @feeds = { _('Features') => rss_features, _('Recent Updates') => rss_updates }
    end

    # try to run from cache
    cache_key = 'index' + @request.uri_prefix + '/' + @request.accept_language.join(':') +
      %{/#{page}/#{features_page}/#{'yes' != @request.cookie('nostatic')}}
    if body = cache[cache_key]
      return @content_for_layout = body
    end

    if full_front_page   # render focuses
      focuses_title = _('Top Focuses')
      focuses =
        Focus.collect_focuses {|focus, usage|
          %{<div class="focus">} + resource_href(
            focus,
            Resource.new(@request, focus).title + %{ (#{usage})}
          ) + %{</div>\n}
        }.join

      focuses <<
        '<div class="foot"><div class="nav"><a href="foci">' <<
        _('more') << '</a></div></div>'
    end

    if render_features
      dataset = features_dataset(config['limit']['features'])
      features = dataset[features_page - 1].collect {|msg,|
        Resource.new(@request, msg).short }

      features_title = _('Features') + page_number(features_page)
      features = features.join <<
        %{<div class="foot">} <<
        nav_rss(rss_features) <<
        nav(dataset, 'features_page') <<
        "</div>\n"
    end

    if render_updates
      dataset = updates_dataset
      updates = dataset[page - 1].collect {|msg,|
        Resource.new(@request, msg).list_item }

      updates_title = _('Recent Updates') + page_number(page)
      updates = list(updates,
        nav_rss(rss_updates) << nav(dataset))
    end

    body =
      if full_front_page
%{<table>
  <thead>
    <tr><th>#{focuses_title}</th><th>#{features_title}</th><th>#{updates_title}</th></tr>
  </thead>
  <tr>
    <td class="focuses">#{focuses}</td>
    <td class="features" rowspan="3">#{features}</td>
    <td class="updates" rowspan="3">#{updates}</td>
  </tr>
  <tr><td class="links-head">} << _('Links') << '</td></tr>
  <tr><td class="links">
    <p><a href="query/run?q=' << CGI.escape('SELECT ?resource WHERE (dc::date ?resource ?date) (s::inReplyTo ?resource ?parent) LITERAL ?parent IS NOT NULL ORDER BY ?date DESC') << '">' << _('All Replies') << '</a></p>
    <p><a href="moderation">' << _('Moderation Log') << '</a></p>' <<
    render_feeds << '
  </td></tr>
</table>'
      elsif render_features
        box(features_title, features)
      elsif render_updates
        box(updates_title, updates)
      end

    # static header and footer
    if full_front_page and 'yes' != @request.cookie('nostatic')
      header = static_text('header')
      footer = static_text('footer')
      body = %{<div id="front-header">#{header}</div>\n} << body if
        header and header =~ /[^\s]/
      body << %{<div id="front-footer">#{footer}</div>\n} if
        footer and footer =~ /[^\s]/
      body = %{<div id="languages">#{language_list}</div>} << body if
        defined? GetText
    end

    # update cache
    cache[cache_key] = body

    @content_for_layout = body
  end

  # RSS feed of features or updates
  #
  def rss
    case @request['feed']
    when 'updates'
      feed = @request['feed']
      feed_title = _('Recent Updates')
    else
      feed = 'features'
      feed_title = _('Features')
    end

    render_rss('frontpage/' + feed) do |maker|
      maker.channel.title = config['site']['name'] + ': ' + feed_title
      maker.channel.description = strip_tags(static_text('header'))
      maker.channel.link = @request.base

      case feed
      when 'updates'
        updates_dataset[0]
      else
        features_dataset(limit_page)[0]
      end
    end
  end

  private

  # messages that are related to any focus (and are not comments or old
  # versions), ordered chronologically by date of relation to a focus (so that
  # when message is edited, it doesn't flow up)
  #
  # in case of multiple related focuses, the one related most recently is used
  #
  def features_dataset(limit)
    RdfDataSet.new(%{
SELECT ?msg
WHERE (rdf::predicate ?stmt dc::relation)
      (rdf::subject ?stmt ?msg)
      (rdf::object ?stmt ?focus)
      (dc::date ?stmt ?date)
      (s::inReplyTo ?msg ?parent)
      (dct::isVersionOf ?msg ?version_of)
      (s::rating ?stmt ?rating FILTER ?rating >= :threshold)
      (s::hidden ?msg ?hidden #{filter_hidden})
LITERAL ?parent IS NULL AND ?version_of IS NULL
GROUP BY ?msg
ORDER BY max(?date) DESC},
      limit,
      { :threshold => config['limit']['features_threshold'] })
  end

  # messages that are not comments or older version, sorted chronologically
  #
  def updates_dataset
    RdfDataSet.new(%{
SELECT ?msg
WHERE (dc::date ?msg ?date)
      (s::inReplyTo ?msg ?parent)
      (dct::isVersionOf ?msg ?version_of)
      (s::hidden ?msg ?hidden #{filter_hidden})
LITERAL ?parent IS NULL AND ?version_of IS NULL
ORDER BY ?date DESC})
  end

  # do Apache's job and include files in static headers
  #
  def gsub_ssi_file(static)
    return '' unless static.kind_of? String
    static.gsub(/<!--#include\s+file="([^"]+)"\s*-->/i) do
      name = $1
      name = @request.document_root + @request.uri_prefix + '/' + name unless '/' == name[0]
      File.open(name) {|f| f.read }
    end
  end

  # read from config and translate static header or footer
  #
  def static_text(name)
    text = config['site'][name]
    text = (
      text[@request.language] or
      text[default_language] or
      text.to_a[0][1]
    ) if text.kind_of? Hash
    gsub_ssi_file(text)
  end

  # language selection
  #
  def language_list
    return '' if
      'yes' == @request.cookie('nostatic') or
      not defined?(GetText)

    list = ''
    old_language = @request.language
    config['locale']['languages'].sort.each do |lang|
      lang.untaint
      @request.language = lang
      name = _('English')
      @request.language = old_language
      next if 'English' == name and 'en' != lang   # broken localization
      name += _(' (default language)') if
        lang == default_language
      name = %{<strong>#{name}</strong>} if lang == old_language
      list += %{<a href="member/set?lang=#{lang}">#{name}</a>\n}
    end
    %{<div id="languages">#{list}</div>}
  end
end
