#
# loader.rb
#
#   Copyright (c) 1998-2001 Minero Aoki <aamine@loveruby.net>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#

require 'socket'
require 'final'
require 'fileutils'
require 'tmail/port'


module TMail

  class MhLoader

    PORT_CLASS = MhPort

    def initialize( dir )
      FileTest.directory? dir or raise ArgumentError, "not directory: #{dir}"

      @dirname = dir
      @oldnext = nil
      @lasttime = nil
    end

    def inspect
      "#<#{type} #{@dirname}>"
    end

    attr_reader :dirname

    def each_port
      each_filename do |fname|
        yield PORT_CLASS.new(fname)
      end
      @lasttime = Time.now
    end

    alias each      each_port
    alias each_mail each_port

    def new_port
      PORT_CLASS.new nextfile
    end

    alias new_mail new_port

    def each_new_port( mtime = nil, &block )
      mtime ||= @lasttime
      return each_mail(&block) unless mtime
      return unless File.mtime(@dirname) >= mtime

      each_filename do |fname|
        if File.mtime(fname) > mtime then
          yield PORT_CLASS.new(fname)
        end
      end
      @lasttime = Time.now
    end

    alias each_newmail each_new_port
    alias each_new_mail each_new_port


    private

    def each_filename
      Dir.foreach( @dirname ) do |fname|
        path = "#{@dirname}/#{fname}"
        next unless /\A\d+\z/ === fname
        next unless FileTest.file? path
        yield path
      end
    end

    def nextfile
      n = @oldnext
      unless n then
        n = 0
        each_filename do |fname, base|
          i = base.to_i
          n = i if i > n
        end
        n += 1
      end

      while FileTest.exist? "#{@dirname}/#{n}" do
        n += 1
      end
      @oldnext = n

      "#{@dirname}/#{n}"
    end

  end



  class MboxLoader

    class << self

      def get_fromline( port )
        ad = 'nobody'
        if h = HeaderField.new_header( port, 'From' ) then
          if a = h.addrs[0] then
            ad = a.address
          end
        end

        t = File.mtime( port.filename )
        t = t.strftime( '%a %b %d %H:%M:%S %Y' )

        "From #{ad} #{t}"
      end

    end


    def initialize( fname, tmpdir = nil )
      @filename = File.expand_path( fname )
      FileTest.file? @filename or
              raise ArgumentError, "'#{fname}' is not normal file"

      unless tmpdir then
        tmpdir = "/tmp/tmail_mboxloader_#{$$}_#{__id__}"
        FileUtils.mkdir_p tmpdir
      else
        FileTest.directory? tmpdir or
                raise ArgumentError, "not directory: #{tmpdir}"
      end
      @real = MhLoader.new(tmpdir)
      @closed = false

      @finalizer = MboxLoader.mkfinal( @real, @filename )
      ObjectSpace.define_finalizer self, @finalizer
    end

    def self.mkfinal( ld, mbfile )
      proc {
        File.open( mbfile, 'w' ) do |f|
          ld.each do |port|
            fr = get_fromline( port )
            port.ropen do |s|
              f.puts fr
              f.puts s.read_all
            end
          end
        end
        FileUtils.rm_rf ld.dirname
      }
    end

    def close
      return if @closed
      
      ObjectSpace.undefine_finalizer self
      @finalizer.call
      @finalizer = nil
      @real = nil
      @closed = true
    end

    def each_mail( &block )
      close_check
      update
      @real.each_mail( &block )
    end

    alias each each_mail

    def each_newmail( mtime = nil )
      close_check
      update
      @real.each_newmail( mtime ) {|p| yield p }
    end

    def new_port
      close_check
      @real.new_port
    end

    alias new_mail new_port


    private

    FROMLINE = /\AFrom \S+ \w+ (\w+) (\d+) (\d+):(\d+):(\d+) (\d+)/

    def update
      return if FileTest.zero? @filename
      wf = t = p = nil

      lock( @filename ) {|f|
        begin
          f.each do |line|
            if /\AFrom / === line then
              wf.close if wf
              File.utime t, t, p.filename if t

              p = @real.new_port
              wf = p.wopen
              if m = FROMLINE.match(line) then
                t = Time.local( m[6].to_i, m[1], m[2].to_i,
                                m[3].to_i, m[4].to_i, m[5].to_i )
              else
                t = nil
              end
            else
              wf << line if wf
            end
          end
        ensure
          if wf and not wf.closed? then
            wf.close
            File.utime t, t, p.filename if t
          end
        end
      }
      File.truncate @filename, 0
    end
  
    def lock( fname )
      begin
        f = File.open( fname )
        f.flock File::LOCK_EX
        yield f
      ensure
        f.flock File::LOCK_UN
        f.close
      end
    end

    def close_check
      @closed and raise ArgumentError, 'accessing already closed mbox'
    end

  end



  class MaildirLoader

    PORT_CLASS = MaildirPort

    def initialize( dname = nil )
      @dirname = dname || ENV['MAILDIR']
      FileTest.directory? @dirname or
                      raise ArgumentError, "not directory: #{@dirname}"
      FileUtils.mkdir_p @new = "#{@dirname}/new"
      FileUtils.mkdir_p @tmp = "#{@dirname}/tmp"
      FileUtils.mkdir_p @cur = "#{@dirname}/cur"
    end

    def close
      @closed = true
    end


    def each_filename( dn )
      Dir.foreach( dn ) do |fn|
        full = "#{dn}/#{fn}"
        if fn[0] != ?. and FileTest.file? full then
          yield full, fn
        end
      end
    end

    def each_port
      each_filename( @cur ) do |full, fname|
        yield PORT_CLASS.new( full )
      end
    end

    alias each      each_port
    alias each_mail each_port

    def new_port
      fn = nil
      tmp = nil
      i = 0

      while true do
        fn = "#{Time.now.to_i}.#{$$}.#{Socket.gethostname}"
        tmp = "#{@tmp}/#{fn}"
        break unless FileTest.exist? tmp
        i += 1
        if i == 3 then
          raise IOError, "can't create new file in maildir"
        end
        sleep 1 #2
      end
      File.open( tmp, 'w' ){|f| f.write "\n\n" }
      cur = "#{@cur}/#{fn}"
      File.rename tmp, cur

      PORT_CLASS.new( cur )
    end

    alias new_mail new_port

    def each_new_port
      each_filename( @new ) do |src, fname|
        dest = @cur + '/' + fname
        File.rename src, dest
        yield PORT_CLASS.new(dest)
      end

      check_tmp
    end

    alias each_newmail each_new_port


    TOO_OLD = 60 * 60 * 36   # 36 hour

    def check_tmp
      old = Time.now.to_i - TOO_OLD
      
      each_filename( @tmp ) do |full, fname|
        if FileTest.file? full then
          if File.stat(full).mtime.to_i < nt then
            FileUtils.rm_f full
          end
        end
      end
    end
    
  end

end   # module TMail
