/*
 * @(#)AnalysisModuleData.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.report;


import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import net.sourceforge.groboutils.codecoverage.v2.IAnalysisMetaData;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;
import net.sourceforge.groboutils.codecoverage.v2.IChannelLogReader;
import net.sourceforge.groboutils.codecoverage.v2.IChannelLogRecord;
import net.sourceforge.groboutils.codecoverage.v2.IClassChannelLogReader;
import net.sourceforge.groboutils.codecoverage.v2.datastore.ClassRecord;
import net.sourceforge.groboutils.codecoverage.v2.datastore.IClassMetaDataReader;
import net.sourceforge.groboutils.codecoverage.v2.datastore.IMetaDataReader;
import net.sourceforge.groboutils.codecoverage.v2.datastore.MarkRecord;


/**
 * Combine all post-compilation data along with the coverage logs for
 * a specific analysis module.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     December 17, 2002
 * @see       IAnalysisMetaData
 */
public class AnalysisModuleData
{
    private IChannelLogReader logReader;
    private IClassMetaDataReader mdReader;
    private IAnalysisModule am;
    
    
    /**
     * 
     */
    public AnalysisModuleData( IAnalysisModule module,
            IMetaDataReader mdr, IChannelLogReader clr )
            throws IOException
    {
        if (module == null || mdr == null || clr == null)
        {
            throw new IllegalArgumentException( "No null args." );
        }
        this.mdReader = mdr.getClassReader( module );
        this.logReader = clr;
        this.am = module;
    }
    
    
    /**
     * Retrieves the corresponding analysis module.
     */
    public IAnalysisModule getAnalysisModule()
    {
        return this.am;
    }
    
    
    /**
     * Return the list of all class signatures for the module from the
     * post-compilation reader.
     */
    public String[] getClassSignatures()
            throws IOException
    {
        String cs[] = this.mdReader.getClassSignatures();
        return cs;
    }
    
    
    /**
     * Retrieves the class record with all marks.
     */
    public ClassRecord getClassRecord( String classSig )
            throws IOException
    {
        return this.mdReader.readClass( classSig );
    }
    
    
    /**
     * Retrieves all the mark records for the given class.
     */
    public MarkRecord[] getAllClassMarks( String classSig )
            throws IOException
    {
        return getClassRecord( classSig ).getMarksForAnalysisModule( this.am );
    }
    
    
    /**
     * Retrieves all unique log records for this module.
     */
    public IChannelLogRecord[] getChannelLogRecords( String classSig )
            throws IOException
    {
        Set set = new HashSet();
        IClassChannelLogReader reader = this.logReader.
            getReaderForClassSignature( classSig );
        IChannelLogRecord clr = reader.nextRecord();
        while (clr != null)
        {
            set.add( clr );
            clr = reader.nextRecord();
        }
        
        IChannelLogRecord[] records = (IChannelLogRecord[])set.toArray(
            new IChannelLogRecord[ set.size() ] );
        return records;
    }
    
    
    /**
     * Retrieves the sorted (by covered/not covered) set of marks
     * for the class.
     */
    public ClassMarkSet createClassMarkSet( String classSig )
            throws IOException
    {
        ClassRecord cr = getClassRecord( classSig );
        ClassMarkSet cms = new ClassMarkSet(
            cr.getClassName(),
            cr.getMethods(),
            getAllClassMarks( classSig ),
            getChannelLogRecords( classSig ) );
        return cms;
    }
}

