/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file game_ai.tpp
 * \brief Implmentation de fonctions d'intelligence artificielle.
 * \author Julien Jorge & Sbastien Angibaud
 */
#include <claw/max_vector.hpp>

//**************************** gamestate **************************************

/*---------------------------------------------------------------------------*/
/**
 * \brief Destructeur.
 */
template <class Action, class Numeric>
claw::ai::game::game_state<Action, Numeric>::~game_state()
{

} // ~game_state() [destructeur]

/*---------------------------------------------------------------------------*/
/**
 * \brief Indique le score minimal d'un tat. 
 */
template <class Action, class Numeric>
Numeric claw::ai::game::game_state<Action, Numeric>::min_score()
{
  return s_min_score; 
} // min_score()

/*---------------------------------------------------------------------------*/
/** 
 * \brief Indique le score maximal d'un tat. 
 */
template <class Action, class Numeric>
Numeric claw::ai::game::game_state<Action, Numeric>::max_score()
{
  return s_max_score; 
} // max_score()

/*---------------------------------------------------------------------------*/
/** 
 * \brief Troncature de l'valuation par les bornes min et max.
 * \param score_val Score  tronquer
 */
template <class Action, class Numeric>
typename claw::ai::game::game_state<Action, Numeric>::score
claw::ai::game::game_state<Action, Numeric>::fit
( score score_val ) const 
{ 
  if ( s_max_score < score_val ) 
    return s_max_score;
  else if ( score_val < s_min_score )
    return s_min_score;
  else
    return score_val;
} // fit()


//**************************** action_eval ************************************


/*---------------------------------------------------------------------------*/
/**
 * \brief Constructeur.
 * \param a Action.
 * \param e Evaluation de l'action.
 */
template <class Action, class Numeric>
claw::ai::game::action_eval<Action, Numeric>::action_eval( const Action& a,
                                                           const Numeric& e)
  : action(a), eval(e)
{

} // action_eval() [constructeur]

/*---------------------------------------------------------------------------*/
/**
 * \brief Comparaison de deux actions.
 * \return vrai si this->eval < ae.eval.
 */
template <class Action, class Numeric>
bool claw::ai::game::action_eval<Action, Numeric>::operator<
  ( const action_eval& ae ) const 
{
  return eval <  ae.eval; 
} // operator<()

/*---------------------------------------------------------------------------*/
/**
 * \brief Egalit de deux actions.
 * \return vrai si this->eval == ae.eval.
 */
template <class Action, class Numeric>
bool claw::ai::game::action_eval<Action, Numeric>::operator==
  ( const action_eval& ae ) const 
{
  return eval == ae.eval; 
} // operator==()




//********************************* min_max ***********************************


/*---------------------------------------------------------------------------*/
/**
 * \brief Evaluation d'un coup par la mthode MinMax.
 *
 * \todo Voir si un coup est gagnant, les suivants du mme niveau ne sont pas
 *       tests.
 *
 * \param depth Profondeur.
 * \param current_state Etat du jeu.
 * \param computer_turn Indique si c'est au tour de l'ordinateur.
 */
template <class State>
typename State::score claw::ai::game::min_max<State>::operator()
  ( int depth, const state& current_state, bool computer_turn ) const
{
  score score_val;

  // on a atteint la profondeur demande ou la partie est termine
  if ( current_state.final() || (depth == 0) )
    score_val = current_state.evaluate(); // on retourne l'valuation de l'tat.
  else
    {
      std::list<action> next_actions;
      typename std::list<action>::const_iterator it;
      state* new_state;

      // on rcupre les tats suivants.
      current_state.nexts_actions( next_actions );

      if ( next_actions.empty() )       // si on ne peut plus rien faire
        // on value l'tat o l'on est.
        score_val = current_state.evaluate();   
      else
        {
          if (computer_turn)    // si c'est l'ordi qui va jouer
            {                                   
              score_val = current_state.min_score();
                          
              for (it = next_actions.begin(); it!=next_actions.end(); ++it)
                {
                  new_state=static_cast<state*>(current_state.do_action(*it));

                  // on value le coup du joueur
                  score s = (*this)( depth-1, *new_state, false );
                                          
                  // on choisit l'action qui nous rapporte le plus
                  if (s > score_val)
                    score_val = s;

                  delete new_state;
                }
            }
          else  // c'est le tour du joueur
            {           
              score_val = current_state.max_score();
                          
              for (it = next_actions.begin(); it!=next_actions.end(); ++it)
                {
                  new_state=static_cast<state*>(current_state.do_action(*it));
                                  
                  // on value le coup de l'ordi
                  score s = (*this)( depth-1, *new_state, true );
                                  
                  // on choisit l'action qui lui rapporte le moins.
                  if (s < score_val)
                    score_val = s;
                                  
                  delete new_state;
                }
            }
        }
    }
  
  return score_val;
} // min_max::operator()




                
//******************************** alpha_beta *********************************


  /*---------------------------------------------------------------------------*/
/**
 * \brief Evaluation d'un coup par la mthode alpha_beta avec lagage 
 *        alpha bta.
 * \param depth profondeur.
 * \param current_state tat du jeu.
 * \param computer_turn indique si c'est au tour de l'ordinateur .
 */
template <class State>
typename State::score claw::ai::game::alpha_beta<State>::operator()
  ( int depth, const state& current_state, bool computer_turn ) const
{
  return this->calcul( depth, current_state, computer_turn, 
                       current_state.min_score(), current_state.max_score() ) ;
} // alpha_beta::operator()


/*---------------------------------------------------------------------------*/
/**
 * \brief Evaluation d'un coup par la mthode alpha_beta
 *
 * \todo Voir si un coup est gagnant, les suivants du mme niveau ne sont pas
 *       tests.
 *
 * \param depth profondeur.
 * \param current_state tat du jeu.
 * \param computer_turn indique si c'est au tour de l'ordinateur.
 * \param alpha Borne infrieure.
 * \param beta Borne suprieure.
 */
template <class State>
typename State::score claw::ai::game::alpha_beta<State>::calcul
( int depth, const state& current_state, bool computer_turn, score alpha,
  score beta ) const
{
  score score_val;
                
  // on a atteint la profondeur demande ou la partie est termine
  if ( current_state.final() || (depth == 0) )
    score_val = current_state.evaluate(); // on retourne l'valuation de l'tat.
  else
    {
      std::list<action> next_actions;
      typename std::list<action>::const_iterator it;
      State* new_state;

      // on rcupre les tats suivants.
      current_state.nexts_actions( next_actions );
          
      if ( next_actions.empty() )       // si on ne peut plus rien faire
        score_val = current_state.evaluate();   // on value l'tat o l'on est.
      else
        {
          if (computer_turn)    // si c'est l'ordi qui va jouer
            {             // c'est un noeud MAX : sort l'action la plus forte
              score_val = current_state.min_score();
                          
              it = next_actions.begin();

              while ( it!=next_actions.end() && (score_val < beta) )
                {
                  // le coup appliqu remontera le min de ses fils
                  new_state=static_cast<state*>(current_state.do_action(*it));

                  // on value le coup du joueur
                  score s = calcul( depth-1, *new_state, false, 
                                    std::max(alpha, score_val), beta );

                  // on choisit l'action qui nous rapporte le plus
                  if (s > score_val) 
                    score_val = s;
                                          
                  delete new_state;
                                          
                  ++it;
                }
            }
          else    // c'est le tour du joueur
            {     // c'est un noeud MIN : sort l'action la plus faible
              score_val = current_state.max_score();
                                        
              it = next_actions.begin();

              while ( it!=next_actions.end() && (score_val > alpha) )
                {
                  // le coup appliqu remontera le max de ses fils
                  new_state=static_cast<state*>(current_state.do_action(*it));

                  // on value le coup de l'ordi
                  score s = calcul( depth-1, *new_state, true, alpha,
                                    std::min(beta, score_val) );
                                                
                  // on choisit l'action qui lui rapporte le moins.
                  if (s < score_val)
                    score_val = s;
                  ++it;
                                                
                  delete new_state;
                }
            }
        }
    }

  return score_val;
} // alpha_beta::calcul()





//***************************** select_action *********************************




/*---------------------------------------------------------------------------*/
/**
 * \brief Slection d'une action.
 * \param depth Profondeur
 * \param current_state Etat du jeu
 * \param new_action <I>entre / sortie</I> L'action selectionne. Si aucune 
 *        action n'a un score suprieur au score minimal, ce paramtre n'est
 *        pas modifi.
 * \param computer_turn Indique si c'est au tour de l'ordinateur.
 */
template<class Method>
void claw::ai::game::select_action<Method>::operator()
  ( int depth, const state& current_state, action& new_action, 
    bool computer_turn ) const
{
  std::list<action> l;
  typename std::list<action>::iterator it;
  score best_eval;              
  Method method;

  // actions jouables par l'ordi
  current_state.nexts_actions( l );
  best_eval = current_state.min_score();

  for (it=l.begin(); it!=l.end(); ++it)
    {
      state* new_state;
      score eval;
                        
      // on effectue chaque action
      new_state = static_cast<state*>(current_state.do_action(*it));
      // et on regarde ce qu'elle vaut.
      eval = method(depth-1, *new_state, !computer_turn);

      delete new_state;

      // si c'est la meilleure, on la garde.
      if (eval > best_eval)
        {
          best_eval = eval;
          new_action = *it;
        }
    }
} // select_action::operator()


//*************************** select_random_action ****************************

/**
 * \brief Slection d'une action.
 * La slection se fait alatoirement sur les coups ayant le meilleur score.
 * \param depth Profondeur.
 * \param current_state Etat du jeu.
 * \param new_action <I>entre / sortie</I> L'action selectionne. Si aucune 
 *        action n'a un score suprieur au score minimal, ce paramtre n'est 
 *        pas modifi. 
 * \param computer_turn Indique si c'est au tour de l'ordinateur.
 */    
template<class Method>
void claw::ai::game::select_random_action<Method>::operator()
  ( int depth, const state& current_state, action& new_action, 
    bool computer_turn ) const
{
  std::list<action> l;
  typename std::list<action>::iterator it;
  action_eval<action, score> eval( new_action, current_state.min_score() );
  Method method;
  max_vector< action_eval<action, score> > events( eval );

  // actions jouables par l'ordi
  current_state.nexts_actions( l );

  for (it=l.begin(); it!=l.end(); ++it)
    {
      state* new_state;
                        
      // on effectue chaque action
      new_state = static_cast<state*>(current_state.do_action(*it));

      // et on regarde ce qu'elle vaut.
      eval.action = *it;
      eval.eval = method(depth-1, *new_state, !computer_turn);

      delete new_state;

      // si c'est la meilleure, on la garde.
      events.add( eval );
    }

  int i = rand() % events.get_v().size();
  new_action = events.get_v()[i].action;
} // select_random_action::operator()

