/* 
 **  mod_random2.c -- 
 */ 

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "http_request.h"
#include "util_script.h"
#include "http_connection.h"

#include "ap_config.h"
#include "apr_fnmatch.h"
#include "apr_strings.h"

module AP_MODULE_DECLARE_DATA random_module;

typedef struct {
  int enabled;
  apr_array_header_t *urls;
  apr_array_header_t *section_quotes;
  apr_array_header_t *ads;
  apr_table_t *handlers;
} random_conf;

typedef struct {
  char *name;
  apr_array_header_t *text;
} message_bank;

message_bank * rn_get_bank(apr_array_header_t *bank, const char *key) 
{
  unsigned int x;
  message_bank **members= (message_bank **) bank->elts;

  for (x= 0; x < bank->nelts; x++) 
  {
    if (!strcmp(key, members[x]->name))
      return members[x];
  }

  return NULL;
}

message_bank * rn_create_bank(apr_pool_t *p, const char *section) 
{
  message_bank *bank;

  if (section == NULL) 
    return NULL;

  bank= apr_pcalloc(p, sizeof (message_bank));
  bank->name= (char *)apr_pstrdup(p, section);

  bank->text= apr_array_make(p, 5, sizeof (char *));

  return bank;
}

int rn_check_table(const char *a) 
{
  if (a == NULL) 
    return 0;

  if ('1' == a[0])
    return 1;

  return 0;
}

int rn_table_find(const apr_table_t *t, const char *key) 
{
  const apr_array_header_t *hdrs_arr= NULL;
  const apr_table_entry_t *elts= NULL;
  unsigned int x;

  if (t == NULL)
    return 0;

  if (key == NULL)
    return 0;

  hdrs_arr= apr_table_elts(t);
  elts= (const apr_table_entry_t *)hdrs_arr->elts;

  for (x= 0; x < hdrs_arr->nelts; ++x) 
  {
    if (!apr_fnmatch(elts[x].key, key, APR_FNM_CASE_BLIND))
      if (rn_check_table(elts[x].val))
        return 1;
  }

  return 0;
}

char * rn_add_file(cmd_parms *cmd, const char *file) 
{
  apr_file_t *file_ptr;
  char buf[HUGE_STRING_LEN];
  char *content= NULL;
  apr_status_t rc;

  rc= apr_file_open(&file_ptr, file, APR_READ | APR_BINARY | APR_XTHREAD, 
                    APR_OS_DEFAULT, cmd->pool);
  if (rc != APR_SUCCESS) 
  {
    ap_log_error(APLOG_MARK, APLOG_WARNING, rc, cmd->server,
                 "mod_random: unable to open ad file(%s, O_RDONLY), skipping", file);
    return NULL;
  }

  while ((rc= apr_file_gets(buf, HUGE_STRING_LEN, file_ptr)) == APR_SUCCESS) 
  {
    if (content)
      content= (char *)apr_pstrcat(cmd->temp_pool, content, buf, NULL);
    else
      content= (char *)apr_pstrcat(cmd->temp_pool, buf, NULL);
  }
  apr_file_close(file_ptr);

  return content;
}

static int random_fixup(request_rec * r) 
{
  unsigned int token= 0;
  random_conf *cfg= NULL;
  unsigned int x= 0;
  char *type= NULL;
  message_bank **members= NULL;
  message_bank *bank= NULL;
  char **quotes= NULL;

  cfg= ap_get_module_config (r->per_dir_config, &random_module);

  if (cfg->enabled == 0)
    return DECLINED;

  if (cfg->handlers) 
  {
    if (r->handler) 
      type= (char *)apr_pstrdup(r->pool, r->handler);
    else 
      type= (char *)apr_pstrdup(r->pool, r->content_type);

    if (!rn_table_find(cfg->handlers, type))
      return DECLINED;
  }

  if (cfg->section_quotes) 
  {
    members= (message_bank **) cfg->section_quotes->elts;
    for (x= 0; x < cfg->section_quotes->nelts; x++) 
    {
      bank= members[x];

      if (bank->text->nelts) 
      {
        token= ((unsigned int)random() % bank->text->nelts);
        quotes= (char **) bank->text->elts;
        apr_table_setn(r->subprocess_env, bank->name, quotes[token]);
        apr_table_setn(r->notes, bank->name, quotes[token]);
      }
    }
  }

  if (cfg->ads)
  {
    members= (message_bank **) cfg->ads->elts;

    for (x= 0; x < cfg->ads->nelts; x++) 
    {
      bank= members[x];
      if (bank->text->nelts) 
      {
        token= ((unsigned int)random() % bank->text->nelts);
        quotes= (char **) bank->text->elts;
        apr_table_setn(r->subprocess_env, bank->name, quotes[token]);
        apr_table_setn(r->notes, bank->name, quotes[token]);
      }
    }
  } 


  return DECLINED;
}

static int random_handler(request_rec * r) 
{
  unsigned int token= 0;
  struct timeval tv_struct[2];
  random_conf *cfg= NULL;
  char **members= NULL;

  if (strcmp(r->handler,"random"))
    return DECLINED;

  gettimeofday(tv_struct, 0);
  srandom (tv_struct[0].tv_sec % tv_struct[0].tv_usec);

  cfg= ap_get_module_config (r->per_dir_config, &random_module);

  /* We need to be sending a better warning */
  if (cfg->urls == NULL) 
    return HTTP_NOT_FOUND;

  members= (char **) cfg->urls->elts;

  if (!cfg->urls->nelts) 
    return HTTP_NOT_FOUND;

  token= ((unsigned int)random() % cfg->urls->nelts);
  apr_table_setn(r->headers_out, "Cache-Control", "no-cache");
  apr_table_setn (r->headers_out, "Location", members[token]);

  return HTTP_MOVED_TEMPORARILY;
}

static int random_page_handler(request_rec * r) 
{
  const char *string= NULL;
  struct timeval tv_struct[2];

  gettimeofday(tv_struct, 0);
  srandom (tv_struct[0].tv_sec % tv_struct[0].tv_usec);

  if (strcmp(r->handler,"random-ad-page") && strcmp(r->handler,"random-quote-page"))
    return DECLINED;

  if (!strcmp(r->handler,"random-quote-page")) 
    string= apr_table_get(r->notes, "RANDOM_QUOTE");
  else
    string= apr_table_get(r->notes, "RANDOM_AD");

  if (string == NULL)
    return HTTP_NOT_FOUND;

  r->content_type= "text/html";      

  if (r->header_only)
    return OK;

  ap_rputs(string, r);
  return OK;
}

static void random2_register_hooks(apr_pool_t *p) 
{
  ap_hook_handler(random_handler, NULL, NULL, APR_HOOK_MIDDLE);
  ap_hook_handler(random_page_handler, NULL, NULL, APR_HOOK_MIDDLE);
  ap_hook_fixups(random_fixup, NULL, NULL, APR_HOOK_FIRST);
}

static void *mconfig_for_directory(apr_pool_t *p, char *dir) 
{
  random_conf *cfg= NULL;
  cfg= apr_palloc(p, sizeof(random_conf));

  memset(cfg, 0, sizeof(random_conf));

  return cfg;
}


static const char * add_random_url(cmd_parms * cmd, void *mconfig, const char *param) 
{
  apr_file_t *file_ptr;
  char buf[HUGE_STRING_LEN];
  random_conf *cfg= (random_conf *) mconfig;
  struct stat sbuf;
  apr_status_t rc;

  if (cfg->urls == NULL)
    cfg->urls= apr_array_make (cmd->pool, 5, sizeof (char *));

  if (stat(param, &sbuf) == 0)
  {
    rc= apr_file_open(&file_ptr, param, APR_READ | APR_BINARY | APR_XTHREAD, 
                      APR_OS_DEFAULT, cmd->pool);

    if (rc != APR_SUCCESS) 
    {
      ap_log_error(APLOG_MARK, APLOG_WARNING, rc, cmd->server,
                   "mod_random: unable to open(%s, O_RDONLY), skipping", param);
      return NULL;
    }

    while ((rc= apr_file_gets(buf, HUGE_STRING_LEN, file_ptr)) == APR_SUCCESS) 
      *(const char **) apr_array_push (cfg->urls)= (const char *)apr_pstrdup (cmd->pool, buf);

    apr_file_close(file_ptr);
  } 
  else 
  {
    *(const char **) apr_array_push (cfg->urls)= (const char *)apr_pstrdup (cmd->pool, param);
  }


  return NULL;
}

static const char * add_random_quote(cmd_parms * cmd, void *mconfig, const char *param, const char *section) 
{
  apr_file_t *file_ptr;
  char buf[HUGE_STRING_LEN];
  random_conf *cfg= (random_conf *) mconfig;
  message_bank *bank;
  struct stat sbuf;
  apr_status_t rc;

  if (cfg->section_quotes == NULL)
    cfg->section_quotes= apr_array_make (cmd->pool, 5, sizeof (message_bank *));

  /* We default to RANDOM_QUOTE if nothing is specified */
  section= section ? section : "RANDOM_QUOTE";

  if ((bank= rn_get_bank(cfg->section_quotes, section)) == NULL) 
  {
    bank= rn_create_bank(cmd->pool, section);
    *(message_bank **) apr_array_push (cfg->section_quotes)= (message_bank *) bank;
  }

  if (stat(param, &sbuf) == 0)
  {
    rc= apr_file_open(&file_ptr, param, APR_READ | APR_BINARY | APR_XTHREAD, 
                      APR_OS_DEFAULT, cmd->pool);

    if (rc != APR_SUCCESS) 
    {
      ap_log_error(APLOG_MARK, APLOG_WARNING, rc, cmd->server,
                   "mod_random: unable to open file(%s, O_RDONLY), skipping", param);
      return NULL;
    }

    while ((rc= apr_file_gets(buf, HUGE_STRING_LEN, file_ptr)) == APR_SUCCESS)
      *((char **) apr_array_push(bank->text))= (char *)apr_pstrdup(cmd->pool, buf);

    apr_file_close(file_ptr);
  } 
  else 
  {
    *(char **) apr_array_push(bank->text)= (char *)apr_pstrdup(cmd->pool, param);
  }

  return NULL;
}

static const char * add_random_ad(cmd_parms * cmd, void *mconfig, const char *param, const char *section) 
{
  random_conf *cfg= (random_conf *) mconfig;
  message_bank *bank;
  char *temp= NULL;
  struct stat sbuf;
  apr_dir_t *directory;
  apr_finfo_t dirent;
  apr_status_t rc;

  if (cfg->ads == NULL)
    cfg->ads= apr_array_make(cmd->pool, 5, sizeof (message_bank *));

  /* We default to RANDOM_AD if nothing is specified */
  section= section ? section : "RANDOM_AD";

  if ((bank= rn_get_bank(cfg->ads, section)) == NULL) 
  {
    bank= rn_create_bank(cmd->pool, section);
    *(message_bank **) apr_array_push (cfg->ads)= (message_bank *) bank;
  }

  if (stat(param, &sbuf) == 0)
  {
    if (S_ISDIR(sbuf.st_mode)) 
    {
      if ((rc= apr_dir_open(&directory, param, cmd->temp_pool)) != APR_SUCCESS) 
      {
        ap_log_error(APLOG_MARK, APLOG_WARNING, rc, cmd->server,
                     "mod_random: unable to open directory(%s, O_RDONLY), skipping", param);
        return NULL;
      }

      while (apr_dir_read(&dirent, APR_FINFO_MIN | APR_FINFO_NAME, directory) == APR_SUCCESS) 
      {
        if (dirent.filetype == APR_REG) 
        {
          temp= rn_add_file(cmd, (const char *)apr_pstrcat(cmd->temp_pool, param, "/", dirent.name, NULL));

          if (temp) 
            *(char **) apr_array_push (bank->text)= (char *)apr_pstrdup (cmd->pool, temp);
        }
      }

      apr_dir_close(directory);
    } 
    else 
    {
      temp= rn_add_file(cmd, param);

      if (temp) 
        *(char **) apr_array_push(bank->text)= (char *)apr_pstrdup(cmd->pool, temp);
    }
  } 
  else 
  {
    *(char **) apr_array_push(bank->text)= (char *)apr_pstrdup(cmd->pool, param);
  }

  return NULL;
}

static const char * add_handler(cmd_parms * cmd, void *mconfig, const char *handler) 
{
  random_conf *cfg= (random_conf *) mconfig;

  if (cfg->handlers == NULL)
    cfg->handlers= (apr_table_t *)apr_table_make(cmd->pool, 3);

  apr_table_setn(cfg->handlers, handler, "1");

  return NULL;
}

static const command_rec random_module_cmds[]= {
  AP_INIT_FLAG("RandomEngine", ap_set_flag_slot, 
               (void *) APR_OFFSETOF(random_conf, enabled), OR_ALL,
               "Use this to turn on and off random quotes."),
  AP_INIT_TAKE1("RandomURL", add_random_url, NULL, OR_ALL,
                "A filename with one URL per-line."),
  AP_INIT_TAKE12("RandomQuote", add_random_quote, NULL, OR_ALL,
                 "Takes either a double quoted string or a filename. An optional second parameter lets you adjust whate section the quote is added to."),
  AP_INIT_TAKE12("RandomAd", add_random_ad, NULL, OR_ALL,
                 "Takes either a double quoted string, a filename, or a directory name to read files from. An optional second parameter lets you adjust whate section the ad is added to."),
  AP_INIT_TAKE1("RandomHandler", add_handler, NULL, OR_ALL,
                "Enable which handled types will be supplied with ads or quotes."),
  {NULL},
};

/* Dispatch list for API hooks */
module AP_MODULE_DECLARE_DATA random_module= {
  STANDARD20_MODULE_STUFF, 
  mconfig_for_directory, /* create per-dir    config structures */
  NULL,                  /* merge  per-dir    config structures */
  NULL,                  /* create per-server config structures */
  NULL,                  /* merge  per-server config structures */
  random_module_cmds,    /* table of config file commands       */
  random2_register_hooks /* register hooks                      */
};

