/***************************************************************************
                          languagedefinition.h  -  description
                             -------------------
    begin                : Wed Nov 28 2001
    copyright            : (C) 2001 by Andre Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef LANGUAGEDEFINITION_H
#define LANGUAGEDEFINITION_H

#include <string>
#include <map>
#include <iostream>
#include <fstream>
#include <iterator>
#include <sstream>

#include "configurationreader.h"
//#include "stringtools.h"
#include "platform_fs.h"
#include "enums.h"


namespace highlight {

/** maps keywords and the corresponding class IDs*/
typedef map <string, int> KeywordMap;

/** maps keyword prefixes and the corresponding class IDs*/
typedef map <unsigned char, int> PrefixMap;

/**\brief Contains specific data of the programming language being processed.

   The load() method will only read a new language definition if the given
   file path is not equal to the path of the current language definition.

* @author Andre  Simon
*/

class LanguageDefinition  {

  public:

    LanguageDefinition();

    /**\return Symbol string, containg all known symbols with the referencing state ids*/
    string &getSymbolString();

    /** \return Prefix of raw strings */
    unsigned char getRawStringPrefix();

   /** \return Continuation Character */
    unsigned char getContinuationChar();

    /** \return List of characters allowed within identifiers */
    string &getAllowedChars();

    /** \return true if syntax highlighting is enabled*/
    bool getSyntaxHighlight();

    /** \return True if language is case sensitive */
    bool isIgnoreCase();

    /** \param s String
         \return class id of keyword, 0 if s is not a keyword */
    int isKeyword(const string &s);


    /** \return true if c is member of prefix list*/
    bool isPrefix(unsigned char c);

    /** Load new language definition
        \param langDefPath Path of language definition
        \param clear Test if former data should be deleted
        \return True if successfull  */
    bool load(const string& langDefPath, bool clear=true);

    /** \return  True if programming language is VHDL */
    bool isVHDL();

    /** \return  True if programming language is Java */
    bool isJava();

    /** \return True if multi line comments may be nested */
    bool allowNestedMLComments();

    /** \return True if highlighting is disabled */
    bool highlightingDisabled();

    /** \return True if single line comments must start at coloumn 1 */
    bool isFullLineComment();

    /** \return True the next load() call will load a new language definition
        \param  langDefPath Path to language definition  */
    bool needsReload(const string &langDefPath);

    /** \return True if current language may be reformatted (c, c++, c#, java) */
    bool enableReformatting();

    /** \return Class ID of given keyword delimiter prefix
        \param prefix Keyword delimiter prefix   */
    unsigned int getDelimPrefixClassID(const string& prefix);

    /** \return keywords*/
    const KeywordMap& getKeywords() const;

    /** \return keyword classes*/
    const vector<string>& getKeywordClasses() const;

  private:
    // string containing symbols and their IDs of the programming language
    string symbolString;

    // string with special characters that may occour in keywords
    string allowedChars;

    // path to laoed language definition
    string currentPath;

    KeywordMap keywords;

    vector <string> keywordClasses;

    KeywordMap delimiterPrefixes;

    PrefixMap prefixes;

    // keywords are not case sensitive if set
    bool ignoreCase,
    disableHighlighting,
    //escCharsAllowedOutsideStrings,

    // switch to enable VHDL workarounds
    vhdl_mode,

    // switch to enable Java workarounds
    java_mode,

    // allow nested multi line comment blocks
    allowNestedComments,

    // single line comments have to start in coloumn 1 if set
    fullLineComment,

    // code formatting is enabled if set
    reformatCode;

    // Character, die eine Variable bzw. ein Keyword kennzeichnen
    unsigned char rawStringPrefix,
                  continuationChar;

    /** setzt Membervariablen auf Defaultwerte */
    void reset();

    // add a symbol sequencs to the symbolStream
    void addSimpleSymbol(stringstream& symbolStream, State state,
                         const string& paramValues );

    // add a delimiter symbol sequencs to the symbolStream
    void addDelimiterSymbol(stringstream& symbolStream,
                            State stateBegin, State stateEnd,
                            const string& paramValues,
                            unsigned int classID=0);

    bool getFlag( string& paramValue);

    unsigned char getSymbol(const string& paramValue);

    // generate a unique class ID if the class name
    unsigned int generateNewKWClass(const string& newClassName);

    // add keywords to the given class
    void addKeywords(const string &kwList, int classID);

  };

}
#endif
