#
# Copyright 2009 Martin Owens
#
# This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>
#
"""
Small app for selecting launchpad projects
"""

# Import standard python libs
import os
import gtk
import gtk.gdk
import pango
import logging
import yaml

# Various required variables and locations
from GroundControl.gtkviews import (
    GtkApp, ThreadedWindow,
    IconManager, TreeView
)
from GroundControl.gtkcommon import StatusApp
from GroundControl.launchpad import (
    LaunchpadProject, LaunchpadProjects, get_launchpad
)

project_icons = IconManager('project')
PROJECT_FILE = '.gcproject'
ITEM_MARKUP = """<b><big>%s</big></b>
%s
  <small><i>lp:%s</i></small>"""

from shutil import copyfile
from bzrlib import transport, bzrdir


class Project(object):
    """Simple object for managing a project directory"""
    def __init__(self, path):
        self._path   = path
        self._config = None
        self.broken  = False
        if self.config and not self.broken:
            logging.debug("Loaded project %s" % self.pid)

    @property
    def config(self):
        """Load a configuration file"""
        if not self._config:
            cfile = os.path.join(self._path, PROJECT_FILE)
            self.load_project(cfile)
            if not self._config:
                raise IOError("Can't find project configuration file")
            if type(self._config) != dict:
                self._config = None
        return self._config

    def load_project(self, filename, regenerate=True):
        """Try and load a project's configuration file"""
        if os.path.isdir(self._path) and os.path.exists(filename):
            fhl = open(filename, 'r')
            self._config = yaml.load(fhl)
            if type(self._config) != dict and regenerate:
                self.broken = True
            fhl.close()

    def regenerate(self, widget, window, callback=None):
        """Try and regenerate a project's configuration"""
        if not callback:
            callback = self.regen_done
        ProjectCreateApp(project=os.path.basename(self._path),
            path=os.path.dirname(self._path),
            parent=window,
            callback=callback,
            start_loop=True)

    def regen_done(self):
        """What happens when we've regnerated the configuration"""
        self.load_project(self._filename, False)

    @property
    def pid(self):
        """Return a project id (nickname)"""
        return str(self.config['id'])

    @property
    def name(self):
        """Return a project name (with spaces)"""
        return str(self.config['name'])

    def logo(self):
        """Return the prospective logo filename"""
        return os.path.join(self._path, '.logo.png')


class ProjectCreateApp(StatusApp):
    """Show a status window when we're making a project"""
    task_name = _("Creating Project")

    def do_work(self):
        """Make our project directory and populate it"""
        project = self.project

        # When regenerating a launchpad project's data we won't have
        # A complete project object, just it's id. So Load the object.
        if type(project) in (str, unicode):
            self.update(0, _("Getting Launchpad Details"))
            lpa = get_launchpad().launchpad
            try:
                project = LaunchpadProject(lpa, oid=project)
            except KeyError:
                logging.error("Couldn't find existing project, removing.")
                return os.unlink(os.path.join(self.path, project, PROJECT_FILE))
                
        self.path = os.path.join(self.path, project.id)
        if not os.path.exists(self.path):
            self.update(0.1, _("Creating Bazaar Cache"))
            to_transport = transport.get_transport(self.path)
            to_transport.ensure_base()
            repo_format = bzrdir.format_registry.make_bzrdir('default')
            newdir = repo_format.initialize_on_transport(to_transport)
            repo = newdir.create_repository(shared=True)
            repo.set_make_working_trees(True)
        filename = os.path.join(self.path, PROJECT_FILE)
        if filename:
            self.update(0.3, _("Creating Project Configuration"))
            filename = os.path.join(self.path, PROJECT_FILE)
            fhl = open(filename, 'w')
            fhl.write(yaml.dump({
                'id' : project.id,
                'name' : project.name,
                'desc' : project.desc,
            }))
            fhl.close()
            self.update(0.5, _("Getting Branding"))
            self._save_image(project, "brand")
            self.update(0.7, _("Getting Logo"))
            self._save_image(project, "logo")
            self.update(0.9, _("Getting Icon"))
            self._save_image(project, "icon")
            self.update(1, _("All Done"))

    def _save_image(self, project, name):
        """Save images to the disk as required"""
        filename = os.path.join( self.path, ".%s.png" % name )
        result = project._image_file(name)
        if result:
            copyfile( result, filename )
        else:
            logging.debug("Not saving image - %s doesn't exist." % name)

    def load_vars(self, args, kwargs):
        """Load our project creation"""
        self.project = kwargs.pop('project')
        self.path    = kwargs.pop('path')


class SelectionWindow(ThreadedWindow):
    """Select a project to import."""
    name = 'project_search'

    def load(self, path, *args, **kwargs):
        """Load the project selection GUI"""
        self.slist      = None
        self.child      = None
        self._path      = path
        self._selected  = None
        self._launchpad = None
        # Setup the list of projects from a search
        self.unselected()
        self.slist = ProjectsView(self.widget('projectslist'),
            selected=self.selected,
            unselected=self.unselected)
        super(SelectionWindow, self).load(*args, **kwargs)

    @property
    def launchpad(self):
        """Return a launchpad object"""
        if not self._launchpad:
            self._launchpad = get_launchpad()
        return self._launchpad

    def inital_thread(self):
        """What to run when we execute the thread."""
        # Set up the launchpad projects object and connect search
        self.lpp = LaunchpadProjects(self.launchpad.launchpad)
        self.lpp.connect_signal("search_finish", self.call, 'finish_search')
        self.lpp.connect_signal("search_result", self.call, 'add_search_item')
        self.call('finish_search')

    def add_search_item(self, item):
        """Add an item to the slist"""
        return self.slist.add_item(item)

    def signals(self):
        """project window signals"""
        return {
            'search' : self.project_search,
        }

    def get_args(self):
        """Return the selected project"""
        return {
            'project' : self.selected(),
        }

    def selected(self, item=None):
        """An item has surely been selected."""
        if item:
            self._selected = item
            self.widget('buttonok').set_sensitive(True)
        return self._selected

    def unselected(self):
        """All items unselected"""
        self._selected = None
        self.widget('buttonok').set_sensitive(False)

    def project_search(self, button=None):
        """Search for projects in launchpad."""
        self.slist.clear()
        self.widget("findButton").hide()
        self.widget("term").hide()
        self.widget("searchLabel").set_text(
            _("Searching Launchpad - please wait..."))
        terms = self.widget("term").get_text()
        # This is a threaded search, so be careful about
        # What is loaded and what is updated in gtk.
        self.slist.search_term(term=terms)
        self.start_thread(self.lpp.search, terms)

    def finish_search(self, widget=None):
        """Event of finishing the search process."""
        self.widget("findButton").show()
        self.widget("term").show()
        self.widget("searchLabel").set_text("Name:")

    def is_valid(self):
        """Return true is a project is selected"""
        return self._selected != None


class ProjectSelection(GtkApp):
    """Application for loading a project"""
    gtkfile = 'project-select.glade'
    windows = [ SelectionWindow ]


class ProjectsView(TreeView):
    """Controls and operates a table as a projects view."""
    # We can set variables here because this is expected to be
    # A single object class and not multiples.
    _search_term = None

    def get_item_id(self, item):
        """Return a project's unique id"""
        return item.id

    def setup(self):
        """Setup a treeview for showing services"""
        def text_cell_func(cell_layout, renderer, model, item_iter):
            """Render the text of the services tree view"""
            item = model.get_value(item_iter, 0)
            name = str(item.name).replace('&', '&amp;')
            desc = str(item.desc).replace('&', '&amp;')
            markup = ITEM_MARKUP % (name, desc, item.id)
            renderer.set_property("markup", markup)

        def icon_cell_func(column, cell, model, item_iter):
            """Reender the icons of the services tree view"""
            item = model.get_value(item_iter, 0)
            img = project_icons.get_icon(item.logo())
            img2 = img.scale_simple(64, 64, gtk.gdk.INTERP_BILINEAR)
            cell.set_property("pixbuf", img2)
            cell.set_property("visible", True)
            
        def sort_projects_func(model, iter1, iter2):
            """Sorts projects by name to have them ordered by search param, 
               rather than the order returned by LP"""
            search_term = self.search_term().lower()
            
            proj1 = str(model.get_value(iter1, 0).name).lower()
            proj2 = str(model.get_value(iter2, 0).name).lower()
            proj1 = cmp(search_term, proj1)
            proj2 = cmp(search_term, proj2)
            
            if proj1 == 0:
                return -1
            elif proj2 == 0:
                return 1
            else:
                return 0

        svlist = super(ProjectsView, self).setup()
        model = svlist.get_model()
        model.set_sort_func(0, sort_projects_func)
        model.set_sort_column_id(0, gtk.SORT_ASCENDING)
        
        column = gtk.TreeViewColumn((_("Online Projects")))
        column.set_sizing(gtk.TREE_VIEW_COLUMN_AUTOSIZE)
        column.set_expand(True)
        # The icon
        renderer_icon = gtk.CellRendererPixbuf()
        renderer_icon.set_property("ypad", 8)
        renderer_icon.set_property("xpad", 8)
        column.pack_start(renderer_icon, False)
        column.set_cell_data_func(renderer_icon, icon_cell_func)
        # The name
        renderer = gtk.CellRendererText()
        renderer.props.wrap_width = 660
        renderer.props.wrap_mode = pango.WRAP_WORD
        column.pack_start(renderer, True)
        column.set_cell_data_func(renderer, text_cell_func)
        # Add the required coluns to the treeview
        svlist.append_column(column)
        
    def search_term(self, term=None):
        """ Returns the most recently searched term, and sets search term
            when passed as a named parameter"""
        if term is not None:
            self._search_term = term
        return self._search_term

