
#include <sys/time.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <glib/glist.h>

#include "stream.h"
#include "libhttp.h"

extern struct http_server	*hserver;

static void stream_remove_receiver(struct http_receiver_s *hr) {
	struct stream_s		*s=hr->stream;
	struct http_connection	*hc=hr->hc;

	s->http_receiver=g_list_remove(s->http_receiver, hr);
	s->receiver--;

	logwrite(LOG_INFO, "stream_http: dropping connection to %s for %s",
				inet_ntoa(hc->sin.sin_addr),
				hc->url);

	http_drop_connection(hc);
}

static int stream_cb_http(struct http_connection *hc, int cbtype, void *arg) {

	switch(cbtype) {
		case(HCB_QUERY): {
			struct http_receiver_s	*hr;
			struct stream_s		*s=arg;	/* HCB_QUERY returns http_url args */

			hr=calloc(1, sizeof(struct http_receiver_s));
			hr->hc=hc;
			hr->stream=s;

			/* Put into stream output list */
			s->http_receiver=g_list_append(s->http_receiver, hr);
			s->receiver++;

			/* Store http_receiver into http_connection structure */
			hc->arg=hr;

			/* Return head */
			http_header_start(hc, "200 OK", "application/octet-stream");
			http_header_nocache(hc);
			http_header_clength(hc, -1);
			http_header_end(hc);

			logwrite(LOG_INFO, "stream_http: connection from %s for %s",
						inet_ntoa(hc->sin.sin_addr),
						hc->url);
			break;
		}
		case(HCB_ERROR): {
			stream_remove_receiver(hc->arg);
			break;
		}
	}
	return 1;
}

#define SOUT_HTTP_CHUNK_SIZE	20000

int stream_init_http(struct stream_s *s) {

	s->buffer=calloc(1, SOUT_HTTP_CHUNK_SIZE);
	if (s->buffer == NULL)
		return 0;

	s->hurl=calloc(2, sizeof(struct http_url));

	s->hurl->url=s->url;
	s->hurl->cb=stream_cb_http;
	s->hurl->arg=(void *) s;

	http_register_url(hserver, s->hurl);

	return 0;
}

#define HTTP_MAX_QUEUED		(200*1024)
#define HTTP_MAX_OVERFLOW	20

void stream_send_http_one(gpointer data, gpointer user_data) {
	struct http_receiver_s	*hr=data;
	struct stream_s		*s=user_data;

	/*
	 * Check how many bytes we already have queued on this
	 * HTTP connection. Users might connect from low bandwidth
	 * links not beeing able to transmit the full feed. We must
	 * avoid consuming all memory.
	 *
	 * If the situation persists too long we drop the connection
	 *
	 */
	if (http_get_queue(hr->hc) > HTTP_MAX_QUEUED) {
		hr->overflow++;
		if (hr->overflow > HTTP_MAX_OVERFLOW)
			stream_remove_receiver(hr);
		return;
	}
	hr->overflow=0;

	/*
	 * We cant reuse evbuffer as they are empty after
	 * passing the the buffevent layer - Thus we recreate
	 * the buffer every time we send it out. This involes
	 * a little more memcpy as really necessary but for now
	 * its enough
	 *
	 */
	http_return_stream(hr->hc, s->buffer, s->buffervalid);
}

void stream_send_http(struct stream_s *s, uint8_t *tsp) {
	/* Copy TS packet to packet buffer */
	memcpy(&s->buffer[s->buffervalid], tsp, TS_PACKET_SIZE);
	s->buffervalid+=TS_PACKET_SIZE;

	/* check whether another packet would fit ? */
	if (s->buffervalid + TS_PACKET_SIZE > SOUT_HTTP_CHUNK_SIZE) {
		/*
		 * If the output buffer is full - loop on all http sesssions
		 * and send out the buffer as a http chunk
		 */
		g_list_foreach(s->http_receiver,
				stream_send_http_one, (gpointer) s);
		s->buffervalid=0;
	}
}
