#! /usr/bin/env python

#Common imports
import os, sys
import unittest
import DiskSearchTest
sys.path.append(os.path.join(os.path.dirname(os.path.abspath(__file__)), '..'))

#import specific to the test
from libdisksearch import Disk
from libdisksearch import DSExport
import ConstantsForTests
import tempfile
from xml.dom.minidom import getDOMImplementation

class TestCase(DiskSearchTest.TestCase):
    def setUp(self):
        self.export = DSExport.ExportXML()
        self.dirname = None
        self.filename = None

    def tearDown(self):
        self.rmtree(self.dirname)
        self.rmfile(self.filename)

    def testInit(self):
        self.assertEqual(self.export.id, 0)
        self.assertEqual(self.export.formats, {'flat': 'Flat',  'tree': 'Tree'})

    def testGetUniqueId(self):
        self.assertEqual(self.export.id, 0)
        self.assertEqual(self.export.getUniqueId(), str(1))
        self.assertEqual(self.export.id, 1)
        self.assertEqual(self.export.getUniqueId(), str(2))
        self.assertEqual(self.export.id, 2)

    def testGetDomByPath(self):
        impl = getDOMImplementation()
        newdoc = impl.createDocument(None, "root", None)
        top_element = newdoc.documentElement
        top_element.setAttribute('id', '0')
        dir = newdoc.createElement('folder')
        dir.setAttribute('path', '/')
        dir.setAttribute('id', '10')
        top_element.appendChild(dir)

        dir2 = newdoc.createElement('folder')
        dir2.setAttribute('path', 'example-1')
        dir2.setAttribute('id', '20')
        dir.appendChild(dir2)

        dir3 = newdoc.createElement('folder')
        dir3.setAttribute('path', 'example-2')
        dir3.setAttribute('id', '30')
        dir.appendChild(dir3)

        dir4 = newdoc.createElement('folder')
        dir4.setAttribute('path', 'nested_1')
        dir4.setAttribute('id', '40')
        dir3.appendChild(dir4)

        dir5 = newdoc.createElement('folder')
        dir5.setAttribute('path', 'nested_2')
        dir5.setAttribute('id', '50')
        dir3.appendChild(dir5)

        #already created dirs
        parent = self.export.getDomByPath(top_element, newdoc, '/example-1')
        self.assertEqual(parent.getAttribute('id'),  '20')

        parent = self.export.getDomByPath(top_element, newdoc, '/example-2')
        self.assertEqual(parent.getAttribute('id'),  '30')

        parent = self.export.getDomByPath(top_element, newdoc, '/example-2/nested_1')
        self.assertEqual(parent.getAttribute('id'),  '40')

        parent = self.export.getDomByPath(top_element, newdoc, '/example-2/nested_2')
        self.assertEqual(parent.getAttribute('id'),  '50')

    def testGetDomByPath2(self):
        impl = getDOMImplementation()
        newdoc = impl.createDocument(None, "root", None)
        top_element = newdoc.documentElement
        top_element.setAttribute('id', '0')
        #not created dirs
        dir = self.export.getDomByPath(top_element, newdoc, '/')
        self.assertEqual(dir.getAttribute('id'),  '1')

        dir = self.export.getDomByPath(top_element, newdoc, '/example-2/nested_3')
        self.assertEqual(dir.getAttribute('id'),  '3') #number 2 is /example-2


    def testProcessDir(self):
        impl = getDOMImplementation()
        newdoc = impl.createDocument(None, "root", None)
        top_element = newdoc.documentElement
        top_element.setAttribute('id', '0')
        dir = newdoc.createElement('folder')
        dir.setAttribute('path', '/')
        top_element.appendChild(dir)
        self.export.procesDir(newdoc, dir, ConstantsForTests.diskContentList[0])
        self.assertEqual(len(dir.childNodes), 2)
        self.assertEqual(dir.childNodes[0].getAttribute('path'), 'file_1')
        self.assertEqual(dir.childNodes[1].getAttribute('path'), 'file_2')


    def testExportDiskFlat (self):
        disk1 = Disk.Disk(ConstantsForTests.name, ConstantsForTests.type, ConstantsForTests.comment, '1234567890', ConstantsForTests.diskContentList)
        self.filename = os.tempnam('/tmp', 'ds')
        self.export.printXML(disk1, self.filename, 'flat')
        self.assertFileContainsXML(self.filename, ConstantsForTests.exportXmlFlat %
                {
                    'path': os.path.abspath(os.path.join(os.path.dirname(__file__), '..', 'resource', 'print.xsl')),
                    'comment': ConstantsForTests.comment,
                    'type': ConstantsForTests.type,
                    'name': ConstantsForTests.name
                }
            )

    def testExportDiskTree (self):
        disk1 = Disk.Disk(ConstantsForTests.name, ConstantsForTests.type, ConstantsForTests.comment, '1234567890', ConstantsForTests.diskContentList)
        self.filename = os.tempnam('/tmp', 'ds')
        self.export.printXML(disk1, self.filename, 'tree')
        self.assertFileContainsXML(self.filename, ConstantsForTests.exportXmlTree %
                {
                    'path': os.path.abspath(os.path.join(os.path.dirname(__file__), '..', 'resource', 'print.xsl')),
                    'comment': ConstantsForTests.comment,
                    'type': ConstantsForTests.type,
                    'name': ConstantsForTests.name
                }
            )

    def testExportDiskTreeWeirdList (self):
        disk1 = Disk.Disk(ConstantsForTests.name, ConstantsForTests.type, ConstantsForTests.comment, '1234567890', ConstantsForTests.weirdList)
        self.filename = os.tempnam('/tmp', 'ds')
        self.export.printXML(disk1, self.filename, 'tree')
        self.assertFileContainsXML(self.filename, ConstantsForTests.weirdXmlTree %
                {
                    'path': os.path.abspath(os.path.join(os.path.dirname(__file__), '..', 'resource', 'print.xsl')),
                    'comment': ConstantsForTests.comment,
                    'type': ConstantsForTests.type,
                    'name': ConstantsForTests.name
                }
            )


def suite():
    suite = unittest.TestSuite()
    suite.addTest(unittest.makeSuite(TestCase))
    return suite

if __name__ == '__main__':
    unittest.main()
