/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/macros/invariance_annotation.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref
 * DIAGNOSTICS_BASE_CLASS_INVARIANCE_ENTER and @ref
 * DIAGNOSTICS_BASE_CLASS_INVARIANCE_EXIT
 *
 * $Id: invariance_annotation.t.cpp,v 1.14 2005/06/23 09:54:22 esdentem Exp $
 *
 * @author Christian Schallhart
 * 
 * @todo test @ref DIAGNOSTICS_BASE_CLASS_INVARIANCE_GUARD and @ref
 * DIAGNOSTICS_BASE_CLASS_INVARIANCE_ASSERT
 */

#include <diagnostics/unittest.hpp>

// component
#include <diagnostics/macros/invariance_annotation.hpp>

// backdoor
#include <diagnostics/unittest/test_system/current_test_logger.hpp>

// used component
#include <diagnostics/frame/logging_config.hpp>
#include <diagnostics/util/to_string.hpp>

// test support
#include <diagnostics/util/dummy_logger.ts.hpp>
#include <diagnostics/util/assert_record.ts.hpp>

#define TEST_COMPONENT_NAME invariance_annotation
#define TEST_COMPONENT_NAMESPACE diagnostics


DIAGNOSTICS_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;
using namespace unittest;
using ::diagnostics::unittest::internal::Current_Test_Logger;

#define STR_WHAT(e) (::std::string("EXCEPTION=\"")+e.name()+"\" WHAT=\"" + e.what() + "\"")

#define STR_WHAT_INT(e) (::std::string("EXCEPTION=\"int\" WHAT=\"") + ::diagnostics::internal::to_string(e) +"\"")

class Dummy_Class_With_Invariance
{
    typedef Dummy_Class_With_Invariance Self;
public:
    Dummy_Class_With_Invariance() 
	: m_class_invariance_called(0),
	  m_throw(false)
    {
    }

    void method_with_guard(bool const t1, 
			   bool const t2)
    {
	m_class_invariance_called=0;
	m_throw=t1;
	::diagnostics::internal::Class_Invariance_Guard_Throw<Self> g(this); 
	m_throw=t2;
    }
    

    void method_with_macros(bool const t1,
			    bool const t2)
    {
	m_class_invariance_called=0;
	m_throw=t1;
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER;
	m_throw=t2;
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(LEVEL_AUDIT,10,0,::diagnostics::Exception,STR_WHAT);

    }

    void method_with_return(bool const t1,
			    bool const t2)
    {
	m_class_invariance_called=0;
	m_throw=t1;
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER;
	m_throw=t2;
	return;
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(LEVEL_AUDIT,20,0,::diagnostics::Exception,STR_WHAT);
    }


    void method_with_throw_implicit_te(bool const t1,
				       bool const t2)
    {
	m_class_invariance_called=0;
	m_throw=t1;
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER;
	m_throw=t2;
	throw Test_Exception("Method throwing");
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(LEVEL_AUDIT,30,0,::diagnostics::Exception,STR_WHAT);
    }

    void method_with_throw_explicit_te(bool const t1,
				       bool const t2)
    {
	m_class_invariance_called=0;
	m_throw=t1;
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER;
	m_throw=t2;
	throw Test_Exception("Method throwing");
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(LEVEL_AUDIT,40,1,int,STR_WHAT_INT);
    }

    void method_with_throw_implicit_te_int_as_base(bool const t1,
						   bool const t2)
    {
	m_class_invariance_called=0;
	m_throw=t1;
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_ENTER;
	m_throw=t2;
	throw Test_Exception("Method throwing");
	DIAGNOSTICS_BASE_CLASS_THROWING_INVARIANCE_EXIT(LEVEL_AUDIT,50,0,int,STR_WHAT_INT);
    }


    void m_class_invariance() const
    {
	++m_class_invariance_called;
	if(m_throw) throw Test_Exception("Invariance throwing");
    }

    int class_invariance_called() const
    {
	return m_class_invariance_called;
    }
    
private:
    mutable int m_class_invariance_called;

    bool m_throw;
};


void guard(Test_Data & data)
{
    // never mind the leak
    Dummy_Logger * logger=new Dummy_Logger;
    Dummy_Logger::Records_t & records(logger->records());
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);	

    Dummy_Class_With_Invariance d;

    TEST_EXCEPTIONLESS_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_guard(false,false);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_EXCEPTIONLESS_BLOCK_EXIT;
    TEST_ASSERT(d.class_invariance_called()==2);
    
    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_guard(true,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Invariance throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==1);

    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_guard(false,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Invariance throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==2);

    Logging_Config::unregister_logger(logger);	

    // checking for the trace of the dropped exception
    Iter_t current(records.begin()); // skip LOG_OPEN
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN); 
    ++current;
    TEST_ASSERT(current->type()==TYPE_LOG_CLOSE); 
    TEST_ASSERT(current+1==records.end());  
    delete logger;
}


void macros(Test_Data & data)
{
    // never mind the leak
    Dummy_Logger * logger=new Dummy_Logger;
    Dummy_Logger::Records_t & records(logger->records());
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);	

    Dummy_Class_With_Invariance d;

    d.method_with_macros(false,false);
    TEST_ASSERT(d.class_invariance_called()==2);
    
    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_macros(true,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Invariance throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==1);

    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_macros(false,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Invariance throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==2);

    Logging_Config::unregister_logger(logger);	

    // checking for the trace of the dropped exception
    Iter_t current(records.begin()); // skip LOG_OPEN
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN); 
    ++current;
    TEST_ASSERT(current->type()==TYPE_LOG_CLOSE); 
    TEST_ASSERT(current+1==records.end());  
    delete logger;
}


void returning(Test_Data & data)
{
    // never mind the leak
    Dummy_Logger * logger=new Dummy_Logger;
    Dummy_Logger::Records_t & records(logger->records());
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);	

    Dummy_Class_With_Invariance d;

    d.method_with_return(false,false);
    TEST_ASSERT(d.class_invariance_called()==2);
    
    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_return(true,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Invariance throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==1);

    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_return(false,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Invariance throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==2);

    Logging_Config::unregister_logger(logger);	

    // checking for the trace of the dropped exception
    Iter_t current(records.begin()); // skip LOG_OPEN
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN); 
    ++current;
    TEST_ASSERT(current->type()==TYPE_LOG_CLOSE); 
    TEST_ASSERT(current+1==records.end());  
    delete logger;
}


void throw_with_implicit_te(Test_Data & data)
{
    // never mind the leak
    Dummy_Logger * logger=new Dummy_Logger;
    Dummy_Logger::Records_t & records(logger->records());
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);	

    Dummy_Class_With_Invariance d;

    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_throw_implicit_te(false,false);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Method throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==2);
    
    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_throw_implicit_te(true,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Invariance throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==1);

    TEST_ASSERT(records.size()==1);

    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_throw_implicit_te(false,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Method throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==2);

    Logging_Config::unregister_logger(logger);	

    Iter_t current(records.begin());
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN); 
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_AUDIT,TYPE_TRACE,30,"dropped invariance exception "
			"EXCEPTION=\"Test_Exception\" WHAT=\"Invariance throwing\"");
    ++current;
    TEST_ASSERT(current->type()==TYPE_LOG_CLOSE); 
    TEST_ASSERT(current+1==records.end());  
    delete logger;
}


void throw_with_explicit_te(Test_Data & data)
{
    // never mind the leak
    Dummy_Logger * logger=new Dummy_Logger;
    Dummy_Logger::Records_t & records(logger->records());
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);	

    Dummy_Class_With_Invariance d;

    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_throw_explicit_te(false,false);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Method throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==2);
    
    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_throw_explicit_te(true,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Invariance throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==1);

    TEST_ASSERT(records.size()==1);

    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_throw_explicit_te(false,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Method throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==2);

    Logging_Config::unregister_logger(logger);	

    Iter_t current(records.begin());
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN); 
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_AUDIT,TYPE_TRACE,40,"dropped invariance exception "
			"EXCEPTION=\"Test_Exception\" WHAT=\"Invariance throwing\"");
    ++current;
    TEST_ASSERT(current->type()==TYPE_LOG_CLOSE); 
    TEST_ASSERT(current+1==records.end());  
    delete logger;
}


void throw_with_implicit_te_int_as_base(Test_Data & data)
{
    // never mind the leak
    Dummy_Logger * logger=new Dummy_Logger;
    Dummy_Logger::Records_t & records(logger->records());
    typedef Dummy_Logger::Records_t::iterator Iter_t;

    Logging_Config::register_logger(logger);	

    Dummy_Class_With_Invariance d;

    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_throw_implicit_te_int_as_base(false,false);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Method throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==2);
    
    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_throw_implicit_te_int_as_base(true,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Invariance throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==1);

    TEST_ASSERT(records.size()==1);

    TEST_THROWING_BLOCK_ENTER;
    TEST_NO_IMPLICIT_LOGGING_ENTER;
    d.method_with_throw_implicit_te_int_as_base(false,true);
    TEST_NO_IMPLICIT_LOGGING_EXIT;
    TEST_THROWING_BLOCK_EXIT1(Test_Exception,::std::string("Method throwing").c_str());
    TEST_ASSERT(d.class_invariance_called()==2);

    Logging_Config::unregister_logger(logger);	

    Iter_t current(records.begin());
    TEST_ASSERT(current->type()==TYPE_LOG_OPEN); 
    ++current;
    TEST_ASSERT_RECORD1(LEVEL_AUDIT,TYPE_TRACE,50,"dropped invariance exception");
    ++current;
    TEST_ASSERT(current->type()==TYPE_LOG_CLOSE); 
    TEST_ASSERT(current+1==records.end());  
    delete logger;
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&guard,LEVEL_PROD);
TEST_NORMAL_CASE(&macros,LEVEL_PROD);
TEST_NORMAL_CASE(&returning,LEVEL_PROD);
TEST_NORMAL_CASE(&throw_with_implicit_te,LEVEL_PROD);
TEST_NORMAL_CASE(&throw_with_explicit_te,LEVEL_PROD);
TEST_NORMAL_CASE(&throw_with_implicit_te_int_as_base,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;

// vim:ts=4:sw=4
