// BaseDisplay.cc for Blackbox - an X11 Window manager
// Copyright (c) 1997 - 1999 Brad Hughes (bhughes@tcac.net)
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

// stupid macros needed to access some functions in version 2 of the GNU C
// library

#ifndef   _GNU_SOURCE
#define   _GNU_SOURCE
#endif // _GNU_SOURCE

#include "../config.h"

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#ifdef    SHAPE
#  include <X11/extensions/shape.h>
#endif // SHAPE

#include "BaseDisplay.hh"
#include "LinkedList.hh"
#include "Timer.hh"

#ifdef    HAVE_FCNTL_H
#  include <fcntl.h>  
#endif // HAVE_FCNTL_H

#ifdef    HAVE_STDIO_H
#  include <stdio.h>
#endif // HAVE_STDIO_H

#ifdef    STDC_HEADERS
#  include <stdlib.h>
#  include <string.h>
#endif // STDC_HEADERS 

#ifdef    HAVE_UNISTD_H
#  include <sys/types.h>
#  include <unistd.h>
#endif // HAVE_UNISTD_H

#ifdef    HAVE_SYS_SELECT_H
#  include <sys/select.h>
#endif // HAVE_SYS_SELECT_H

#ifdef    HAVE_SIGNAL_H
#  include <signal.h>  
#endif // HAVE_SIGNAL_H

#ifdef    HAVE_SYS_SIGNAL_H
#  include <sys/signal.h>
#endif // HAVE_SYS_SIGNAL_H

#ifndef   SA_NODEFER
#  ifdef   SA_INTERRUPT
#    define SA_NODEFER SA_INTERRUPT
#  else // !SA_INTERRUPT
#    define SA_NODEFER (0)
#  endif // SA_INTERRUPT
#endif // SA_NODEFER

#ifdef    HAVE_SYS_WAIT_H
#  include <sys/types.h>
#  include <sys/wait.h>
#endif // HAVE_SYS_WAIT_H


// X error handler to handle any and all X errors while the application is
// running
static Bool internal_error = False;
BaseDisplay *base_display;

static int handleXErrors(Display *d, XErrorEvent *e) {
  char errtxt[128];
  XGetErrorText(d, e->error_code, errtxt, 128);
  fprintf(stderr, "%s:  X error: %s(%d) opcodes %d/%d\n"
                  "  resource 0x%lx\n",
          base_display->getApplicationName(), errtxt, e->error_code,
          e->request_code, e->minor_code, e->resourceid);

  if (internal_error) abort();

  return(False);
}


// signal handler to allow for proper and gentle shutdown

#ifndef   HAVE_SIGACTION
static RETSIGTYPE signalhandler(int sig) {
#else //  HAVE_SIGACTION
static void signalhandler(int sig) {
#endif // HAVE_SIGACTION

  static int re_enter = 0;

  switch (sig) {
  case SIGCHLD:
    int status;
    waitpid(-1, &status, WNOHANG | WUNTRACED);

#ifndef   HAVE_SIGACTION
    // assume broken, braindead sysv signal semantics
    signal(SIGCHLD, (RETSIGTYPE (*)(int)) signalhandler);
#endif // HAVE_SIGACTION

    break;

  default:
    if (base_display->handleSignal(sig)) {

#ifndef   HAVE_SIGACTION
      // assume broken, braindead sysv signal semantics
      signal(sig, (RETSIGTYPE (*)(int)) signalhandler);
#endif // HAVE_SIGACTION

      return;
    }

    fprintf(stderr, "%s:  signal %d caught\n",
            base_display->getApplicationName(), sig);

    if (! base_display->isStartup() && ! re_enter) {
      internal_error = True;

      re_enter = 1;
      fprintf(stderr, "shutting down\n");
      base_display->shutdown();
    }

    if (sig != SIGTERM && sig != SIGINT) {
      fprintf(stderr, "aborting... dumping core\n");
      abort();
    }

    exit(0);

    break;
  }
}


BaseDisplay::BaseDisplay(char *app_name, char *dpy_name) {
  application_name = app_name;

  _startup = True;
  _shutdown = False;
  server_grabs = 0;

  ::base_display = this;

#ifdef    HAVE_SIGACTION
  struct sigaction action;

  action.sa_handler = signalhandler;    
  action.sa_mask = sigset_t();
  action.sa_flags = SA_NOCLDSTOP | SA_NODEFER;
    
  sigaction(SIGSEGV, &action, NULL);
  sigaction(SIGFPE, &action, NULL);
  sigaction(SIGTERM, &action, NULL);
  sigaction(SIGINT, &action, NULL);
  sigaction(SIGCHLD, &action, NULL);
  sigaction(SIGHUP, &action, NULL);
  sigaction(SIGUSR1, &action, NULL);
  sigaction(SIGUSR2, &action, NULL);
#else // !HAVE_SIGACTION
  signal(SIGSEGV, (RETSIGTYPE (*)(int)) signalhandler);
  signal(SIGFPE, (RETSIGTYPE (*)(int)) signalhandler);
  signal(SIGTERM, (RETSIGTYPE (*)(int)) signalhandler);
  signal(SIGINT, (RETSIGTYPE (*)(int)) signalhandler);
  signal(SIGUSR1, (RETSIGTYPE (*)(int)) signalhandler);
  signal(SIGUSR2, (RETSIGTYPE (*)(int)) signalhandler);
  signal(SIGHUP, (RETSIGTYPE (*)(int)) signalhandler);
  signal(SIGCHLD, (RETSIGTYPE (*)(int)) signalhandler);
#endif // HAVE_SIGACTION

  if (! (display = XOpenDisplay(dpy_name))) {
    fprintf(stderr,
            "BaseDisplay::BaseDisplay: connection to X server failed.\n");
    ::exit(2);
  } else if (fcntl(ConnectionNumber(display), F_SETFD, 1) == -1) {
    fprintf(stderr,
            "BaseDisplay::BaseDisplay: couldn't mark display connection "
            "as close-on-exec\n");
    ::exit(2);
  }

  number_of_screens = ScreenCount(display);
  display_name = XDisplayName(dpy_name);

#ifdef    SHAPE
  shape.extensions = XShapeQueryExtension(display, &shape.event_basep,
                                          &shape.error_basep);
#else // !SHAPE
  shape.extensions = False;
#endif // SHAPE
  
  xa_wm_colormap_windows =
    XInternAtom(display, "WM_COLORMAP_WINDOWS", False);
  xa_wm_protocols = XInternAtom(display, "WM_PROTOCOLS", False);
  xa_wm_state = XInternAtom(display, "WM_STATE", False);
  xa_wm_change_state = XInternAtom(display, "WM_CHANGE_STATE", False);
  xa_wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", False);
  xa_wm_take_focus = XInternAtom(display, "WM_TAKE_FOCUS", False);
  motif_wm_hints = XInternAtom(display, "_MOTIF_WM_HINTS", False);

  net_hints = XInternAtom(display, "_NET_HINTS", False);
  net_attributes = XInternAtom(display, "_NET_ATTRIBUTES", False);
  net_change_attributes =
    XInternAtom(display, "_NET_CHANGE_ATTRIBUTES", False);

  net_structure_messages =
    XInternAtom(display, "_NET_STRUCTURE_MESSAGES", False);
  net_notify_startup =
    XInternAtom(display, "_NET_NOTIFY_STARTUP", False);
  net_notify_window_add =
    XInternAtom(display, "_NET_NOTIFY_WINDOW_ADD", False);
  net_notify_window_del =
    XInternAtom(display, "_NET_NOTIFY_WINDOW_DEL", False);
  net_notify_current_workspace =
    XInternAtom(display, "_NET_NOTIFY_CURRENT_WORKSPACE", False);
  net_notify_workspace_count =
    XInternAtom(display, "_NET_NOTIFY_WORKSPACE_COUNT", False);
  net_notify_window_focus =
    XInternAtom(display, "_NET_NOTIFY_WINDOW_FOCUS", False);
  net_notify_window_raise =
    XInternAtom(display, "_NET_NOTIFY_WINDOW_RAISE", False);
  net_notify_window_lower =
    XInternAtom(display, "_NET_NOTIFY_WINDOW_LOWER", False);

  net_change_workspace= XInternAtom(display, "_NET_CHANGE_WORKSPACE", False);
  net_change_window_focus =
    XInternAtom(display, "_NET_CHANGE_WINDOW_FOCUS", False);
  net_cycle_window_focus =
    XInternAtom(display, "_NET_CYCLE_WINDOW_FOCUS", False); 
  
  cursor.session = XCreateFontCursor(display, XC_left_ptr);
  cursor.move = XCreateFontCursor(display, XC_fleur);
  cursor.ll_angle = XCreateFontCursor(display, XC_ll_angle);
  cursor.lr_angle = XCreateFontCursor(display, XC_lr_angle);

  XSetErrorHandler((XErrorHandler) handleXErrors);

  timerList = new LinkedList<BTimer>;

  screenInfoList = new LinkedList<ScreenInfo>;
  int i;
  for (i = 0; i < number_of_screens; i++) {
    ScreenInfo *screeninfo = new ScreenInfo(this, i);
    screenInfoList->insert(screeninfo);
  }
}


BaseDisplay::~BaseDisplay(void) {
  while (screenInfoList->count())
    delete screenInfoList->remove(0);
  delete screenInfoList;

  // we don't create the BTimers, we don't delete them
  while (timerList->count())
    timerList->remove(0);
  delete timerList;

  XCloseDisplay(display);
}


void BaseDisplay::eventLoop(void) {
  run();

  int xfd = ConnectionNumber(display);

  while ((! _shutdown) && (! internal_error)) {
    if (XPending(display)) {
      XEvent e;
      XNextEvent(display, &e);
      process_event(&e);
    } else {
      fd_set rfds;
      timeval now, tm, *timeout = (timeval *) 0;

      FD_ZERO(&rfds);
      FD_SET(xfd, &rfds);

      if (timerList->count()) {
        gettimeofday(&now, 0);

        tm.tv_sec = tm.tv_usec = 0l;

        BTimer *timer = timerList->first();

        tm.tv_sec = timer->getStartTime().tv_sec +
          timer->getTimeout().tv_sec - now.tv_sec;
        tm.tv_usec = timer->getStartTime().tv_usec +
          timer->getTimeout().tv_usec - now.tv_usec;

        while (tm.tv_usec >= 1000000) {
          tm.tv_sec++;
          tm.tv_usec -= 1000000;
        }

        while (tm.tv_usec < 0) {
          if (tm.tv_sec > 0) {
            tm.tv_sec--;
            tm.tv_usec += 1000000;
          } else {
            tm.tv_usec = 0;
            break;
          }
        }

        timeout = &tm;
      }

      select(xfd + 1, &rfds, 0, 0, timeout);

      // check for timer timeout
      gettimeofday(&now, 0);

      LinkedListIterator<BTimer> it(timerList);
      for(; it.current(); it++) {
        tm.tv_sec = it.current()->getStartTime().tv_sec +
          it.current()->getTimeout().tv_sec;
        tm.tv_usec = it.current()->getStartTime().tv_usec +
          it.current()->getTimeout().tv_usec;

        if ((now.tv_sec < tm.tv_sec) ||
            (now.tv_sec == tm.tv_sec && now.tv_usec < tm.tv_usec))
          break;

        it.current()->fireTimeout();
  
        // restart the current timer so that the start time is updated
        if (! it.current()->doOnce()) it.current()->start();
        else it.current()->stop();
      }
    }
  }
}


const Bool BaseDisplay::validateWindow(Window window) {  
  XEvent event;
  if (XCheckTypedWindowEvent(display, window, DestroyNotify, &event)) {
    XPutBackEvent(display, &event);

    return False;
  } 

  return True;
}


void BaseDisplay::grab(void) {
  if (! server_grabs++)
    XGrabServer(display);
}


void BaseDisplay::ungrab(void) {
  if (! --server_grabs)
    XUngrabServer(display);

  if (server_grabs < 0) server_grabs = 0;
}


void BaseDisplay::addTimer(BTimer *timer) {
  if (! timer) return;

  LinkedListIterator<BTimer> it(timerList);
  int index = 0;
  for (; it.current(); it++, index++)
    if ((it.current()->getTimeout().tv_sec > timer->getTimeout().tv_sec) ||
        ((it.current()->getTimeout().tv_sec == timer->getTimeout().tv_sec) &&
         (it.current()->getTimeout().tv_usec >= timer->getTimeout().tv_usec)))
      break;

  timerList->insert(timer, index);
}


void BaseDisplay::removeTimer(BTimer *timer) {
  timerList->remove(timer);
}


ScreenInfo::ScreenInfo(BaseDisplay *d, int num) {
  basedisplay = d;
  screen_number = num;

  root_window = RootWindow(basedisplay->getXDisplay(), screen_number);
  visual = DefaultVisual(basedisplay->getXDisplay(), screen_number);
  depth = DefaultDepth(basedisplay->getXDisplay(), screen_number);

  width =
    WidthOfScreen(ScreenOfDisplay(basedisplay->getXDisplay(), screen_number));
  height =
    HeightOfScreen(ScreenOfDisplay(basedisplay->getXDisplay(), screen_number));
}

