/* BenzenoidGenerator.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.generators.graphs.chemical;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import org.grinvin.factories.FactoryParameterException;
import org.grinvin.generators.GeneratorException;
import org.grinvin.generators.graphs.AbstractGraphGenerator;
import org.grinvin.generators.graphs.AbstractGraphGeneratorInstance;
import org.grinvin.generators.graphs.GraphGeneratorInstance;
import org.grinvin.generators.graphs.GraphGeneratorSink;
import org.grinvin.graphs.Annotation;
import org.grinvin.graphs.Embedding;
import org.grinvin.graphs.Graph;
import org.grinvin.graphs.GraphBundle;
import org.grinvin.graphs.Vertex;
import org.grinvin.gui.icons.ChemicalIconFactory;
import org.grinvin.gui.icons.EdgesOnlyIconFactory;
import org.grinvin.gui.icons.GraphIconFactory;
import org.grinvin.preferences.GrinvinPreferences;

/**
 * Generator for benzenoids.
 */
public class BenzenoidGenerator extends AbstractGraphGenerator {
    
    //
    @Override protected void checkParameters(Object[] values) throws FactoryParameterException {
        super.checkParameters(values);
        int hexMinCount = (Integer)values[0];
        int hexMaxCount = (Integer)values[1];
        int maxCount = (Integer)values[2];
        
        if (hexMinCount <= 0 || hexMaxCount < hexMinCount)
            throw new FactoryParameterException("Invalid number of hexagons");
        if (maxCount < -1)
            throw new FactoryParameterException("Invalid graph count");
    }
    
    //
    public GraphGeneratorInstance createInstance() {
        return new BenzenoidGeneratorInstance(this,
                ((Integer)values[0]).intValue(),
                ((Integer)values[1]).intValue(),
                ((Integer)values[2]).intValue(),
                ((Boolean)values[3]).booleanValue()
                );
    }
    
    private static final int[][] VECTORS = {{1,0},{1,1},{0,1},{-1,0},{-1,-1},{0,-1}}; //run through a hexagon clockwise
    
    //
    private class BenzenoidGeneratorInstance extends AbstractGraphGeneratorInstance {
        
        // minimum number of hexagons
        private final int hexMinCount;
        
        // maximum number of hexagons
        private final int hexMaxCount;
        
        // maximum number of graphs (-1 is as many as possible)
        private final int maxCount;
        
        // show H-atoms
        private final boolean showH;
        
        // icon factory
        private final GraphIconFactory gif;
        
        //
        public BenzenoidGeneratorInstance(AbstractGraphGenerator agg,
                int hexMinCount, int hexMaxCount, int maxCount, boolean showH) {
            super(agg);
            this.hexMinCount = hexMinCount;
            this.hexMaxCount = hexMaxCount;
            this.maxCount = maxCount;
            this.showH = showH;
            if(showH)
                gif = ChemicalIconFactory.getInstance();
            else
                gif = EdgesOnlyIconFactory.getInstance();
        }
        
        /**
         * Generate benzenoids.
         */
        public void generate(GraphGeneratorSink sink) throws GeneratorException {
            int graphCount = 1;
            File path = GrinvinPreferences.getInstance().getProgramPath("benzene");
            for(int hexCount = hexMinCount; hexCount <= hexMaxCount; hexCount++){
                try {
                    ProcessBuilder pb = new ProcessBuilder(path.toString(), "" + hexCount, "b", "p");
                    Process generator = pb.start();
                    in = new BufferedInputStream(generator.getInputStream());
                } catch (IOException ex) {
                    throw new GeneratorException("The program benzene was not found.", ex);
                }
                
                int[] planarcode = getNextGraph();
                
                while(planarcode != null && (maxCount==-1 || graphCount <= maxCount)) {
                    GraphBundle gb = sink.createGraphBundle();
                    Graph graph = gb.createGraph();
                    Embedding embedding = gb.createEmbedding();
                    Annotation annotation = gb.createAnnotation();
                    gb.setGraphIconFactory(gif);
                    embedGraph(planarcode, graph, embedding, annotation, showH);
                    sink.receiveGraphBundle(gb);
                    graphCount++;
                    planarcode = getNextGraph();
                }
                
            }
            
            
        }
        
        private void embedGraph(int[] planarcode, Graph graph, Embedding embedding, Annotation annotation, boolean showH){
            int nrOfVertices = planarcode[1];
            int[][] benzene = new int[nrOfVertices][];
            int pos = 2;
            int firstDegreeThree = -1;
            int nrOfH = 0;
            for(int vertex=0; vertex < nrOfVertices; vertex++){
                if(planarcode[pos+3]==0){ //degree 3
                    if(firstDegreeThree == -1)
                        firstDegreeThree = vertex;
                    benzene[vertex] = new int[3];
                    benzene[vertex][0] = planarcode[pos++]-1;
                    benzene[vertex][1] = planarcode[pos++]-1;
                    benzene[vertex][2] = planarcode[pos++]-1;
                } else { //degree 2
                    nrOfH++;
                    benzene[vertex] = new int[2];
                    benzene[vertex][0] = planarcode[pos++]-1;
                    benzene[vertex][1] = planarcode[pos++]-1;
                }
                pos++;
            }
            int[][] coordinates;
            if(showH)
                coordinates = new int[nrOfVertices + nrOfH][];
            else
                coordinates = new int[nrOfVertices][];
            coordinates[firstDegreeThree] = new int[] {0,0};
            coordinates[benzene[firstDegreeThree][0]] = new int[] {1,0};
            coordinates[benzene[firstDegreeThree][1]] = new int[] {0,1};
            coordinates[benzene[firstDegreeThree][2]] = new int[] {-1,-1};
            
            giveCoordinates(firstDegreeThree, benzene[firstDegreeThree][0], 0, coordinates, benzene);
            giveCoordinates(firstDegreeThree, benzene[firstDegreeThree][1], 2, coordinates, benzene);
            giveCoordinates(firstDegreeThree, benzene[firstDegreeThree][2], 4, coordinates, benzene);
            
            Vertex[] vertices;
            if(showH)
                vertices = new Vertex[benzene.length + nrOfH];
            else
                vertices = new Vertex[benzene.length];
            for(int j=0; j < benzene.length; j++) {
                vertices[j] = graph.addNewVertex();
                annotation.setAnnotation(vertices[j], "C");
            }
            if(showH){
                for(int j=0; j < nrOfH; j++) {
                    vertices[nrOfVertices + j] = graph.addNewVertex();
                    annotation.setAnnotation(vertices[nrOfVertices + j], "H");
                }
            }
            for(int j=0; j < benzene.length; j++){
                for(int i = 0; i<benzene[j].length; i++)
                    if(graph.getVertex(j).getIndex() < graph.getVertex(benzene[j][i]).getIndex())
                        // maybe more efficient to just check: graph.getVertex(j) < graph.getVertex(benzene[j][i])
                        graph.addNewEdge(graph.getVertex(j),graph.getVertex(benzene[j][i]));
            }
            
            if(showH){
                //calculate coordinats for H-atoms
                for(int j=0; j < benzene.length; j++){
                    coordinates[j][0]=coordinates[j][0]*3;
                    coordinates[j][1]=coordinates[j][1]*3;
                }
                int h = 0;
                for(int j=0; j < benzene.length; j++){
                    if(benzene[j].length==2) {
                        graph.addNewEdge(graph.getVertex(j),graph.getVertex(nrOfVertices + h));
                        coordinates[nrOfVertices + h] = new int[2];
                        coordinates[nrOfVertices + h][0] = coordinates[j][0] + (2*coordinates[j][0]-coordinates[benzene[j][0]][0]-coordinates[benzene[j][1]][0])/3;
                        coordinates[nrOfVertices + h][1] = coordinates[j][1] + (2*coordinates[j][1]-coordinates[benzene[j][0]][1]-coordinates[benzene[j][1]][1])/3;
                        h++;
                    }
                }
            }
            
            //TODO: find correct rotation for display
            int order = nrOfVertices;
            if(showH)
                order += nrOfH;
            if (embedding != null)
                embedding.setDimension(2);
            double[][] realCoordinates = new double[order][2];
            double minX = Double.POSITIVE_INFINITY;
            double maxX = Double.NEGATIVE_INFINITY;
            double minY = Double.POSITIVE_INFINITY;
            double maxY = Double.NEGATIVE_INFINITY;
            for(int i=0; i<order; i++){
                realCoordinates[i][0]=coordinates[i][1]*Math.cos(Math.PI/6);
                if(realCoordinates[i][0]<minX)
                    minX=realCoordinates[i][0];
                if(realCoordinates[i][0]>maxX)
                    maxX=realCoordinates[i][0];
                realCoordinates[i][1]=coordinates[i][0] - coordinates[i][1]*Math.sin(Math.PI/6);
                if(realCoordinates[i][1]<minY)
                    minY=realCoordinates[i][1];
                if(realCoordinates[i][1]>maxY)
                    maxY=realCoordinates[i][1];
            }
            double transX = - (minX+maxX)/2;
            double transY = - (minY+maxY)/2;
            double scale = (maxX - minX) > (maxY - minY) ? 2/(maxX - minX) : 2/(maxY - minY);
            for(int i=0; i<order; i++){
                realCoordinates[i][0]=(realCoordinates[i][0]+transX)*scale;
                realCoordinates[i][1]=(realCoordinates[i][1]+transY)*scale;
                //System.out.println(Arrays.toString(c));
                if (embedding != null)
                    embedding.setCoordinates(vertices[i],realCoordinates[i]);
            }
        }
        
        private InputStream in;
        
        private int[] getNextGraph() throws GeneratorException {
            int planarcode[] = null;
            try {
                int c = in.read();
                //debug
                //InputStream errorIn = generator.getErrorStream();
                //int error = errorIn.read();
                //while(error!=-1){
                //    System.out.print((char)error);
                //    error = errorIn.read();
                //}
                
                if(c==-1)
                    return null; // EOF
                else if(c==0){
                    throw new GeneratorException("Graph is too big to be translated from planar code.");
                } else {
                    int numberOfZeroes=0;
                    int numberOfVertices = c;
                    int pos=1;
                    if(c=='>'){
                        //possibly a header
                        int c1 = in.read();
                        if(c1==0)
                            numberOfZeroes++;
                        int c2 = in.read();
                        if(c2==0)
                            numberOfZeroes++;
                        if(c1=='>' && c2=='p') {
                            //definitely a header
                            while(c!='<')
                                c=in.read();
                            c=in.read();
                            if(c!='<')
                                return null; // error in header
                            c=in.read();
                            if(!(c>0))
                                return null; //error in file or EOF
                            numberOfVertices = c;
                            planarcode = new int[(numberOfVertices*numberOfVertices) + 1];
                            planarcode[pos++]=c;
                        } else {
                            //no header
                            planarcode = new int[(numberOfVertices*numberOfVertices) + 1];
                            planarcode[pos++]=c;
                            planarcode[pos++]=c1;
                            planarcode[pos++]=c2;
                        }
                    } else {
                        planarcode = new int[(numberOfVertices*numberOfVertices) + 1];
                        planarcode[pos++]=c;
                    }
                    while(numberOfZeroes < numberOfVertices){
                        c=in.read();
                        if(c==0)
                            numberOfZeroes++;
                        planarcode[pos++]=c;
                    }
                    planarcode[0]=pos-1;
                    return planarcode;
                }
            } catch (IOException e) {
                return null;
            }
        }
        
        
        //vertex[previous] has degree 2
        @SuppressWarnings("PMD.AvoidReassigningParameters")
        private boolean checkClockwise(int previouspreviousVertex, int previousVertex, int previousStep, int[][] coordinates, int[][] benzene) {
            int step = (previousStep+1)%6;
            int[] tempCoords = addVector(coordinates[previousVertex], step);
            int vertex;
            if(benzene[previousVertex][0]==previouspreviousVertex)
                vertex=benzene[previousVertex][1];
            else
                vertex = benzene[previousVertex][0];
            int nodes = 2; //make sure we only check a hexagon
            while(coordinates[vertex]==null && nodes < 6){
                previouspreviousVertex = previousVertex;
                previousVertex = vertex;
                int previousIndex = -1;
                for(int i=0; i < benzene[previousVertex].length; i++)
                    if(benzene[previousVertex][i]==previouspreviousVertex)
                        previousIndex = i;
                vertex = benzene[previousVertex][(previousIndex-1+benzene[previousVertex].length)%benzene[previousVertex].length];
                step = (step + 1)%6;
                tempCoords = addVector(tempCoords, step);
                nodes++;
            }
            
            return (coordinates[vertex] != null) && (coordinates[vertex][0] == tempCoords[0]) && (coordinates[vertex][1] == tempCoords[1]);
        }
        
        private int[] addVector(int[] vector, int vectorToAdd){
            int[] newVector = new int[2];
            newVector[0] = vector[0] + VECTORS[vectorToAdd][0];
            newVector[1] = vector[1] + VECTORS[vectorToAdd][1];
            return newVector;
        }
        
        private void giveCoordinates(int previousVertex, int vertex, int step, int[][] coordinates, int[][] benzene){
            coordinates[vertex] = addVector(coordinates[previousVertex], step);
            if(benzene[vertex].length == 2) {
                int index = (benzene[vertex][0]==previousVertex) ? 1 : 0;
                if(coordinates[benzene[vertex][index]]==null)
                    if(this.checkClockwise(previousVertex, vertex, step, coordinates, benzene))
                        giveCoordinates(vertex, benzene[vertex][index], (step+1)%6, coordinates, benzene);
                    else
                        giveCoordinates(vertex, benzene[vertex][index], (step-1+6)%6, coordinates, benzene);
            } else {
                int previousIndex = -1;
                for(int i=0; i < 3; i++)
                    if(benzene[vertex][i]==previousVertex)
                        previousIndex = i;
                if(coordinates[benzene[vertex][(previousIndex-1+3)%3]]==null)
                    giveCoordinates(vertex, benzene[vertex][(previousIndex-1+3)%3], (step+1)%6, coordinates, benzene);
                if(coordinates[benzene[vertex][(previousIndex+1)%3]]==null)
                    giveCoordinates(vertex, benzene[vertex][(previousIndex+1)%3], (step-1+6)%6, coordinates, benzene);
            }
        }
    }
    
}
