/*
    libparted
    Copyright (C) 1998-2000  Andrew Clausen  <clausen@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "fat.h" 

/*
    returns the maximum number of clusters for a given filesystem type
*/
FatCluster
fat_max_cluster_count (FatType fat_type) {
	switch (fat_type) {
		case FAT_TYPE_FAT12: return 0xff0;
		case FAT_TYPE_FAT16: return 0xfff0;
		case FAT_TYPE_FAT32: return 0x0ffffff0;
	}
	return 0;
}

/*
    returns the minimum number of clusters for a given filesystem type
*/
FatCluster
fat_min_cluster_count (FatType fat_type) {
	switch (fat_type) {
		case FAT_TYPE_FAT12:
		case FAT_TYPE_FAT16:
			return fat_max_cluster_count (fat_type) / 2;

		case FAT_TYPE_FAT32: return 0xfff0;
	}
	return 0;
}


/* what is this supposed to be?  What drugs are M$ on?  (Can I have some? :-) */
PedSector
fat_min_reserved_sector_count (FatType fat_type)
{
	return (fat_type == FAT_TYPE_FAT32) ? 32 : 1;
}

int
fat_check_resize_geometry (PedFileSystem* fs, PedGeometry* geom,
			   FatCluster cluster_count)
{
	FatSpecific*	fs_info = FAT_SPECIFIC (fs);

	PED_ASSERT (geom != NULL, return 0);

	if (cluster_count < fat_min_cluster_count (FAT_TYPE_FAT16 /*fs_info->fat_type*/)
	    || cluster_count <
		(fs_info->fat->cluster_count - fs_info->fat->free_cluster_count)
		+ fs_info->total_dir_clusters + 1) {
		ped_exception_throw (PED_EXCEPTION_ERROR,
				     PED_EXCEPTION_CANCEL,
				     _("Partition length of %dk is too small."),
				     (int) geom->length / 2);
		return 0;
	}

	if (cluster_count > fat_max_cluster_count (FAT_TYPE_FAT32)) {
		ped_exception_throw (PED_EXCEPTION_FATAL,
				     PED_EXCEPTION_CANCEL,
				     _("Insane!  %d clusters!"),
				     (int) fs_info->cluster_count);
		return 0;
	}

	return 1;
}


/******************************************************************************/

/* DO NOT EDIT THIS ALGORITHM!
 * As far as I can tell, this is the same algorithm used by Microsoft to
 * calculate the size of the file allocaion tables, and the number of clusters.
 * I have not verified this by dissassembling Microsoft code - I came to this
 * conclusion by empirical analysis (i.e. trial and error - this was HORRIBLE).
 *
 * If you think this code makes no sense, then you are right.  I will restrain
 * the urge to inflict serious bodily harm on Microsoft people.
 */

static int
entries_per_sector (FatType fat_type)
{
	switch (fat_type) {
		case FAT_TYPE_FAT12:
			return 512 * 3 / 2;
		case FAT_TYPE_FAT16:
			return 512 / 2;
		case FAT_TYPE_FAT32:
			return 512 / 4;
	}
	return 0;
}

static int
calc_sizes (PedGeometry* geom, PedSector align, int cluster_size,
	    PedSector root_dir_sectors, FatCluster* out_cluster_count,
	    PedSector* out_fat_size, FatType fat_type)
{
	PedSector	data_fat_size;
	PedSector	fat_sectors;
	PedSector	cluster_sectors;
	FatCluster	cluster_count;
	int		i;

	PED_ASSERT (geom != NULL, return 0);
	PED_ASSERT (out_cluster_count != NULL, return 0);
	PED_ASSERT (out_fat_size != NULL, return 0);

	data_fat_size = geom->length - fat_min_reserved_sector_count (fat_type)
			- align;
	if (fat_type == FAT_TYPE_FAT16)
		data_fat_size -= root_dir_sectors;

	fat_sectors = 0;
	for (i = 0; i < 2; i++) {
		if (fat_type == FAT_TYPE_FAT32)
			cluster_sectors = data_fat_size - fat_sectors;
		else
			cluster_sectors = data_fat_size - 2 * fat_sectors;

		cluster_count = cluster_sectors / (cluster_size / 512);
		fat_sectors = div_round_up (cluster_count + 2,
					    entries_per_sector (fat_type));
	}

	cluster_sectors = data_fat_size - 2 * fat_sectors;
	cluster_count = cluster_sectors / (cluster_size / 512);

	if (cluster_count > fat_max_cluster_count (fat_type)
	    || cluster_count < fat_min_cluster_count (fat_type))
		return 0;

	*out_cluster_count = cluster_count;
	*out_fat_size = fat_sectors;

	return 1;
}

/* end of insane code */
/****************************************************************************/

int
fat_calc_sizes (PedGeometry* geom, PedSector align, FatType fat_type,
		PedSector root_dir_sectors, int* out_cluster_size,
		FatCluster* out_cluster_count, PedSector* out_fat_size)
{
	int	cluster_size = (fat_type == FAT_TYPE_FAT16) ? 512 : 4096;

	PED_ASSERT (geom != NULL, return 0);
	PED_ASSERT (out_cluster_size != NULL, return 0);
	PED_ASSERT (out_cluster_count != NULL, return 0);
	PED_ASSERT (out_fat_size != NULL, return 0);

	for (; cluster_size <= 65536; cluster_size *= 2) {
		if (calc_sizes (geom, align, cluster_size, root_dir_sectors,
			        out_cluster_count, out_fat_size, fat_type)) {
			*out_cluster_size = cluster_size;
			return 1;
		}
	}
	return 0;
}

/* Same as fat_calc_sizes, except it only attempts to match a particular
 * cluster size.  This is useful, because the FAT resizer can't change cluster
 * sizes.
 */
int
fat_calc_resize_sizes (PedGeometry* geom, PedSector align, int cluster_size,
		       PedSector root_dir_sectors,
		       FatCluster* out_cluster_count, PedSector* out_fat_size,
		       FatType* fat_type)
{
	PED_ASSERT (geom != NULL, return 0);
	PED_ASSERT (out_cluster_count != NULL, return 0);
	PED_ASSERT (out_fat_size != NULL, return 0);
	PED_ASSERT (fat_type != NULL, return 0);

/* try FAT16 first */
	if (calc_sizes (geom, align, cluster_size, root_dir_sectors,
			out_cluster_count, out_fat_size, FAT_TYPE_FAT16)) {
		*fat_type = FAT_TYPE_FAT16;
		return 1;
	}

/* try FAT32 */
	if (calc_sizes (geom, align, cluster_size, root_dir_sectors,
			out_cluster_count, out_fat_size, FAT_TYPE_FAT32)) {
		*fat_type = FAT_TYPE_FAT32;
		return 1;
	}

	return 0;
}

/*  Calculates the number of sectors needed to be added to cluster_offset,
    to make the cluster on the new filesystem match up with the ones
    on the old filesystem.
	However, some space is reserved by fat_calc_resize_sizes() and
    friends, to allow room for this space.  If too much of this space is left
    over, everyone will complain, so we have to be greedy, and use it all up...
 */
PedSector
fat_calc_align_sectors (PedFileSystem* new_fs, PedFileSystem* old_fs)
{
	FatSpecific*	old_fs_info = FAT_SPECIFIC (old_fs);
	FatSpecific*	new_fs_info = FAT_SPECIFIC (new_fs);
	PedSector	raw_old_meta_data_end;
	PedSector	new_meta_data_size;
	PedSector	min_new_meta_data_end;
	PedSector	new_data_size;
	PedSector	new_clusters_size;
	PedSector	align;

	new_meta_data_size
		= fat_min_reserved_sector_count (new_fs_info->fat_type) 
		  + new_fs_info->fat_sectors * 2;

	if (new_fs_info->fat_type == FAT_TYPE_FAT16)
		new_meta_data_size += new_fs_info->root_dir_sector_count;

	raw_old_meta_data_end = old_fs->geom->start
				 + old_fs_info->cluster_offset
				 + 2 * old_fs_info->cluster_sectors;

	min_new_meta_data_end = new_fs->geom->start + new_meta_data_size;

	if (raw_old_meta_data_end > min_new_meta_data_end)
		align = (raw_old_meta_data_end - min_new_meta_data_end)
			% new_fs_info->cluster_sectors;
	else
		align = (new_fs_info->cluster_sectors
		         - (   (min_new_meta_data_end - raw_old_meta_data_end)
				% new_fs_info->cluster_sectors   ))
			% new_fs_info->cluster_sectors;

	new_data_size = new_fs->geom->length - new_meta_data_size;
	new_clusters_size = new_fs_info->cluster_count
				* new_fs_info->cluster_sectors;

	while (new_clusters_size + align + new_fs_info->cluster_sectors
			<= new_data_size)
		align += new_fs_info->cluster_sectors;

	return align;
}

