/*
 **************************************************************************
 *
 * Boot-ROM-Code to load an operating system across a TCP/IP network.
 *
 * Module:  load.c
 * Purpose: Get boot image from server
 * Entries: load
 *
 **************************************************************************
 *
 * Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <general.h>
#include <memory.h>
#include <net.h>
#include <arpa.h>
#include <romlib.h>
#include "./bootpriv.h"
#include "./load.h"



/*
 **************************************************************************
 * 
 * Global variables:
 */
static unsigned long execptr;		/* Pointer to execution point	*/
static unsigned long highmem;		/* Top of conventional memory	*/
static unsigned long topmem;		/* Top of memory		*/
static unsigned long laststart;		/* Start of last memory block	*/
static unsigned long lastend;		/* End of last memory block	*/
static unsigned long nextput;		/* Address to put next block	*/
static unsigned long header;		/* Linear address of header	*/
static unsigned long remaining;		/* Remaining bytes in record	*/
static struct imghdr headbuf;		/* Temporary header buffer	*/
static struct loadrec *recp;		/* Pointer to load record	*/
static int recnum;			/* Number of current record	*/
#ifdef USEMODE
static short mode;			/* Loading mode			*/
#endif



/*
 **************************************************************************
 * 
 * Put TFTP buffer into memory, either into conventional or extended
 * memory.
 * This routine requires the following global variables to be setup
 * properly:  nextput, highmem, topmem, header
 */
static int put_mem(buf, len, conv_only)
char *buf;
int   len;
int   conv_only;
{
  register char *errmsg = "ERROR: Cannot write into ";
  int chunk_size;

  /* Check if destination pointer is valid */
  if (nextput < (LOWMEM << 4)) {
	printf("%slow mem\n", errmsg);
	return(FALSE);
  }
  if (nextput >= header && nextput < (header + sizeof(struct imghdr))) {
	printf("%sheader\n", errmsg);
	return(FALSE);
  }
  if (len == 0)
	return(TRUE);

  /* Put block into lower memory */
  chunk_size = len;
  if (nextput < highmem) {
	if ((nextput + len) >= highmem)
		len = (int)(highmem - nextput);
#ifdef LDDEBUG
	printf(" low=%lx, len=%d ", nextput, len);
#endif
	lmove(nextput, buf, len);
	chunk_size -= len;
	nextput += len;
	buf += len;
	if (chunk_size > 0)
		nextput = (EXTMEM << 4);
  } else if (nextput < (EXTMEM << 4)) {
	printf("%s98000-FFFFF\n", errmsg);
	return(FALSE);
  }

  /* Put block into high memory */
  if (chunk_size > 0) {
	if (conv_only || nextput >= topmem || nextput + chunk_size > topmem) {
		printf("%shigh mem\n", errmsg);
		return(FALSE);
	}
#ifdef LDDEBUG
	printf(" high=%lx, len=%d ", nextput, len);
#endif
	lmove(nextput, buf, chunk_size);
	nextput += chunk_size;
  }
  return(TRUE);
}



/*
 **************************************************************************
 * 
 * Decode record header and advance pointer to next record.
 */
static int next_record()
{
  register struct loadrec *rp = recp;
  int offset;

  printf("Image %d: Start=%lx, End=%lx\n", recnum++, laststart, lastend);
  if (rp == NULL)
	return(FALSE);

  nextput = rp->lr_addr;
  if (LR_IS_B0(rp->lr_flags)) {
	if (LR_IS_B1(rp->lr_flags)) {
		/*
		 * B0 = 1  &&  B1 = 1
		 * Load address is subtracted from the start of
		 * the last image loaded.
		 */
		if (nextput > laststart)
			nextput = 0;		/* will give error	*/
		else
			nextput = laststart - nextput;
	} else {
		/*
		 * B0 = 1  &&  B1 = 0
		 * Load address is added to the last byte of the memory area
		 * used by the previous image.
		 */
		nextput += lastend;
	}
  } else {
	if (LR_IS_B1(rp->lr_flags)) {
		/*
		 * B0 = 0  &&  B1 = 1
		 * Load address is subtracted from the last writable location
		 * in memory.
		 */
		if (nextput > topmem)
			nextput = 0;		/* will give error	*/
		else
			nextput = topmem - nextput;
	} else {
		/*
		 * B0 = 0  &&  B1 = 0
		 * Load address is used as is.
		 */
	}
  }
  laststart = nextput;
  lastend = laststart + rp->lr_mlen;
  remaining = rp->lr_ilen;

  /* Compute pointer to next load record */
  if (nextput == 0 ||
      (offset = LR_HDRLEN(rp->lr_flags)) < sizeof(struct loadrec)) {
	printf("ERROR: invalid load record\n");
	return(FALSE);
  }
  offset += LR_VENDLEN(rp->lr_flags);
  if (!LR_IS_EOF(rp->lr_flags)) {
	rp = (struct loadrec *)((unsigned char *)rp + offset);
	if (((int)rp - (int)(&headbuf)) >= sizeof(struct imghdr)) {
		printf("ERROR: load rec too large\n");
		return(FALSE);
	}
  } else
	rp = NULL;

  recp = rp;
  return(TRUE);
}



/*
 **************************************************************************
 * 
 * Decode the header information. We have three different types of headers:
 *   -  the header is smaller than SEGSIZE, or there are no magic ids: just
 *      print all ASCII characters onto the screen (mode 0).
 *   -  the header contains 0xaa/0x55 at the end of the block: this is the
 *      magic ID of a DOS boot sector, so load this sector into 0x07c00,
 *      and the rest starting at 0x10000, and then jump to 0x07c00 (mode 1).
 *   -  the header contains a magic word at the beginning of the block: this
 *      indicates a special header block with loading and starting information
 *      for the rest of the image (modes 2 and 3). If the RETURN flag is set
 *      in the header, this indicates that the loaded image will possibly
 *      use interrupts and services provided by the bootrom and also might
 *      return to the bootrom.
 */
static int decode_header(buf, len)
char *buf;
int   len;
{
  register struct imghdr *hp;
  int offset;

  /*
   * Copy the header into a temporary buffer, so that we don't have to
   * use far pointers.
   */
  if (len > sizeof(headbuf))
	len = sizeof(headbuf);
  hp = &headbuf;
  memcpy(hp, buf, len);

#ifdef USEMODE
  /* First determine the mode of the header */
  mode = 0;
  if (len == sizeof(struct imghdr)) {
	if (hp->ih_magic1 == IH_MAGIC1)
		mode = IH_IS_RETURN(hp->ih_flags) ? 3 : 2;
	else if (hp->ih_magic2 == IH_MAGIC2)
		mode = 1;
  }
#ifdef LDDEBUG
  printf("mode %d, ", mode);
#endif

  /* For mode 0, just printout this block. */
  if (mode == 0) {
	printf("\n\n%ls", hp, len);
	return(TRUE);
  }

  /*
   * For mode 1 setup all the parameters needed by put_mem. Use the
   * saved header block in order to restore it lateron at it's correct
   * position.
   */
  if (mode == 1) {
	nextput = LOWMEM << 4;
	execptr = BOOTBLOCK;
	header = ((unsigned long)getds() << 4) + (unsigned long)hp;
	return(TRUE);
  }

#else /* USEMODE */

  /* Check that the header is correct */
  if (hp->ih_magic1 != IH_MAGIC1) {
	printf("ERROR: invalid header\n");
	return(FALSE);
  }

#endif /* USEMODE */

  /*
   * With modes 2 and 3 first save the header into it's final place. Then
   * setup the first record pointer.
   */
  header = 0;
  nextput = laststart = far2long(hp->ih_locn);
#ifdef LDDEBUG
  printf("header=%lx, ", nextput);
#endif
  if (!put_mem((char *)hp, len, TRUE))
	return(FALSE);
  header = laststart;
  lastend = laststart + len;
  execptr = hp->ih_execute;
#ifdef LDDEBUG
  printf("headerend=%lx, ", lastend);
#endif

  offset = IH_HDRLEN(hp->ih_flags) + IH_VENDLEN(hp->ih_flags);
  if (offset < IH_HEADERSIZE ||
      offset >= sizeof(struct imghdr) - sizeof(struct loadrec) - 2) {
	printf("ERROR: Invalid header\n");
	return(FALSE);
  }
  recp = (struct loadrec *)((unsigned char *)hp + offset);
#ifdef LDDEBUG
  printf("record=%x\n", recp);
#endif
  printf("Block 1 ");
  if (!next_record())
	return(FALSE);

  return(TRUE);
}



/*
 **************************************************************************
 * 
 * Load boot image from server
 */
int load()
{
  int len, blocknum;
  char *inbuf;
  char *cp;

  /* Setup global values */
  recp = NULL;
  recnum = 0;

#ifndef NODISK
  /* Load a bootblock from disk if requested */
  if (!memcmp(boot_rec.bp_file, "/dev/", 5)) {
	loaddisk(&(boot_rec.bp_file[5]));
	printf("Cannot load bootblock from disk\n");
	return(FALSE);
  }
#endif

  /* Determine top of memory pointers */
  if ((highmem = convmem()) > (HIGHMEM << 4))
	highmem = HIGHMEM << 4;
  if ((topmem = extmem() + (EXTMEM << 4)) == (EXTMEM << 4))
	topmem = highmem;
#ifdef LDDEBUG
  printf("highmem=%lx  topmem=%lx\n", highmem, topmem);
#endif

  /* Read the image file header */
  if ((inbuf = tftp_open(ntohl(boot_rec.bp_siaddr),
			boot_rec.bp_file, sizeof(boot_rec.bp_file))) == NULL)
	return(FALSE);
  if ((len = tftp_get()) < 0 || !decode_header(inbuf, len))
	return(FALSE);

  /* Read all blocks of image file */
  blocknum = 1;
  while ((len = tftp_get()) > 0) {
	blocknum++;
#ifdef USEMODE
	if (mode == 0) {
		printf("%ls", inbuf, len);
		continue;
	}
#endif
	printf("\rBlock %d ", blocknum);
#ifdef USEMODE
	if (mode == 1) {
		if (!put_mem(inbuf, len, FALSE))
			return;
		continue;
	}
#endif
	/* Handle modes 2 and 3 loading. */
	cp = inbuf;
	if (len > remaining) {
		int xlen;

		xlen = len - (int)remaining;
		len = (int)remaining;
		if (!put_mem(inbuf, len, FALSE))
			return;
		cp += len;
		len = xlen;
		if (!next_record())
			break;
	}
	if (!put_mem(cp, len, FALSE))
		return(FALSE);
	remaining -= len;
	if (remaining == 0 && !next_record())
		break;
  }

  /* Check for errors */
  if (recp != 0)
	return(FALSE);

  if (remaining != 0) {
	printf("ERROR: unexpected end\n");
	return(FALSE);
  }

  /* Discard any leftover stuff. This will also close the TFTP connection */
  while (len == SEGSIZE)
	len = tftp_get();

  /* Call the loaded image */
#ifdef USEMODE
  if (mode > 0) {
	printf("\n\nStarting image...\n");
	return(exec_image(mode, execptr, long2far(header), &boot_rec));
  }
  return(0);
#else
  printf("\n\nStarting image...\n");
  return(exec_image(IH_IS_RETURN(headbuf.ih_flags) ? 3 : 2, execptr,
						long2far(header), &boot_rec));
#endif
}

