/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <pwd.h>
#include <unistd.h>
#include <sys/stat.h>

#include <glib.h>

#include "prefs.h"
#include "bezerk.h"
#include "util.h"
#include "debug.h"

Preference preferences[] = {
  { "bezerk.version",        NULL, NULL },
  { "font.advanced",         NULL, "0" },
  { "font.type",             NULL, "f" },
  { "font.size",             NULL, "14" },
  { "font.std_f_normal",     NULL, "-misc-fixed-medium-r-*-*-" },
  { "font.std_f_bold",       NULL, "-misc-fixed-bold-r-*-*-" },
  { "font.std_f_italic",     NULL, "-misc-fixed-bold-r-*-*-" },
  { "font.std_f_bolditalic", NULL, "-misc-fixed-bold-r-*-*-" },
  { "font.std_v_normal",     NULL, "-adobe-helvetica-medium-r-normal--" },
  { "font.std_v_bold",       NULL, "-adobe-helvetica-bold-r-normal--" },
  { "font.std_v_italic",     NULL, "-adobe-helvetica-medium-o-normal--" },
  { "font.std_v_bolditalic", NULL, "-adobe-helvetica-bold-o-normal--" },
  { "font.normal",           NULL, "-misc-fixed-medium-r-*-*-*-140-*-*-*-*-*-*" },
  { "font.bold",             NULL, "-misc-fixed-bold-r-*-*-*-140-*-*-*-*-*-*" },
  { "font.italic",           NULL, "-misc-fixed-bold-r-*-*-*-140-*-*-*-*-*-*" },
  { "font.bolditalic",       NULL, "-misc-fixed-bold-r-*-*-*-140-*-*-*-*-*-*" },
  { NULL,                    NULL, NULL },
};

char *prefs_dir;

void preference_set(char *key, char *value)
{
  int pref_num = 0;
  bs_function_enter();

  if ( !key ) {
    bs_function_leave();
    return;
  }

  while ( preferences[pref_num].key ) {
    if ( !strcmp(key, preferences[pref_num].key) ) {
      if ( preferences[pref_num].value ) {
	g_free(preferences[pref_num].value);
	preferences[pref_num].value = NULL;
      }
      if ( value ) {
	preferences[pref_num].value = g_strdup(value);
      }
      break;
    }
    pref_num++;
  }
	
  bs_function_leave();
  return;
}

char *preference_get(char *key)
{
  int pref_num = 0;

  bs_function_enter();

  if ( !key ) {
    bs_function_leave();
    return(NULL);
  }

  while ( preferences[pref_num].key ) {
    if ( !strcmp(key, preferences[pref_num].key) ) {
      if ( preferences[pref_num].value ) {
	bs_function_leave();
	return(preferences[pref_num].value);
      } else {
	bs_function_leave();
	return(preferences[pref_num].default_value);
      }
    }
    pref_num++;
  }

  bs_function_leave();
  return(NULL);
}

char *preference_get_default(char *key)
{
  int pref_num = 0;

  bs_function_enter();

  if ( !key ) {
    bs_function_leave();
    return(NULL);
  }

  while ( preferences[pref_num].key ) {
    if ( !strcmp(key, preferences[pref_num].key) ) {
      bs_function_leave();
      return(preferences[pref_num].default_value);
    }
    pref_num++;
  }

  bs_function_leave();
  return(NULL);
}

void preference_save()
{
  FILE *prefs_fd;
  char buff[BUFFLEN];
  int pref_num = 0;

  bs_function_enter();

  g_snprintf(buff, BUFFLEN, "%s/%s", prefs_dir, PREFS_FILE);
  if ( (prefs_fd = fopen(buff, "w")) == NULL) {
    bs_printf(3, "Error saving pres to %s", buff);
    bs_function_leave();
    return;
  }

  fputs("# Preference Settings\n", prefs_fd);
  fputs("# DO NOT EDIT - Automatically Generated\n", prefs_fd);

  while ( preferences[pref_num].key ) {
    if ( preferences[pref_num].value ) {
      fputs(preferences[pref_num].key, prefs_fd);
      fputs(" ", prefs_fd);
      fputs(preferences[pref_num].value, prefs_fd);
      fputs("\n", prefs_fd);
    }
    pref_num++;
  }

  fclose(prefs_fd);
  bs_function_leave();
  return;
}

gint preference_init()
{
  int uid;
  struct stat stat_buf;
  struct passwd *pwdent;
  FILE *prefs_fd;
  char buff[BUFFLEN];
  char *key, *value;

  bs_function_enter();

  /* Get the UID of the current user */
  uid = getuid();

 /* Get the password entry using the UID */
  if ((pwdent = getpwuid(uid)) == NULL) {
    g_error("Can't find password file entry");
  }

  g_snprintf(buff, BUFFLEN, "%s/.bezerk", pwdent->pw_dir);
  prefs_dir = g_strdup(buff);

  if (stat(prefs_dir, &stat_buf) == -1) {
    if (mkdir(prefs_dir, S_IRUSR | S_IWUSR | S_IXUSR) == -1) {
      g_error("Creating bezerk perfs directory");
    }
  } else if ( !S_ISDIR(stat_buf.st_mode) ) {
    g_error("~/.bezerk is not a directory");
  }

  g_snprintf(buff, BUFFLEN, "%s/%s", prefs_dir, PREFS_FILE);
  if ( (prefs_fd = fopen(buff, "r")) == NULL) {
    bs_function_leave();
    return(FALSE);
  }  

  while ( !feof(prefs_fd) ) {
    fgets(buff, BUFFLEN, prefs_fd);
    if ( *buff == '#' ) {
      continue;
    }
    if ( buff[strlen(buff)-1] == '\n' ) {
      buff[strlen(buff)-1] = '\0';
    }
    key = bs_strtok(buff, " ");
    value = bs_strtok(NULL, " ");
    if ( !key || !value ) {
      continue;
    }
    preference_set(key, value);
  }
  fclose(prefs_fd);

  bs_function_leave();
  return(TRUE);
}
