/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module invokes and manages the transmit process from malaga. */

/* includes =================================================================*/

#define _XOPEN_SOURCE
#define _XOPEN_SOURCE_EXTENDED 1
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <setjmp.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/wait.h>
#include <signal.h>
#include <pwd.h>
#include <fcntl.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "files.h"
#include "input.h"
#include "symbols.h"
#include "commands.h"
#include "value_parser.h"
#include "rule_type.h"
#include "rules.h"

#undef GLOBAL
#define GLOBAL

#include "transmit.h"

/* variables ================================================================*/

LOCAL string_t transmit_command_line; /* command line for transmit output */
LOCAL FILE *transmit_input_stream; /* to read data from transmit process */
LOCAL FILE *transmit_output_stream; /* to write data to transmit process */
LOCAL pid_t process_id; /* ID of transmit process */

/* functions ================================================================*/

LOCAL void start_transmit_process (void)
/* Start the Malaga transmit process by executing <transmit_command_line>
 * if it is not already running. */
{
  int to_transmit_fd[2];
  int from_transmit_fd[2];

  if (process_id != 0)
  {
    if (waitpid (process_id, 0, WNOHANG) == 0)
      return;

    process_id = 0;
    close_stream (&transmit_input_stream, NULL);
    close_stream (&transmit_output_stream, NULL);
  }

  if (pipe (to_transmit_fd) == -1 || pipe (from_transmit_fd) == -1)
    error ("can't create pipe to transmit process: %s", strerror (errno));
    
  if (transmit_command_line == NULL) 
    error ("missing transmit command line");

  switch (process_id = fork ())
  {
  case -1:
    error ("can't create transmit process: %s", strerror (errno));
    break;
    
  case 0:
  {
    string_t arguments, argument;
    string_t *args;
    int_t num_args, i;

    dup2 (to_transmit_fd[0], STDIN_FILENO);
    close (to_transmit_fd[0]);
    close (to_transmit_fd[1]);
    dup2 (from_transmit_fd[1], STDOUT_FILENO);
    close (from_transmit_fd[0]);
    close (from_transmit_fd[1]);

    /* Count arguments. */
    num_args = 0;
    arguments = transmit_command_line;
    while (*arguments != EOS)
    {
      argument = parse_word (&arguments);
      free_mem (&argument);
      num_args++;
    }

    /* Create argument vector. */
    args = new_vector (sizeof (string_t), num_args + 1);
    arguments = transmit_command_line;
    for (i = 0; i < num_args; i++)
      args[i] = parse_word (&arguments);
    args[i] = NULL;
    
    execvp (args[0], args);
    fprintf (stderr, "can't start transmit process \"%s\": %s\n", 
	     args[0], strerror (errno));
    exit (1);
  }
  default:
    close (to_transmit_fd[0]);
    transmit_output_stream = fdopen (to_transmit_fd[1], "w");
    close (from_transmit_fd[1]);
    transmit_input_stream = fdopen (from_transmit_fd[0], "r");
    if (transmit_input_stream == NULL || transmit_output_stream == NULL)
      error ("can't open data stream: %s", strerror (errno));
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void stop_transmit_process (void)
/* Stop the Malaga transmit process. */
{
  close_stream (&transmit_input_stream, NULL);
  close_stream (&transmit_output_stream, NULL);
  if (process_id != 0)
  {
    kill (process_id, SIGTERM);
    process_id = 0;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void local_transmit (void)
/* STACK EFFECTS: <value> -> <new_value>
 * Start the Malaga transmit process by executing <transmit_command_line>
 * if it is not already running.
 * Send <value>, in text format, to the transmit process and receive an answer
 * value in text format, convert it into internal format as <new_value>. */
{
  string_t input_line, value_string;

  start_transmit_process ();

  /* Send <argument> to transmit process. */
  value_string = value_to_readable (value_stack[--top], TRUE);
  fprintf (transmit_output_stream, "%s\n", value_string);
  fflush (transmit_output_stream);
  free_mem (&value_string);

  /* Read result and convert it to value. */
  input_line = read_line (transmit_input_stream);
  parse_value_string (input_line);
  free_mem (&input_line);
}

/*---------------------------------------------------------------------------*/

GLOBAL void init_transmit (void)
/* Initialise this module. */
{
  transmit = local_transmit;
}

/*---------------------------------------------------------------------------*/

GLOBAL void terminate_transmit (void)
/* Terminate this module. */
{
  stop_transmit_process ();
  transmit = NULL;
}

/*---------------------------------------------------------------------------*/

LOCAL void do_transmit_option (string_t arguments)
/* Set the command line to start the transmit process. */
{
  if (*arguments == EOS)
    printf ("transmit: \"%s\"\n", 
	    (transmit_command_line == NULL) ? "" : transmit_command_line);
  else
  {
    stop_transmit_process ();
    free_mem (&transmit_command_line);
    transmit_command_line = parse_word (&arguments);
  }

  parse_end (arguments);
}

GLOBAL command_t transmit_option =
{
  "transmit", do_transmit_option,
  "Usage: set transmit \"<transmit_command_line>\"\n"
  "Set the command line that is used to start the transmit process.\n"
};

/*---------------------------------------------------------------------------*/

LOCAL void do_transmit (string_t arguments)
/* Communicate with the transmit process. */
{
  string_t result;

  parse_value_string (arguments);
  local_transmit ();
  result = value_to_readable (value_stack[--top], FALSE);
  printf ("%s\n", result);
  free_mem (&result);
}

GLOBAL command_t transmit_command =
{
  "transmit", do_transmit,
  "Usage: transmit <value>\n"
  "Send <value> to the transmit process and display the result.\n"
};

/* end of file ==============================================================*/
