/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module supports scanning (lexical analysis) of malaga source files. */

/* constants ================================================================*/

/* The next token is read by "read_next_token".
 * Its value is stored in <next_token>.
 * A token that consists of one character has the code of that character.
 * A token value of EOF stands for end-of-file.
 * All other tokens are as follows: */
enum
{
  TOK_STRING = 256, /* a string */
  TOK_IDENT, /* an identifier */
  TOK_VARIABLE, /* a variable name */
  TOK_CONSTANT, /* a constant name */
  TOK_NUMBER, /* a floating number */ 
  TOK_ASSIGN, /* := */
  TOK_ASSIGN_PLUS, /* :=+ */
  TOK_ASSIGN_MINUS, /* :=- */
  TOK_ASSIGN_ASTERISK, /* :=* */
  TOK_ASSIGN_SLASH, /* :=/ */
  TOK_NOT_EQUAL, /* /= */
  TOK_NOT_CONGRUENT, /* /~ */
  TOK_ACCEPT,
  TOK_ALLO_RULE,
  TOK_AND,
  TOK_ASSERT,
  TOK_CHOOSE,
  TOK_COMBI_RULE,
  TOK_DEFINE,
  TOK_ELSE,
  TOK_ELSEIF,
  TOK_END,
  TOK_END_RULE,
  TOK_ERROR,
  TOK_FAIL,
  TOK_FOREACH,
  TOK_GREATER,
  TOK_GREATER_EQUAL,
  TOK_IF,
  TOK_IN,
  TOK_INCLUDE,
  TOK_INITIAL,
  TOK_INPUT_FILTER,
  TOK_LESS,
  TOK_LESS_EQUAL,
  TOK_MATCHES,
  TOK_NOT,
  TOK_OR,
  TOK_OUTPUT_FILTER,
  TOK_PARALLEL,
  TOK_PRUNING_RULE,
  TOK_REPEAT,
  TOK_REQUIRE,
  TOK_RESULT,
  TOK_RETURN,
  TOK_ROBUST_RULE,
  TOK_RULES,
  TOK_SUBRULE,
  TOK_THEN,
  TOK_WHILE
};

#define FIRST_KEYWORD TOK_ACCEPT
#define LAST_KEYWORD TOK_WHILE

#define NUMBER_OF_KEYWORDS (LAST_KEYWORD - FIRST_KEYWORD + 1)

/* variables ================================================================*/

GLOBAL int_t next_token; /* Next token that is to be consumed by parser. */

GLOBAL string_t token_name;
/* If <next_token> == TOK_IDENT, the name is in <token_name>. */

GLOBAL string_t token_string;
/* If <next_token> == TOK_STRING, the name is in <token_string>. */

GLOBAL double token_number;
/* If <next_token> == TOK_NUMBER, its content is in <token_number>. */

/* functions ================================================================*/

extern void set_scanner_input (string_t input);
/* Let the scanner use <input> as scanner input.
 * <input> must remain valid until the scanner has done its work. */

extern void begin_include (string_t file_name);
/* Open a new level of inclusion and read tokens from file <file_name>. */

extern void end_include (void);
/* Stop reading from current source stream and read from former file. */

extern void stop_scanner (void);
/* Stop the scanner in case of an emergency. */

extern string_t current_file_name (void);
/* Return the name of the file reading from or NULL. */

extern int_t current_line_number (void);
/* Return the line number where the last char has been read or -1. */

extern int_t current_column (void);
/* Return the column where the last char has been read or -1. */

extern void read_next_token (void);
/* Read the next token from current source into <next_token>.
 * If end of input stream is reached, return EOF. */

extern void test_token (int_t token);
/* Test if <token> is the next token. If it's not, report an error. */

extern void parse_token (int_t token);
/* Test if <token> is the next token and read next token. */

extern string_t token_as_text (int_t token);
/* Return <token> as a string readable for humans.
 * Note that the string is only valid until this function is called again. */

/* end of file ==============================================================*/
