/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module parses Malaga rule files. */

/* includes =================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "patterns.h"
#include "files.h"
#include "scanner.h"
#include "rule_type.h"
#include "rule_code.h"
#include "rule_symbols.h"

#ifdef HANGUL
#include "hangul.h"
#endif

#undef GLOBAL
#define GLOBAL

#include "rule_parser.h"

/* constants ================================================================*/

#define RULE_SET_MAX 50 /* maximum length of a rule set */
#define PARAMS_MAX 10 /* maximum number of parameters */

/* types ====================================================================*/

typedef struct PATCH_T /* a node in a list of instructions to patch */
{
  instr_t *instr; /* instruction to be patched */
  struct PATCH_T *next; /* next instruction to be patched */
} patch_t;

typedef struct /* a type for parsing conditions */
{
  instr_t *last_jump; /* pointer to last jump (not included below) */
  patch_t *true_jumps; /* a null-terminated list of pointers
			* to instrs that jump if condition is true */
  patch_t *false_jumps; /* a null-terminated list of pointers
			 * to instrs that jump if condition is false */
} condition_t;

typedef struct VAR_NODE_T /* a variable that is currently defined */
{
  string_t name; /* name of a variable */
  struct VAR_NODE_T *next; /* next variable */
} var_node_t;

typedef struct SCOPE_T /* list of variables that are defined in same scope */
{
  struct SCOPE_T *previous; /* previous scope */
  int_t start_index; /* index of the stack as the scope was opened */
  var_node_t *var_list; /* list of all variables defined in the scope */
} scope_t;

typedef enum {VALUE, CONST_VALUE, CONDITION} expr_type_t;
/* the type of a Malaga expression. */

/* variables ================================================================*/

LOCAL int_t last_statement_line;
LOCAL string_t last_statement_file_name;
/* file name and line number at the beginning of the last statement */

LOCAL rule_type_t rule_type; /* type of rule that is to be parsed */
LOCAL scope_t *current_scope = NULL; /* list of variables currently defined */

LOCAL struct /* the pattern variables defined in a "matches" condition */
{
  int_t number; /* number of pattern variables */
  string_t name[PATTERN_VAR_MAX]; /* names of the pattern variables */
} pattern_vars;

/* forward declarations =====================================================*/

FORWARD expr_type_t local_parse_value (condition_t *condition);
FORWARD expr_type_t parse_expression (condition_t *condition);
FORWARD void parse_statements (void);
FORWARD void parse_pattern (string_t *pattern);

/* support for source line associations =====================================*/

LOCAL void new_source_line (void)
/* Associate the next instructions with the current source line. */
{
  if (current_line_number () != last_statement_line
      || current_file_name () != last_statement_file_name)
  {
    src_line_t src_line;
    
    src_line.file = pool_index (code.string_pool, current_file_name ());
    src_line.line = current_line_number ();
    src_line.instr = code.next_instr_index;
    copy_to_pool (code.src_line_pool, &src_line, 1, NULL);

    last_statement_line = current_line_number ();
    last_statement_file_name = current_file_name ();
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void no_source_line (void)
/* Associate the next instructions generated with no source. */
{
  if (last_statement_line != -1 || last_statement_file_name != NULL) 
  {
    src_line_t src_line;
    
    src_line.file = -1;
    src_line.line = -1;
    src_line.instr = code.next_instr_index;
    copy_to_pool (code.src_line_pool, &src_line, 1, NULL);

    last_statement_line = -1;
    last_statement_file_name = NULL;
  }
}

/* support for parsing conditions ===========================================*/

LOCAL patch_t *concat_patches (patch_t *patches1, patch_t *patches2)
/* Concatenate <patches1> and <patches2>. */
{
  patch_t *patch;

  if (patches1 == NULL)
    return patches2;

  /* Find the end of <patches1>. */
  for (patch = patches1; patch->next != NULL; patch = patch->next)
    /*empty */;

  patch->next = patches2;
  return patches1;
}

/*---------------------------------------------------------------------------*/

LOCAL patch_t *add_to_patches (patch_t *patches, instr_t *instr)
/* Add instruction <instr> to <patches>. */
{
  patch_t *new_patch = new_mem (sizeof (patch_t));

  new_patch->instr = instr;
  new_patch->next = NULL;
  return concat_patches (patches, new_patch);
}

/*---------------------------------------------------------------------------*/

LOCAL void patch_instr (instr_t *instr_ptr, int_t info)
/* Patch *<instr_ptr> to have <info> as info. */
{
  if (info >= INSTR_INFO_MAX)
    error ("instruction value overflow");

  *instr_ptr = INSTR (OPCODE (*instr_ptr), info);
}

/*---------------------------------------------------------------------------*/

LOCAL void patch_jumps (patch_t **patches, int_t index)
/* Patch all instructions in <*patches> to jump to instruction no. <index>.
 * All instructions pointed to by <*patches> must be jump instructions.
 * The patch list is freed. */
{
  patch_t *patch, *next_patch;

  for (patch = *patches; patch != NULL; patch = next_patch)
  {
    next_patch = patch->next;
    patch_instr (patch->instr, index);
    free_mem (&patch);
  }
  *patches = NULL;
}

/*---------------------------------------------------------------------------*/

LOCAL void negate_jump (instr_t *jump)
/* If *<jump> is a conditional jump, negate its condition. */
{
  int_t opcode;

  opcode = OPCODE (*jump);
  switch (opcode) 
  {
  case INS_JUMP_IF_EQUAL:
    opcode = INS_JUMP_IF_NOT_EQUAL; 
    break;
  case INS_JUMP_IF_NOT_EQUAL:
    opcode = INS_JUMP_IF_EQUAL;
    break;
  case INS_JUMP_IF_CONGR:
    opcode = INS_JUMP_IF_NOT_CONGR;
    break;
  case INS_JUMP_IF_NOT_CONGR:
    opcode = INS_JUMP_IF_CONGR;
    break;
  case INS_JUMP_IF_IN:
    opcode = INS_JUMP_IF_NOT_IN;
    break;
  case INS_JUMP_IF_NOT_IN:
    opcode = INS_JUMP_IF_IN;
    break;
  case INS_JUMP_IF_LESS:
    opcode = INS_JUMP_IF_NOT_LESS;
    break;
  case INS_JUMP_IF_NOT_LESS:
    opcode = INS_JUMP_IF_LESS;
    break;
  case INS_JUMP_IF_GREATER:
    opcode = INS_JUMP_IF_NOT_GREATER;
    break;
  case INS_JUMP_IF_NOT_GREATER:
    opcode = INS_JUMP_IF_GREATER;
    break;
  case INS_JUMP_IF_NULL:
    opcode = INS_JUMP_IF_NOT_NULL; 
    break;
  case INS_JUMP_IF_NOT_NULL:
    opcode = INS_JUMP_IF_NULL;
    break;
  case INS_JUMP_IF_YES:
    opcode = INS_JUMP_IF_NO;
    break;
  case INS_JUMP_IF_NO:
    opcode = INS_JUMP_IF_YES;
    break;
  default:
    break;
  }

  *jump = INSTR (opcode, INSTR_INFO (*jump));
}

/* variable scopes ==========================================================*/

LOCAL void open_scope (void)
/* Open a new scope. */
{
  scope_t *scope;

  /* Allocate and initialise a new scope. */
  scope = new_mem (sizeof (scope_t));
  scope->start_index = code.stack_index;
  scope->var_list = NULL;
  
  scope->previous = current_scope;
  current_scope = scope;
}

/*---------------------------------------------------------------------------*/

LOCAL void close_scope (bool_t do_pop)
/* Close the current scope. Emit a pop instruction that resets
 * stack index to initial stack index if <do_pop> is TRUE. */
{
  scope_t *scope;
  var_node_t *var_node, *next_node; 

  scope = current_scope;
  DB_ASSERT (scope != NULL);

  /* Mark all variables in the current scope as undefined. */
  for (var_node = scope->var_list; var_node != NULL; var_node = next_node) 
  {
    next_node = var_node->next;
    undefine_variable (var_node->name);
    free_mem (&var_node);
  }

  /* See if we have to pop to reach the start index again. */
  if (do_pop && code.stack_index > scope->start_index)
  {
    no_source_line ();
    emit_instr (INS_POP, code.stack_index - scope->start_index);
  }

  /* Reset stack index. */
  code.stack_index = scope->start_index;
  current_scope = scope->previous;
  free_mem (&scope);
}

/*---------------------------------------------------------------------------*/

LOCAL void add_to_scope (string_t var_name)
/* Add another variable name to the current scope.
 * The variable name must be valid until the scope is closed. */ 
{
  scope_t *scope;
  var_node_t *var_node;

  scope = current_scope;
  DB_ASSERT (scope != NULL);

  /* Get a new <var_node> and enter it into the list. */
  var_node = new_mem (sizeof (var_node_t));
  var_node->name = var_name;
  var_node->next = scope->var_list;
  scope->var_list = var_node;
}

/*---------------------------------------------------------------------------*/

LOCAL void define_var_in_scope (string_t name, int_t var_index)
/* Define a variable and add it to the current scope. */
{
  string_t name_in_pool;

  name_in_pool = define_variable (name, var_index);
  add_to_scope (name_in_pool);
}

/* simple Malaga parse functions ============================================*/

LOCAL void parse_var_name (string_t *var_name)
/* Parse variable name and allocate memory to save its name in "*var_name". */
{
  test_token (TOK_VARIABLE);
  *var_name = new_string (token_name, NULL);
  read_next_token (); 
}

/*---------------------------------------------------------------------------*/

LOCAL int_t parse_rule_set_part (int_t rule_set[], int_t rule_set_size)
{
  int_t rule_number, i;
  rule_t *rule;

  test_token (TOK_IDENT);
  rule = find_rule_or_function (token_name, &rule_number);
  if (rule == NULL)
    error ("\"%s\" is a standard function", token_name);
  else if (rule->type == -1)
    rule->type = COMBI_RULE;
  else if (rule->type != COMBI_RULE && rule->type != END_RULE)
    error ("\"%s\" is not a combi_rule or end_rule", token_name);
  
  /* Check if we already have this rule in this rule set. */
  for (i = 0; i < rule_set_size; i++) 
  {
    if (rule_set[i] == rule_number)
      error ("rule \"%s\" twice in rule set", token_name);
  }
  
  read_next_token ();
  
  return rule_number;
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_rule_set (int_t *rule_set_index)
/* Parse a list of rule names and enter it into <code.rule_set_pool>.
 * Return the index of the rule set in *<rule_set_index>. */
{
  int_t rule_set[RULE_SET_MAX];
  int_t rule_set_size;
  bool_t has_parentheses;

  rule_set_size = 0;

  parse_token (TOK_RULES);
  
  has_parentheses = (next_token == '(');
  if (has_parentheses)
    parse_token ('(');

  while (TRUE)
  {
    while (TRUE)
    {
      if (rule_set_size >= RULE_SET_MAX - 2)
	error ("too many rules in rule set");
      
      rule_set[rule_set_size++] = parse_rule_set_part (rule_set, 
						       rule_set_size);
      if (next_token != ',')
	break;

      read_next_token ();
    }
    
    if (next_token != TOK_ELSE)
      break;
    read_next_token ();

    rule_set[rule_set_size++] = -2;
  }
  
  if (has_parentheses)
    parse_token (')');

  rule_set[rule_set_size++] = -1;
  copy_to_pool (code.rule_set_pool, rule_set, rule_set_size, rule_set_index);
}

/* functions to parse Malaga values =========================================*/

/* Functions to parse a value do not always emit instructions immediately.
 * Instead, they sometimes write instructions in a buffer,
 * so they can compute constant values at compile time.
 * Use the functions "parse_value", "parse_constant_value"
 * and "parse_condition" to get values or conditions. */

/*---------------------------------------------------------------------------*/

LOCAL expr_type_t parse_value (void)
/* Parse any value. */
{
  expr_type_t type;
  condition_t condition;

  pattern_vars.number = 0;
  type = local_parse_value (&condition);
  if (type == CONDITION)
  {
    if (pattern_vars.number != 0)
      error ("no variable definition allowed in values");

    negate_jump (condition.last_jump);
    patch_jumps (&condition.true_jumps, code.next_instr_index);
    emit_instr (INS_PUSH_SYMBOL, YES_SYMBOL);
    emit_instr (INS_JUMP, code.next_instr_index + 2);
    code.stack_index--;
    patch_instr (condition.last_jump, code.next_instr_index);
    patch_jumps (&condition.false_jumps, code.next_instr_index);
    emit_instr (INS_PUSH_SYMBOL, NO_SYMBOL);
    return VALUE;
  }
  else
  {
    flush_buffer ();
    return type;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void convert_to_condition (expr_type_t *type, condition_t *condition)
/* Make sure an expression is a condition. */
{
  if (*type != CONDITION)
  {
    condition->true_jumps = NULL;
    condition->false_jumps = NULL;
    condition->last_jump = emit_instr (INS_JUMP_IF_YES, 0);
    *type = CONDITION;
  }
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_subrule_call (string_t ident)
/* Parse a subrule call. The subrule name <ident> has already been parsed. */
{
  int_t num_params, rule_number;
  rule_t *rule;

  rule = find_rule_or_function (ident, &rule_number);
  if (rule != NULL)
  {
    if (rule->type == -1)
      rule->type = SUBRULE;
    else if (rule->type != SUBRULE)
      error ("\"%s\" is no subrule", ident);
  }

  parse_token ('(');

  num_params = 0;
  while (next_token != ')')
  {
    parse_value ();
    num_params++;
    
    if (next_token != ',')
      break;
    
    read_next_token ();
  }
  parse_token (')');

  if (rule == NULL) /* Call a standard function. */
  {
    if (num_params != 1)
      error ("function \"%s\" takes one parameter", ident);

    emit_instr (INS_STD_FUNCTION, rule_number);
  }
  else /* Call a real subrule. */
  {
    if (rule->num_params == -1)
      rule->num_params = num_params;
    else if (num_params != rule->num_params)
      error ("\"%s\" needs %ld parameters", ident, rule->num_params);
    
    code.stack_index -= num_params; /* subrule jump kills all parameters. */
    emit_instr (INS_JUMP_SUBRULE, rule_number);
  }
}

/*---------------------------------------------------------------------------*/

LOCAL expr_type_t parse_simple_value (condition_t *condition)
/* Parse a simple value or a condition in parentheses.
 * If <constant> is TRUE, the value must be constant.
 * Any jumps are returned in <condition>. */
{
  int_t i; /* number of values in list or record */
  expr_type_t type, elem_type; /* result type of expression */

  switch (next_token) 
  {
  case '<': /* Parse a list. */
    read_next_token ();
    type = CONST_VALUE;
    i = 0;
    if (next_token != '>') 
    {
      while (TRUE) 
      {
	elem_type = local_parse_value (condition);
	if (elem_type == CONDITION)
	  error ("no condition allowed in a list");
	else if (elem_type == VALUE)
	  type = VALUE;
	i++;
	
	if (next_token != ',')
	  break;
	read_next_token ();
      }
    }
    parse_token ('>');
    buffer_instr (INS_BUILD_LIST, i);
    return type;

  case '[': /* Parse a record. */
    read_next_token ();
    type = CONST_VALUE;
    i = 0;
    if (next_token != ']') 
    {
      while (TRUE)
      {
	elem_type = local_parse_value (condition);
	if (elem_type == CONDITION)
	  error ("no condition allowed in records");
	else if (elem_type == VALUE)
	  type = VALUE;
	parse_token (':');
	elem_type = local_parse_value (condition);
	if (elem_type == CONDITION)
	  error ("no condition allowed in records");
	else if (elem_type == VALUE)
	  type = VALUE;
	i++;
	
	if (next_token != ',')
	  break;
	read_next_token ();
      }
    }
    parse_token (']');
    buffer_instr (INS_BUILD_RECORD, i);
    return type;

  case TOK_IDENT:
  {
    string_t ident = new_string (token_name, NULL);
      
    read_next_token ();
    if (next_token == '(')
    {
      parse_subrule_call (ident);
      type = VALUE;
    }
    else
    {
      buffer_instr (INS_PUSH_SYMBOL, find_symbol (ident));
      type = CONST_VALUE;
    }

    free_mem (&ident);
    return type;
  }

  case TOK_STRING:
#ifdef HANGUL
    encode_hangul (&token_string);
#endif
    buffer_push_string_instr (token_string, NULL);
    read_next_token ();
    return CONST_VALUE;

  case TOK_NUMBER:
    buffer_push_number_instr (token_number);
    read_next_token ();
    return CONST_VALUE;

  case TOK_CONSTANT:
    buffer_instr (INS_PUSH_CONST, find_constant (token_name));
    read_next_token ();
    return CONST_VALUE;

  case TOK_VARIABLE:
  {
    string_t var_name;
    int_t var_index;
    
    parse_var_name (&var_name);
    var_index = find_variable (var_name);
    emit_instr (INS_PUSH_VAR, var_index);
    free_mem (&var_name);
    return VALUE;
  }

  case '(':
    read_next_token ();
    type = parse_expression (condition);
    parse_token (')');
    return type;

  default:
    error ("value expected, not %s", token_as_text (next_token));
  }
}

/*---------------------------------------------------------------------------*/

LOCAL expr_type_t parse_dotted_value (condition_t *condition)
/* Parse a value and a sequence of following ".<ident>" or ".<number>". */
{
  expr_type_t type, elem_type;

  type = parse_simple_value (condition);
  if (type == CONDITION)
    return type;

  while (next_token == '.')
  {
    read_next_token ();
    elem_type = parse_simple_value (condition);

    if (elem_type == CONDITION)
      error ("no condition allowed after \".\"");
    else if (elem_type == VALUE) /* expression is not constant */
      type = VALUE;
    
    if (elem_type == CONST_VALUE) /* check if we can optimize the code */
    {
      if (get_value_type (get_buffer_top_value ()) == SYMBOL_SYMBOL)
	buffer_instr (INS_GET_ATTRIBUTE, 
		      value_to_symbol (pop_buffer_top_value ()));
      else
	buffer_instr (INS_DOT_OPERATION, 0);
    }
    else
      buffer_instr (INS_DOT_OPERATION, 0);
  }
  return type;
}

/*---------------------------------------------------------------------------*/

LOCAL expr_type_t parse_term_value (condition_t *condition)
/* Parse a value that may contain the "*" and the "/" operator. */
{
  expr_type_t type, elem_type;

  type = parse_dotted_value (condition);
  if (type == CONDITION)
    return type;

  while (next_token == '*' || next_token == '/') 
  {
    int_t operator_token = next_token;
    
    read_next_token ();
    elem_type = parse_dotted_value (condition);
    if (elem_type == CONDITION)
      error ("no condition allowed after \"*\" and \"/\"");
    else if (elem_type == VALUE) /* expression is not constant */
      type = VALUE;
    
    if (operator_token == '*')
      buffer_instr (INS_ASTERISK_OPERATION, 0);
    else
      buffer_instr (INS_SLASH_OPERATION, 0);
  }
  return type;
}

/*---------------------------------------------------------------------------*/

LOCAL expr_type_t local_parse_value (condition_t *condition)
/* Parse any value.
 * The code is not necessarily emitted, it may still be in the buffer.
 * Use "parse_value" to get real code.
 * If <constant> == TRUE, parse a constant value. */
{
  expr_type_t type, elem_type;

  if (next_token == '-')
  {
    read_next_token ();
    type = parse_term_value (condition);
    if (type == CONDITION)
      error ("no condition allowed after unary \"-\"");

    buffer_instr (INS_UNARY_MINUS_OP, 0);
  }
  else
  {
    type = parse_term_value (condition);
    if (type == CONDITION)
      return type;
  }

  while (next_token == '+' || next_token == '-') 
  {
    int_t operator_token = next_token;
    
    read_next_token ();
    
    if (operator_token == '-')
    {
      elem_type = parse_term_value (condition);
      if (elem_type == CONDITION)
	error ("no condition allowed after \"-\"");
      else if (elem_type == VALUE) /* expression is not constant */
	type = VALUE;
	
    if (elem_type == CONST_VALUE) /* check if we can optimize the code */
    {
      if (get_value_type (get_buffer_top_value ()) == SYMBOL_SYMBOL)
	buffer_instr (INS_REMOVE_ATTRIBUTE,
		      value_to_symbol (pop_buffer_top_value ()));
      else
	buffer_instr (INS_MINUS_OPERATION, 0);
    }
    else
      buffer_instr (INS_MINUS_OPERATION, 0);
    }
    else /* operator_token == '+' */
    {
      elem_type = parse_term_value (condition);
      if (elem_type == CONDITION)
	error ("no condition allowed after \"+\"");
      else if (elem_type == VALUE) /* expression is not constant */
	type = VALUE;
      
      buffer_instr (INS_PLUS_OPERATION, 0);
    }
  }
  return type;
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_pattern (string_t *pattern)
{
  int_t pattern_var_num;
  string_t segment;
  text_t text = new_text ();
  condition_t condition;
  expr_type_t type;

  while (TRUE)
  {
    if (next_token == TOK_VARIABLE) /* Define a variable for this segment. */
    {
      if (pattern_vars.number == PATTERN_VAR_MAX) 
	error ("too many variables in pattern");
      
      parse_var_name (pattern_vars.name + pattern_vars.number);
      parse_token (':');
      pattern_var_num = pattern_vars.number++;
    } 
    else 
      pattern_var_num = -1;

    type = local_parse_value (&condition);
    if (type != CONST_VALUE 
	|| get_value_type (get_buffer_top_value ()) != STRING_SYMBOL)
      error ("constant string expected");
    segment = compile_pattern (value_to_string (pop_buffer_top_value ()),
			       pattern_var_num);
    add_to_text (text, segment);
    free_mem (&segment);

    if (next_token != ',')
      break;
    
    read_next_token ();
  }

  *pattern = text_to_string (&text);
}

/*---------------------------------------------------------------------------*/

LOCAL void define_pattern_vars (void)
/* Generate code to define the pattern variables <pattern_vars>. */
{
  int_t i;

  for (i = 0; i < pattern_vars.number; i++) 
  {
    emit_instr (INS_PUSH_PATTERN_VAR, i);
    define_var_in_scope (pattern_vars.name[i], code.stack_index - 1);
    free_mem (&pattern_vars.name[i]);
  }
}

/*---------------------------------------------------------------------------*/

LOCAL expr_type_t parse_comparison (condition_t *condition)
/* Parse a malaga comparison and set <condition> accordingly. */
{
  int_t opcode;
  expr_type_t type, type2;
  string_t pattern;
  int_t pattern_index;

  type = local_parse_value (condition);
  if (type == CONDITION)
    return type;

  switch (next_token) 
  {
  case '=':
    opcode = INS_JUMP_IF_EQUAL;
    break;
  case TOK_NOT_EQUAL:
    opcode = INS_JUMP_IF_NOT_EQUAL;
    break;
  case '~':
    opcode = INS_JUMP_IF_CONGR;
    break;
  case TOK_NOT_CONGRUENT:
    opcode = INS_JUMP_IF_NOT_CONGR;
    break;
  case TOK_IN:
    opcode = INS_JUMP_IF_IN;
    break;
  case TOK_LESS:
    opcode = INS_JUMP_IF_LESS;
    break;
  case TOK_LESS_EQUAL:
    opcode = INS_JUMP_IF_NOT_GREATER;
    break;
  case TOK_GREATER:
    opcode = INS_JUMP_IF_GREATER;
    break;
  case TOK_GREATER_EQUAL:
    opcode = INS_JUMP_IF_NOT_LESS;
    break;
  case TOK_MATCHES:
    read_next_token ();
    if (next_token == '(')
    {
      parse_token ('(');
      parse_pattern (&pattern);
      parse_token (')');
    }
    else
      parse_pattern (&pattern);
    copy_string_to_pool (code.string_pool, pattern, &pattern_index);
    free_mem (&pattern);
    emit_instr (INS_MATCH, pattern_index);
    
    condition->last_jump = emit_instr (INS_JUMP_IF_YES, 0);
    condition->true_jumps = NULL;
    condition->false_jumps = NULL;
    
    return CONDITION;
  
  default:
    return type;
  }
  
  read_next_token ();
  type2 = local_parse_value (condition);
  if (type2 == CONDITION)
    error ("no condition allowed in comparisons");
  
  /* Emit code and produce <condition>. */
  condition->last_jump = emit_instr (opcode, 0); 
  condition->true_jumps = NULL;
  condition->false_jumps = NULL;
  return CONDITION;
}

/*---------------------------------------------------------------------------*/

LOCAL expr_type_t parse_simple_condition (condition_t *condition)
/* Parse a malaga comparison that may be negated and return <condition>. */
{
  expr_type_t type;
  int_t pattern_var_number = pattern_vars.number;

  if (next_token == TOK_NOT)
  {
    condition_t condition2;

    read_next_token ();
    type = parse_comparison (&condition2);
    if (pattern_vars.number - pattern_var_number != 0)
      error ("no variable definition allowed in negations");

    convert_to_condition (&type, &condition2);
    negate_jump (condition2.last_jump);
    
    condition->last_jump = condition2.last_jump;
    condition->true_jumps = condition2.false_jumps;
    condition->false_jumps = condition2.true_jumps;
  }
  else
    type = parse_comparison (condition);
  
  return type;
}

/*---------------------------------------------------------------------------*/

LOCAL expr_type_t parse_expression (condition_t *condition)
/* Parse any expression, i.e. a value or a regular condition. */
{
  condition_t condition2;
  expr_type_t type, type2;
  int_t pattern_var_number = pattern_vars.number;

  type = parse_simple_condition (condition);
  
  if (next_token == TOK_OR) 
  { /* There may be a chain of or's behind the first simple condition. */
    convert_to_condition (&type, condition);
    while (next_token == TOK_OR) 
    {
      read_next_token ();
      
      /* All jumps in <false_jump> should point to next instruction. */
      patch_jumps (&condition->false_jumps, code.next_instr_index); 
      
      type2 = parse_simple_condition (&condition2);
      convert_to_condition (&type2, &condition2);
  
      /* Create a new true-jump list. */
      condition->true_jumps = concat_patches (condition->true_jumps, 
					      condition2.true_jumps);
      condition->true_jumps = add_to_patches (condition->true_jumps, 
					      condition->last_jump);
      condition->last_jump = condition2.last_jump;
      condition->false_jumps = condition2.false_jumps;
    }
    if (pattern_vars.number - pattern_var_number != 0)
      error ("no variable definition allowed in disjunctions");
  }
  else if (next_token == TOK_AND)
  { /* There may be a chain of and's behind the first simple condition. */
    convert_to_condition (&type, condition);
    while (next_token == TOK_AND) 
    {
      read_next_token ();
      
      /* All jumps in <true_jump> should point to next instruction. */
      patch_jumps (&condition->true_jumps, code.next_instr_index); 
      
      type2 = parse_simple_condition (&condition2);
      convert_to_condition (&type2, &condition2);
  
      /* Create a new false-jump list. */
      negate_jump (condition->last_jump); 
      condition->false_jumps = concat_patches (condition->false_jumps, 
					       condition2.false_jumps);
      condition->false_jumps = add_to_patches (condition->false_jumps, 
					       condition->last_jump);
      condition->last_jump = condition2.last_jump;
      condition->true_jumps = condition2.true_jumps;
    }
  }
  return type;
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_constant_value (int_t *value_index)
/* Parse a constant value and save its index in *<value_index>. */
{
  expr_type_t type;
  condition_t condition;

  pattern_vars.number = 0;
  type = local_parse_value (&condition);
  if (type != CONST_VALUE)
    error ("constant value expected");

  copy_value_to_pool (code.value_pool, pop_buffer_top_value (), value_index);
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_condition (condition_t *condition)
/* Parse any condition and set <condition>. */
{
  expr_type_t type;
  
  pattern_vars.number = 0;
  type = parse_expression (condition);
  convert_to_condition (&type, condition);
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_attribute_path (void)
/* Parse a sequence of values separated by a ".", 
 * it will be returned on the stack as a list. */
{
  int_t path_length;

  path_length = 0;
  while (TRUE)
  {
    parse_simple_value (FALSE);
    path_length++;
    
    if (next_token != '.')
      break;
    
    read_next_token ();
  }

  /* Emit the instruction that pushes the selection path. */
  buffer_instr (INS_BUILD_PATH, path_length);
  flush_buffer ();
}

/* parse functions for statements ===========================================*/

LOCAL void parse_assert_statement (void)
/* Parse an assert statement: "assert <condition>;". */
{
  condition_t condition;

  new_source_line ();
  if (next_token != TOK_ASSERT && next_token != '!')
    error ("\"!\" or \"assert\" expected");
  read_next_token ();
  parse_condition (&condition);
  patch_jumps (&condition.false_jumps, code.next_instr_index);
  emit_instr (INS_ERROR, ASSERTION_ERROR);
  patch_jumps (&condition.true_jumps, code.next_instr_index);
  patch_instr (condition.last_jump, code.next_instr_index);
  define_pattern_vars ();
  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_assignment (void)
/* Parse an assignment: 
 * "<variable> {.<attribute} (:=|:=+|:=-|:=*|:=/) <value> ;". */
{
  string_t var_name; /* the name of the variable */
  int_t index;
  int_t assignment; /* one of TOK_ASSIGN, TOK_ADD, TOK_SUBTRACT */
  bool_t has_path;

  new_source_line ();
  parse_var_name (&var_name);
  index = find_variable (var_name);
  free_mem (&var_name);

  if (next_token != '.')
    has_path = FALSE;
  else
  {
    has_path = TRUE;
    read_next_token ();
    parse_attribute_path ();
  }

  /* Read the assignment token. */
  assignment = next_token;
  if (assignment != TOK_ASSIGN 
      && assignment != TOK_ASSIGN_PLUS && assignment != TOK_ASSIGN_MINUS
      && assignment != TOK_ASSIGN_ASTERISK && assignment != TOK_ASSIGN_SLASH)
    error ("\":=\", \":=+\", \":=-\", \":=*\", or \":=/\" expected");

  read_next_token ();
  parse_value ();
      
  if (has_path)
  {
    switch (assignment)
    {
    case TOK_ASSIGN:
      emit_instr (INS_SET_VAR_PATH, index);
      break;
    case TOK_ASSIGN_PLUS:
      emit_instr (INS_PLUS_VAR_PATH, index);
      break;
    case TOK_ASSIGN_MINUS:
      emit_instr (INS_MINUS_VAR_PATH, index);
      break;
    case TOK_ASSIGN_ASTERISK: 
      emit_instr (INS_ASTERISK_VAR_PATH, index); 
      break;
    case TOK_ASSIGN_SLASH:
      emit_instr (INS_SLASH_VAR_PATH, index);
      break;
    }
  }
  else
  {
    switch (assignment)
    {
    case TOK_ASSIGN:
      emit_instr (INS_SET_VAR, index);
      break;
    case TOK_ASSIGN_PLUS:
      emit_instr (INS_PLUS_VAR, index);
      break;
    case TOK_ASSIGN_MINUS:
      emit_instr (INS_MINUS_VAR, index);
      break;
    case TOK_ASSIGN_ASTERISK:
      emit_instr (INS_ASTERISK_VAR, index);
      break;
    case TOK_ASSIGN_SLASH:
      emit_instr (INS_SLASH_VAR, index);
      break;
    }
  }

  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_choose_statement (void)
/* Parse a choose statement. */
{
  string_t var_name;
  instr_t *patch_ptr;
  int_t label;
  int_t var_index;

  new_source_line ();
  parse_token (TOK_CHOOSE);
  parse_var_name (&var_name);

  parse_token (TOK_IN);

  /* Reserve place for variables. */
  var_index = code.stack_index;
  emit_instr (INS_PUSH_NULL, 1);

  parse_value ();

  emit_instr (INS_PUSH_VAR, var_index + 1);
  emit_instr (INS_GET_1ST_ELEMENT, 0);
  emit_instr (INS_PUSH_VAR, var_index + 2);
  emit_instr (INS_TERMINATE_IF_NULL, 0);
  label = code.next_instr_index;
  emit_instr (INS_PUSH_VAR, var_index + 2);
  emit_instr (INS_SET_VAR, var_index);
  emit_instr (INS_ITERATE, var_index + 2);
  emit_instr (INS_PUSH_VAR, var_index + 2);
  patch_ptr = emit_instr (INS_JUMP_IF_NULL, 0);
  emit_instr (INS_JUMP_LATER, label);
  patch_instr (patch_ptr, code.next_instr_index);
  define_var_in_scope (var_name, var_index);
  free_mem (&var_name);

  /* Pop list and element of this list. */
  emit_instr (INS_POP, 2);
  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_define_statement (void)
/* Parse a define-statement: "define <variable> := <value>;". */
{
  string_t var_name; /* the name of the variable */

  new_source_line ();
  parse_token (TOK_DEFINE);
  parse_var_name (&var_name);
  parse_token (TOK_ASSIGN);
  parse_value ();

  define_var_in_scope (var_name, code.stack_index-1);
  free_mem (&var_name);
  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_foreach_statement (void)
/* Parse a foreach-statement. */
{
  string_t var_name; /* name of iteration variable */
  int_t list_index; /* stack index of list variable */
  int_t label; /* label where to jump back */
  instr_t *patch_ptr; /* where to put the exit jump */

  new_source_line ();

  parse_token (TOK_FOREACH);
  parse_var_name (&var_name);
  parse_token (TOK_IN);
  parse_value ();
  parse_token (':');

  list_index = code.stack_index - 1;

  emit_instr (INS_PUSH_VAR, list_index);
  emit_instr (INS_GET_1ST_ELEMENT, 0);

  /* Emit code to test if loop will be repeated. */
  label = code.next_instr_index;
  emit_instr (INS_PUSH_VAR, list_index + 1);
  patch_ptr = emit_instr (INS_JUMP_IF_NULL, 0);

  /* Open a new scope, create visible copies of iteration variables,
   * parse statements and close scope. */
  open_scope ();
  emit_instr (INS_PUSH_VAR, list_index + 1);
  define_var_in_scope (var_name, code.stack_index - 1);
  parse_statements ();
  close_scope (TRUE);

  parse_token (TOK_END);
  if (next_token == TOK_FOREACH)
    read_next_token ();
  parse_token (';');

  no_source_line ();

  /* Iterate variables. */
  emit_instr (INS_ITERATE, list_index + 1);
  emit_instr (INS_JUMP, label);

  /* Patch the jump to exit. */
  patch_instr (patch_ptr, code.next_instr_index);

  /* Pop iteration variable and list. */
  emit_instr (INS_POP, 2);
  free_mem (&var_name);
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_if_statement (void)
/* Parse an if-statement. */
{
  condition_t condition;
  patch_t *end_jumps;
  patch_t *else_jumps;
  instr_t *end_jump;
  bool_t need_jump_to_end;

  else_jumps = NULL;
  end_jumps = NULL;
  need_jump_to_end = FALSE;

  test_token (TOK_IF);

  do 
  {
    if (need_jump_to_end)
    { /* Emit a jump if there has already been another branch. */
      no_source_line ();
      end_jump = emit_instr (INS_JUMP, 0);
      end_jumps = add_to_patches (end_jumps, end_jump);
    }

    read_next_token ();
    new_source_line ();
    
    /* Jump here if previous conditions were false. */
    patch_jumps (&else_jumps, code.next_instr_index);
    
    parse_condition (&condition);
    parse_token (TOK_THEN);
    
    /* Fall through if condition is true. */
    negate_jump (condition.last_jump);
    else_jumps = add_to_patches (condition.false_jumps, 
				 condition.last_jump);
    patch_jumps (&condition.true_jumps, code.next_instr_index);
    
    open_scope ();
    define_pattern_vars ();
    parse_statements ();
    close_scope (TRUE);
    
    need_jump_to_end = TRUE;
    
  } while (next_token == TOK_ELSEIF);
  
  if (next_token == TOK_ELSE) 
  { /* Emit a jump to the end. */
    no_source_line ();
    end_jump = emit_instr (INS_JUMP, 0);
    end_jumps = add_to_patches (end_jumps, end_jump);
    
    read_next_token ();
    
    patch_jumps (&else_jumps, code.next_instr_index);
    open_scope ();
    parse_statements ();
    close_scope (TRUE);
  }
  else 
    patch_jumps (&else_jumps, code.next_instr_index);
  
  parse_token (TOK_END);
  if (next_token == TOK_IF)
    read_next_token ();

  patch_jumps (&end_jumps, code.next_instr_index);
  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_parallel_statement (void)
/* Parse a parallel-statement. */
{
  patch_t *end_jumps;
  instr_t *patch_ptr;

  patch_ptr = NULL;
  end_jumps = NULL;

  test_token (TOK_PARALLEL);
  do
  {
    read_next_token (); /* Read over "parallel" or "and". */
    
    /* Jump to end and patch JUMP_LATER if there is already a subrule. */
    if (patch_ptr != NULL)
    {
      instr_t *end_jump;
      
      no_source_line ();
      end_jump = emit_instr (INS_JUMP, 0);
      end_jumps = add_to_patches (end_jumps, end_jump);
      patch_instr (patch_ptr, code.next_instr_index);
    }
    else
      new_source_line ();
    
    patch_ptr = emit_instr (INS_JUMP_LATER, 0);
    open_scope ();
    parse_statements ();
    close_scope (TRUE);
    
  } while (next_token == TOK_AND);
  
  /* The last patch is invalid, replace it by an INS_NOP. */
  *patch_ptr = INSTR (INS_NOP, 0);

  patch_jumps (&end_jumps, code.next_instr_index);

  parse_token (TOK_END);
  if (next_token == TOK_PARALLEL)
    read_next_token ();

  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_repeat_statement (void)
/* Parse a repeat-statement. */
{
  condition_t condition;
  int_t label; /* label where to jump back */
  int_t values_to_pop; /* number of values to pop when exiting */

  /* No need for "new_source_line" since no code is generated here. */
  parse_token (TOK_REPEAT);
  label = code.next_instr_index;
  open_scope ();
  parse_statements ();
  values_to_pop = code.stack_index - current_scope->start_index;
  new_source_line ();
  parse_token (TOK_WHILE);
  parse_condition (&condition);
  parse_token (';');
  patch_jumps (&condition.true_jumps, code.next_instr_index);
  define_pattern_vars ();
  parse_statements ();
  close_scope (TRUE);

  parse_token (TOK_END);
  if (next_token == TOK_REPEAT)
    read_next_token ();
  parse_token (';');

  no_source_line ();
  emit_instr (INS_JUMP, label);
  patch_jumps (&condition.true_jumps, code.next_instr_index);
  negate_jump (condition.last_jump);
  patch_instr (condition.last_jump, code.next_instr_index);

  /* We have to pop manually, since we jumped from midst a scope. */
  if (values_to_pop > 0) 
    emit_instr (INS_POP, values_to_pop);
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_require_statement (void)
/* Parse a require statement: "require <condition>;" or "? <condition>;". */
{
  condition_t condition;

  new_source_line ();
  if (next_token != TOK_REQUIRE && next_token != '?')
    error ("\"require\" or \"?\" expected");
  read_next_token ();
  parse_condition (&condition);
  patch_jumps (&condition.false_jumps, code.next_instr_index);
  emit_instr (INS_TERMINATE, 0);
  patch_jumps (&condition.true_jumps, code.next_instr_index);
  patch_instr (condition.last_jump, code.next_instr_index);
  define_pattern_vars ();
  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_result_statement (void)
/* Parse a result statement. */
{
  int_t rule_set;
  
  new_source_line ();
  parse_token (TOK_RESULT);
  parse_value ();

  switch (rule_type)
  {
  case COMBI_RULE:
    parse_token (',');
    if (next_token == TOK_ACCEPT)
    {
      read_next_token ();
      emit_instr (INS_ADD_END_STATE, 0);
    }
    else
    {
      parse_rule_set (&rule_set);
      emit_instr (INS_ADD_STATE, rule_set);
    }
    break;

  case END_RULE:
    parse_token (',');
    parse_token (TOK_ACCEPT);
    emit_instr (INS_ADD_END_STATE, 0);
    break;
  
  case ALLO_RULE:
    parse_token (',');
    parse_value ();
    emit_instr (INS_ADD_ALLO, 0);
    break;

  case FILTER_RULE:
  case ROBUST_RULE:
    emit_instr (INS_ADD_END_STATE, 0);
    break;

  default:
    error ("\"result\" not allowed in this rule");
  }

  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_return_statement (void)
/* Parse a return statement. */
{
  rule_t *rule;
  
  new_source_line ();
  parse_token (TOK_RETURN);
  parse_value ();

  if (rule_type == SUBRULE)
  {
    rule = pool_item (code.rule_pool, code.rule_number);
    emit_instr (INS_RETURN, rule->num_params);
  }
  else if (rule_type == PRUNING_RULE)
    emit_instr (INS_ACCEPT, 0);
  else
    error ("\"return\" is only allowed in subrules and pruning rules");

  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_statements (void)
/* Parse a statement sequence. */
{
  while (TRUE)
  {
    switch (next_token) 
    {
    case TOK_ASSERT:
    case '!':
      parse_assert_statement ();
      break;
      
    case TOK_VARIABLE:
      parse_assignment ();
      break;
      
    case TOK_CHOOSE:
      parse_choose_statement ();
      break;
      
    case TOK_DEFINE:
      parse_define_statement ();
      break;
      
    case TOK_ERROR:
    {
      int_t name_index;

      new_source_line ();
      read_next_token ();
      test_token (TOK_STRING);
      copy_string_to_pool (code.string_pool, token_string, &name_index);
      emit_instr (INS_ERROR, name_index);
      read_next_token ();
      parse_token (';');
      break;
    }
    
    case TOK_FAIL:
      new_source_line ();
      read_next_token ();
      emit_instr (INS_TERMINATE, 0);
      parse_token (';');
      break;
      
    case TOK_FOREACH:
      parse_foreach_statement ();
      break;
      
    case TOK_IF:
      parse_if_statement ();
      break;
      
    case TOK_PARALLEL:
      parse_parallel_statement ();
      break;
      
    case TOK_REPEAT:
      parse_repeat_statement ();
      break;

    case '?':
    case TOK_REQUIRE:
      parse_require_statement ();
      break;
      
    case TOK_RESULT:
      parse_result_statement ();
      break;
      
    case TOK_RETURN:
      parse_return_statement ();
      break;
      
    default:
      return;
    }
  }
}

/* parse functions for rules ================================================*/

LOCAL void parse_rule (void)
/* Parse a rule. */
{
  /* the token at the beginning of this rule */
  int_t rule_token = next_token; 
  int_t num_params, i;
  string_t rule_name;
  string_t params[PARAMS_MAX];

  switch (code.file_type)
  {
  case ALLO_RULE_FILE:
    if (next_token != TOK_ALLO_RULE && next_token != TOK_OUTPUT_FILTER
	&& next_token != TOK_SUBRULE)
      error ("\"allo_rule\", \"output_filter\" or \"subrule\" expected");
    break;
  case MORPHO_RULE_FILE:
    if (next_token != TOK_COMBI_RULE && next_token != TOK_END_RULE 
	&& next_token != TOK_OUTPUT_FILTER && next_token != TOK_ROBUST_RULE 
	&& next_token != TOK_SUBRULE)
      error ("\"combi_rule\", \"end_rule\", \"output_filter\", "
	     "\"robust_rule\" or \"subrule\" expected");
    break;
  case SYNTAX_RULE_FILE:
    if (next_token != TOK_COMBI_RULE && next_token != TOK_END_RULE
	&& next_token != TOK_INPUT_FILTER && next_token != TOK_OUTPUT_FILTER 
	&& next_token != TOK_PRUNING_RULE && next_token != TOK_SUBRULE)
      error ("\"combi_rule\", \"end_rule\", \"input_filter\", "
	     "\"output_filter\", \"pruning_rule\" or \"subrule\" expected");
    break;
  }

  switch (next_token)
  {
  case TOK_ALLO_RULE:
    rule_type = ALLO_RULE;
    break;
  case TOK_COMBI_RULE:
    rule_type = COMBI_RULE;
    break;
  case TOK_END_RULE:
    rule_type = END_RULE;
    break;
  case TOK_INPUT_FILTER:
  case TOK_OUTPUT_FILTER:
    rule_type = FILTER_RULE;
    break;
  case TOK_PRUNING_RULE:
    rule_type = PRUNING_RULE;
    break;
  case TOK_ROBUST_RULE:
    rule_type = ROBUST_RULE;
    break;
  case TOK_SUBRULE:
    rule_type = SUBRULE;
    break;
  }

  /* Remember rule name. */
  read_next_token ();
  test_token (TOK_IDENT);
  rule_name = new_string (token_name, NULL);
  read_next_token ();
  
  /* Read parameter list. */
  num_params = 0;
  parse_token ('(');
  while (next_token != ')') 
  {
    test_token (TOK_VARIABLE);
    if (num_params == PARAMS_MAX)
      error ("too many parameters");

    params[num_params] = new_string (token_name, NULL);
    num_params++;
    read_next_token ();
    if (next_token != ',')
      break;
      
    read_next_token ();
  }
  parse_token (')');

  switch (rule_type)
  {
  case ALLO_RULE:
  case END_RULE:
  case FILTER_RULE:
  case ROBUST_RULE:
  case PRUNING_RULE:
    if (num_params != 1)
      error ("%s must have one parameter", token_as_text (rule_token));
    break;
  case COMBI_RULE:
    if (num_params < 2 || num_params > 4)
      error ("\"combi_rule\" must have 2 to 4 parameters");
    break;
  case SUBRULE:
    break;
  }

  code.rule_number = enter_rule (rule_name, code.next_instr_index, rule_type, 
				 num_params);

  if (rule_token == TOK_PRUNING_RULE)
  {
    if (code.pruning_rule != -1)
      error ("\"pruning_rule\" defined twice");

    code.pruning_rule = code.rule_number;
  }
  else if (rule_token == TOK_ROBUST_RULE)
  {
    if (code.robust_rule != -1)
      error ("\"robust_rule\" defined twice");
      
    code.robust_rule = code.rule_number;
  }
  else if (rule_token == TOK_ALLO_RULE)
  {
    if (code.allo_rule != -1)
      error ("\"allo_rule\" defined twice");

    code.allo_rule = code.rule_number;
  }
  else if (rule_token == TOK_INPUT_FILTER)
  {
    if (code.input_filter != -1)
      error ("\"input_filter\" defined twice");
    
    code.input_filter = code.rule_number;
  }
  else if (rule_token == TOK_OUTPUT_FILTER)
  {
    if (code.output_filter != -1)
      error ("\"output_filter\" defined twice");
    
    code.output_filter = code.rule_number;
  }

  parse_token (':');

  open_scope ();
  for (i = 0; i < num_params; i++)
  {
    if (rule_type == SUBRULE)
      define_var_in_scope (params[i], i - (num_params + 2));
    else
    {
      define_var_in_scope (params[i], code.stack_index);
      code.stack_index++;
    }

    free_mem (&params[i]);
  }
  parse_statements ();
  close_scope (FALSE);

  new_source_line ();
  if (rule_type == SUBRULE || rule_type == PRUNING_RULE)
    emit_instr (INS_ERROR, NO_RETURN_ERROR);
  else
    emit_instr (INS_TERMINATE, 0);

  parse_token (TOK_END);
  if (next_token == rule_token)
    read_next_token ();
  if (next_token == TOK_IDENT)
  {
    if (strcmp_no_case (token_name, rule_name) != 0) 
      error ("\"%s\" expected, not \"%s\"", rule_name, token_name);
    
    read_next_token ();
  }

  free_mem (&rule_name);

  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_constant_definition (void)
/* Parse a constant definition. */
{
  string_t const_name;
  int_t const_index;

  parse_token (TOK_DEFINE);
  test_token (TOK_CONSTANT);
  const_name = new_string (token_name, NULL);
  read_next_token ();
  parse_token (TOK_ASSIGN);
  parse_constant_value (&const_index);
  define_constant (const_name, const_index);
  free_mem (&const_name);
  parse_token (';');
}

/*---------------------------------------------------------------------------*/

LOCAL void parse_rules (void)
/* Parse rules, constant definitions and includes until end-of-file. */
{
  while (next_token != EOF) 
  {
    if (next_token == TOK_INCLUDE) 
    {
      string_t file_name, path;

      /* Parse file name and expand to absolute file name. */
      read_next_token ();
      test_token (TOK_STRING);
      path = absolute_path (token_string, current_file_name ());
      file_name = copy_string_to_pool (code.string_pool, path, NULL);
      free_mem (&path);
      begin_include (file_name);
      parse_rules ();
      end_include ();
      parse_token (';');
    } 
    else if (next_token == TOK_DEFINE)
      parse_constant_definition ();
    else if (next_token == TOK_INITIAL)
    {
      if (code.file_type != MORPHO_RULE_FILE 
	  && code.file_type != SYNTAX_RULE_FILE)
	error ("\"initial\" only allowed in combi_rule files");

      if (code.initial_rule_set != -1)
	error ("initial state already defined");
      
      read_next_token ();
      
      parse_constant_value (&code.initial_cat);
      parse_token (',');
      parse_rule_set (&code.initial_rule_set);
      parse_token (';');
    }
    else
      parse_rule ();
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void parse_rule_file (void)
/* Parse a rule file. */
{
  last_statement_line = -1;
  last_statement_file_name = NULL;

  code.initial_rule_set = -1;
  code.robust_rule = code.pruning_rule = -1;
  code.allo_rule = code.input_filter = code.output_filter = -1;

  parse_rules ();

  if ((code.file_type == MORPHO_RULE_FILE 
       || code.file_type == SYNTAX_RULE_FILE)
      && code.initial_rule_set == -1)
    error ("missing initial state");
  else if (code.file_type == ALLO_RULE_FILE && code.allo_rule == -1)
    error ("missing allo_rule");
}

/* end of file ==============================================================*/
