/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module manages the emission of instructions and keeps track of the 
 * stack index. It also holds buffers for the compiled code. */

/* includes =================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "symbols.h"
#include "rule_type.h"
#include "files.h"
#include "malaga_files.h"

#undef GLOBAL
#define GLOBAL

#include "rule_code.h"

/* variables ================================================================*/

LOCAL instr_t *instr_stack;
/* The instruction stack. The associated constant values are stored in 
 * <values_stack>. */
LOCAL int_t instr_stack_size; /* the size of <instr_stack>. */

/* functions ================================================================*/

GLOBAL void init_rule_code (int_t file_type)
/* Initialise this module. 
 * <code> will contain compilation data for a file of <file_type>.
 * <file_type> may be ALLO_RULE_FILE, MORPHO_RULE_FILE, or SYNTAX_RULE_FILE. */
{
  code.file_type = file_type;
  code.stack_index = 0;
  code.next_instr_index = 0;
  code.rule_pool = new_pool (sizeof (rule_t));
  code.rule_set_pool = new_pool (sizeof (int_t));
  code.instr_pool = new_pool (sizeof (instr_t));
  code.value_pool = new_pool (sizeof (cell_t));
  code.string_pool = new_pool (sizeof (char));
  code.src_line_pool = new_pool (sizeof (src_line_t));
  code.var_pool = new_pool (sizeof (var_t));
  code.var_scope_pool = new_pool (sizeof (var_scope_t));

  instr_stack_size = 50;
  instr_stack = new_vector (sizeof (instr_t), instr_stack_size);
  top = 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL void term_rule_code (void)
/* Terminate this module. */
{
  free_pool (&code.rule_pool);
  free_pool (&code.rule_set_pool);
  free_pool (&code.instr_pool);
  free_pool (&code.value_pool);
  free_pool (&code.string_pool);
  free_pool (&code.src_line_pool);
  free_pool (&code.var_pool);
  free_pool (&code.var_scope_pool);
  free_mem (&instr_stack);
}

/*---------------------------------------------------------------------------*/

GLOBAL void write_code (string_t file_name)
/* Write <code> to <file_name>. */
{ 
  FILE *stream;
  rule_header_t rule_header;

  stream = open_stream (file_name, "wb");

  /* Set rule file header data. */
  set_header (&rule_header.common_header, RULE_FILE, RULE_CODE_VERSION);
  rule_header.initial_rule_set = code.initial_rule_set;
  rule_header.initial_cat = code.initial_cat;
  rule_header.robust_rule = code.robust_rule;
  rule_header.pruning_rule = code.pruning_rule;
  rule_header.allo_rule = code.allo_rule;
  rule_header.input_filter = code.input_filter;
  rule_header.output_filter = code.output_filter;
  rule_header.rules_size = pool_items (code.rule_pool);
  rule_header.rule_sets_size = pool_items (code.rule_set_pool);
  rule_header.instrs_size = pool_items (code.instr_pool);
  rule_header.values_size = pool_items (code.value_pool);
  rule_header.src_lines_size = pool_items (code.src_line_pool);
  rule_header.vars_size = pool_items (code.var_pool);
  rule_header.var_scopes_size = pool_items (code.var_scope_pool);
  rule_header.strings_size = pool_items (code.string_pool);

  /* Write the header. */
  write_vector (&rule_header, sizeof (rule_header), 1, stream, file_name); 

  /* Write the tables. */
  write_pool (code.rule_pool, stream, file_name);
  write_pool (code.rule_set_pool, stream, file_name);
  write_pool (code.instr_pool, stream, file_name);
  write_pool (code.value_pool, stream, file_name);
  write_pool (code.src_line_pool, stream, file_name);
  write_pool (code.var_pool, stream, file_name);
  write_pool (code.var_scope_pool, stream, file_name);
  write_pool (code.string_pool, stream, file_name);

  close_stream (&stream, file_name);
}

/*---------------------------------------------------------------------------*/

LOCAL void set_stack_index (int_t opcode, int_t info)
/* Set the stack index according to the given instruction. */
{ 
  /* We can not check for stack overflow here (sorry), because in
   * "choose" and "foreach", a PUSH_NULL instruction is patched, so
   * stack index is (transiently) not quite the real thing. */

  switch (opcode) 
  {
  case INS_ERROR:
  case INS_TERMINATE:
  case INS_NOP:
  case INS_UNARY_MINUS_OP:
  case INS_GET_ATTRIBUTE:
  case INS_REMOVE_ATTRIBUTE:
  case INS_STD_FUNCTION:
  case INS_MATCH:
  case INS_GET_1ST_ELEMENT:
  case INS_ITERATE:
  case INS_JUMP:
  case INS_JUMP_NOW:
  case INS_JUMP_LATER:
    /* These instructions leave the stack size unchanged. */
    break;
  case INS_PUSH_VAR:
  case INS_PUSH_CONST:
  case INS_PUSH_SYMBOL:
  case INS_PUSH_PATTERN_VAR:
  case INS_JUMP_SUBRULE: /* subtract number of parameters from stack_index */
    code.stack_index++;
    break;
  case INS_PUSH_NULL:
    code.stack_index += info;
    break;
  case INS_ADD_END_STATE:
  case INS_ADD_STATE:
  case INS_TERMINATE_IF_NULL:
  case INS_DOT_OPERATION:
  case INS_PLUS_OPERATION:
  case INS_MINUS_OPERATION:
  case INS_ASTERISK_OPERATION:
  case INS_SLASH_OPERATION:
  case INS_SET_VAR:
  case INS_PLUS_VAR:
  case INS_MINUS_VAR:
  case INS_ASTERISK_VAR:
  case INS_SLASH_VAR:
  case INS_JUMP_IF_NULL:
  case INS_JUMP_IF_NOT_NULL:
  case INS_JUMP_IF_YES:
  case INS_JUMP_IF_NO:
  case INS_ACCEPT: /* no instruction after this instruction is executed. */
  case INS_RETURN: /* no instruction after this instruction is executed. */
    code.stack_index--;
    break;
  case INS_ADD_ALLO:
  case INS_SET_VAR_PATH:
  case INS_PLUS_VAR_PATH:
  case INS_MINUS_VAR_PATH:
  case INS_ASTERISK_VAR_PATH:
  case INS_SLASH_VAR_PATH:
  case INS_JUMP_IF_EQUAL:
  case INS_JUMP_IF_NOT_EQUAL:
  case INS_JUMP_IF_CONGR:
  case INS_JUMP_IF_NOT_CONGR:
  case INS_JUMP_IF_IN:
  case INS_JUMP_IF_NOT_IN:
  case INS_JUMP_IF_LESS:
  case INS_JUMP_IF_NOT_LESS:
  case INS_JUMP_IF_GREATER:
  case INS_JUMP_IF_NOT_GREATER:
    code.stack_index -= 2;
    break;
  case INS_POP:
    code.stack_index -= info;
    break;
  case INS_BUILD_LIST:
  case INS_BUILD_PATH:
    code.stack_index -= (info - 1);
    break;
  case INS_BUILD_RECORD:
    code.stack_index -= (2*info - 1);
    break;
  default:
    error ("internal (instruction %d unknown in rule_code)", opcode);
  }
}

/*---------------------------------------------------------------------------*/

LOCAL instr_t *local_emit_instr (int_t opcode, int_t info)
/* Emit an instruction to the instruction pool. DO NOT FLUSH BUFFER!
 * Return the address of the instruction in the pool. */
{
  instr_t instr;
  instr_t *instr_ptr;

  if (info < INSTR_INFO_MIN || info > INSTR_INFO_MAX)
    error ("internal (instruction info out of range)");

  /* Fill the next instruction */
  instr = INSTR (opcode, info);

  /* Generate instruction. */
  instr_ptr = (instr_t *) copy_to_pool (code.instr_pool, &instr, 1, NULL);
  set_stack_index (opcode, info);
  code.next_instr_index = pool_items (code.instr_pool);

  return instr_ptr;
}

/* functions that support constant folding of values ========================*/

LOCAL void put_instr (int_t opcode, int_t info)
/* Put the instruction (<opcode>,<info>) at <instr_stack[top-1]>. */
{
  if (top > instr_stack_size)
    instr_stack_size = renew_vector (&instr_stack, sizeof (u_int_t), 2 * top);
  instr_stack[top-1] = INSTR (opcode, info);
}

/*---------------------------------------------------------------------------*/

GLOBAL void buffer_instr (int_t opcode, int_t info)
/* Buffer the instructions BUILD_LIST, BUILD_RECORD, PUSH_SYMBOL,
 * and PUSH_CONST for constant folding. */

{
  DB_ASSERT (info >= INSTR_INFO_MIN && info <= INSTR_INFO_MAX);

  switch (opcode) 
  {
  case INS_PUSH_SYMBOL:
    push_symbol_value (info);
    put_instr (INS_PUSH_SYMBOL, info);
    break;
    
  case INS_PUSH_CONST:
    push_value ((value_t) pool_item (code.value_pool, info));
    put_instr (INS_PUSH_CONST, info);
    break;
    
  case INS_BUILD_LIST:
    if (top >= info) /* Execute the operation in the buffer. */
    {
      build_list (info);
      put_instr (INS_PUSH_CONST, -1);
    } 
    else
      emit_instr (opcode, info);
    break;
    
  case INS_BUILD_RECORD:
    if (top >= 2*info) /* Execute the operation in the buffer. */
    {
      build_record (info);
      put_instr (INS_PUSH_CONST, -1);
    }
    else
      emit_instr (opcode, info);
    break;
    
  case INS_BUILD_PATH:
    if (top >= info) /* Execute the operation in the buffer. */
    {
      build_path (info);
      put_instr (INS_PUSH_CONST, -1);
    }
    else
      emit_instr (opcode, info);
    break;

  case INS_DOT_OPERATION:
  case INS_PLUS_OPERATION:
  case INS_MINUS_OPERATION:
  case INS_ASTERISK_OPERATION:
  case INS_SLASH_OPERATION:
    if (top >= 2) /* Execute the operation in the buffer. */
    {
      switch (opcode)
      {
      case INS_DOT_OPERATION: 
	dot_operation ();
	break;
      case INS_PLUS_OPERATION: 
	plus_operation ();
	break;
      case INS_MINUS_OPERATION: 
	minus_operation ();
	break;
      case INS_ASTERISK_OPERATION: 
	asterisk_operation ();
	break;
      case INS_SLASH_OPERATION: 
	slash_operation ();
	break;
      default:
	error ("internal (unknown instruction in constant folding)");
      }
      put_instr (INS_PUSH_CONST, -1);
    }
    else
      emit_instr (opcode, info);
    break;

  case INS_UNARY_MINUS_OP:
  case INS_GET_ATTRIBUTE:
  case INS_REMOVE_ATTRIBUTE:
    if (top >= 1) /* Execute the operation in the buffer. */
    {
      switch (opcode)
      {
      case INS_UNARY_MINUS_OP: 
	unary_minus_operation ();
	break;
      case INS_GET_ATTRIBUTE:
	push_value (get_attribute (value_stack[--top], info));
	break;
      case INS_REMOVE_ATTRIBUTE: 
	remove_attribute (info);
	break;
      default:
	error ("internal (unknown instruction in constant folding)");
      }
      put_instr (INS_PUSH_CONST, -1);
    } 
    else
      emit_instr (opcode, info);
    break;

  default:
    emit_instr (opcode, info);
    break;
  }
}

/*---------------------------------------------------------------------------*/

GLOBAL void buffer_push_number_instr (double number)
/* Buffer the instruction PUSH_CONST with <number> converted to a value. */
{
  push_number_value (number);
  put_instr (INS_PUSH_CONST, -1);
}

/*---------------------------------------------------------------------------*/

GLOBAL void buffer_push_string_instr (string_t string, string_t string_end)
/* Buffer the instruction PUSH_CONST with the given value. */
{
  push_string_value (string, string_end);
  put_instr (INS_PUSH_CONST, -1);
}

/*---------------------------------------------------------------------------*/

GLOBAL void flush_buffer (void)
/* Emit the instructions that are still in the buffer. */
{
  int_t i;

  for (i = 0; i < top; i++) 
  {
    switch (OPCODE (instr_stack[i])) 
    {
    case INS_PUSH_CONST:
    {
      int_t value_index;
      
      if (INSTR_INFO (instr_stack[i]) == -1)
	copy_value_to_pool (code.value_pool, value_stack[i], &value_index);
      else
	value_index = INSTR_INFO (instr_stack[i]);

      local_emit_instr (INS_PUSH_CONST, value_index);
      break;
    }

    case INS_PUSH_SYMBOL: 
      local_emit_instr (INS_PUSH_SYMBOL, INSTR_INFO (instr_stack[i]));
      break;
    }
  }

  top = 0;
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t get_buffer_top_value (void)
/* Test if the buffer contains a value and return the top value. */
{
  if (top == 0)
    error ("internal (buffer is empty)");

  return value_stack[top-1];
}

/*---------------------------------------------------------------------------*/

GLOBAL value_t pop_buffer_top_value (void)
/* Pop the top value in the buffer. */
{
  if (top == 0)
    error ("internal (buffer is empty)");

  return value_stack[--top];
}

/*---------------------------------------------------------------------------*/

GLOBAL instr_t *emit_instr (int_t opcode, int_t info)
/* Emit an instruction to the instruction pool (flushes buffer before)
 * and return the address of the instruction in the pool. */
{
  flush_buffer ();
  return local_emit_instr (opcode, info);
}

/* end of file ==============================================================*/
