/*
 * Copyright 1995,96,97 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: mref.c,v 2.9 1997/04/18 15:50:54 bousch Exp $
 *
 * Operations on mrefs. Mrefs are containers referencing mnodes; they are
 * the preferred way to access SAML, because they handle the gory details
 * of garbage collecting and reference counting. If you want to handle
 * these details yourself, just use mnodes.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "saml.h"
#include "saml-errno.h"
#include "saml-util.h"
#include "mnode.h"

void panic_invalid_mref(mref_t) EXITING;

void panic_invalid_mref (mref_t mr)
{
	char msg[80];
	sprintf(msg, "invalid mref number %d\n", mr);
	saml_panic(msg);
}

#define ALLOC_RING_SIZE 16
static gr_string* alloc_ring[ALLOC_RING_SIZE];
static int arindex = 0;

/*
 * Implementation notes: mrefs are stored in a dynamically-allocated array
 * mref_table[], whose current size is mref_active. Each element ("slot")
 * of this array contains either a pointer to a mnode, or a reference to
 * the next free slot (see below).
 *
 * Assuming that all mnodes are allocated at even addresses, we can steal
 * the least significant bit to indicate which member of the union is valid.
 * Even values are pointers; an odd value of 2*k+1 means that the next free
 * slot has index k, and -1 means no more free slots.
 *
 * One must also assume sizeof(s_mnode*) == sizeof(long) at least on
 * big-endian machines, otherwise the LSBs of slot.node and slot.xnext
 * wouldn't be the same.
 */

typedef union {
	s_mnode* node;	/* Either a mnode address, if even */
	long xnext;	/* or (1+2*next_free_slot) if odd */
} mref_slot;

static mref_slot* mref_table = NULL;
static int mref_active = 0;
static int first_free = -1;

static inline s_mnode** get_mnaddress (mref_t mr)
{
	if ((unsigned)mr < mref_active) {
		mref_slot* aslot = &mref_table[mr];
		if (0 == (aslot->xnext & 1))
			return &(aslot->node);
	}
	panic_invalid_mref(mr);
}

static inline s_mnode* get_mnode (mref_t mr)
{
	if ((unsigned)mr < mref_active) {
		mref_slot slot = mref_table[mr];
		if (0 == (slot.xnext & 1))
			return slot.node;
	}
	panic_invalid_mref(mr);
}

static int get_error_code (void_mnode *v)
{
	int errno = v->number;
	(*saml_error_handler)(errno, v->where);
	return -errno;
}

static inline int saml_check_mnode (s_mnode *mn)
{
	int ret = 0;
	if (mn->type == ST_VOID)
	  ret = get_error_code((void_mnode*)mn);
	return ret;
}

static void refill_mref_free_list (void)
{
	int i, new_size;

	new_size = mref_active ? 2*mref_active : 256;
	mref_table = realloc(mref_table, new_size * sizeof(mref_slot));
	if (mref_table == NULL)
		panic_out_of_memory();
	/*
	 * Mark the new slots as "free", and link them together. The first
	 * free slot has index mref_active, since the previous table was full.
	 *
	 * start -> start+1 -> ... -> new_size-1 -> END
	 */
	first_free = mref_active;
	for (i = first_free; i < new_size-1; i++)
		mref_table[i].xnext = 1 + ((i+1) << 1);
	mref_table[new_size-1].xnext = -1;

	/* Update the table size */
	mref_active = new_size;
}

mref_t mnode_to_mref (s_mnode *mn)
{
	mref_t slot;
	s_mnode **ptr;

	slot = mref_new();
	ptr = get_mnaddress(slot);
	unlink_mnode(*ptr);
	*ptr = copy_mnode(mn);
	return slot;
}

s_mnode* mref_to_mnode (mref_t mr)
{
	return copy_mnode(get_mnode(mr));
}

mref_t mref_new (void)
{
	static s_mnode *v = NULL;
	int mr;

	/* Get a free entry */
	if (first_free < 0)
		refill_mref_free_list();
	mr = first_free;
	first_free = mref_table[mr].xnext >> 1;
	/*
	 * If this is the first call to mref_new(), make sure that the
	 * library is initialized. Otherwise, the "Void" type won't exist
	 * and mnode_error() will die from infinite recursion.
	 */
	if (v == NULL) {
		saml_init();
		v = mnode_error(SE_EMPTY, "mref_new");
		++nb_mnodes_reserved;
	}
	mref_table[mr].node = copy_mnode(v);
	return mr;
}

void mref_free (mref_t mr)
{
	unlink_mnode(get_mnode(mr));
	mref_table[mr].xnext = 1 + (first_free << 1);
	first_free = mr;
}

char* mref_string (mref_t mr)
{
	gr_string *grs = mnode_stringify(get_mnode(mr));

	if (grs == NULL) {
		/* Bleah. */
		grs = new_gr_string(0);
		grs = grs_append(grs, "(null)", 6);
	}
	/* Make the string null-terminated */
	grs = grs_append1 (grs, '\0');

	free(alloc_ring[arindex]);
	alloc_ring[arindex] = grs;
	if (++arindex == ALLOC_RING_SIZE)
		arindex = 0;
	return grs->s;
}

int mref_type (mref_t mr)
{
	return get_mnode(mr)->type;
}

const char* mref_stype (mref_t mr)
{
	return mtype_table[mref_type(mr)]->name;
}

int mref_length (mref_t mr)
{
	s_mnode *mn = get_mnode(mr);
	return ((smn_ptr)mn)->length;
}

int mref_notzero (mref_t mr)
{
	return mnode_notzero(get_mnode(mr));
}

int mref_isneg (mref_t mr)
{
	return mnode_isneg(get_mnode(mr));
}

int mref_differ (mref_t mr1, mref_t mr2)
{
	return mnode_differ(get_mnode(mr1), get_mnode(mr2));
}

int mref_lessthan (mref_t mr1, mref_t mr2)
{
	return mnode_lessthan(get_mnode(mr1), get_mnode(mr2));
}

int mref_build (mref_t mr, int typeid, const char *str)
{
	s_mnode **p = get_mnaddress(mr);
	unlink_mnode(*p);
	return saml_check_mnode(*p = mnode_build(typeid,str));
}

int mref_cast (mref_t mr, int typeid)
{
	s_mnode **p = get_mnaddress(mr), *mn, *s;

	mn = *p;
	s = mnode_cast(mn, typeid);
	unlink_mnode(mn);
	return saml_check_mnode(*p = s);
}

int mref_promote (mref_t mr, mref_t mr1)
{
	s_mnode **p = get_mnaddress(mr), *mn, *s;

	mn = *p;
	s = mnode_promote(mn, get_mnode(mr1));
	unlink_mnode(mn);
	return saml_check_mnode(*p = s);
}

#define UNARY_OPERATION(opname)					\
  int mref_##opname (mref_t mr, mref_t mr1)			\
  {								\
	s_mnode **p = get_mnaddress(mr), *s;			\
	s = mnode_##opname (get_mnode(mr1));			\
	unlink_mnode(*p);					\
	return saml_check_mnode(*p = s);			\
  }

#define BINARY_OPERATION(opname)				\
  int mref_##opname (mref_t mr, mref_t mr1, mref_t mr2)		\
  {								\
	s_mnode **p = get_mnaddress(mr), *s;			\
	s = mnode_##opname (get_mnode(mr1),get_mnode(mr2));	\
	unlink_mnode(*p);					\
	return saml_check_mnode(*p = s);			\
  }

#define TERNARY_OPERATION(opname)				\
  int mref_##opname (mref_t mr, mref_t mr1, mref_t mr2, mref_t mr3) \
  {								\
	s_mnode **p = get_mnaddress(mr), *s;			\
	s = mnode_##opname (get_mnode(mr1),get_mnode(mr2),get_mnode(mr3)); \
	unlink_mnode(*p);					\
	return saml_check_mnode(*p = s);                        \
  }

UNARY_OPERATION(zero)
UNARY_OPERATION(negate)
UNARY_OPERATION(one)
UNARY_OPERATION(invert)
UNARY_OPERATION(sqrt)
UNARY_OPERATION(det)

BINARY_OPERATION(add)
BINARY_OPERATION(sub)
BINARY_OPERATION(mul)
BINARY_OPERATION(div)
BINARY_OPERATION(mod)
BINARY_OPERATION(gcd)
BINARY_OPERATION(diff)

TERNARY_OPERATION(subs)
TERNARY_OPERATION(elim)
TERNARY_OPERATION(move_lit)

void mref_copy (mref_t mr, mref_t mr1)
{
	s_mnode **p = get_mnaddress(mr), *s = copy_mnode(get_mnode(mr1));
	unlink_mnode(*p); *p = s;
}

void mref_swap (mref_t mr1, mref_t mr2)
{
	s_mnode **p = get_mnaddress(mr1);
	s_mnode **q = get_mnaddress(mr2);
	s_mnode *tmp;

	tmp = *p; *p = *q; *q = tmp;
}

int mref_power (mref_t mr, mref_t mr1, int power)
{
	s_mnode **p = get_mnaddress(mr), *s;

	s = mnode_power(get_mnode(mr1), power);
	unlink_mnode(*p);
	return saml_check_mnode(*p = s);
}

int mref_error (mref_t mr, int errno, const char *where)
{
	s_mnode **p = get_mnaddress(mr), *s;

	s = mnode_error(errno, where);
	unlink_mnode(*p);
	return saml_check_mnode(*p = s);
}

/*
 * The following functions (mref_array, mref_getitem, mref_setitem) allow
 * you to read and modify standard types (i.e. arrays) directly.
 * Use with care.
 */

int mref_array (mref_t mr, int typeid, int length)
{
	s_mnode **p = get_mnaddress(mr), *s;
	static s_mnode *v;
	std_mnode *array;
	int i;

	unlink_mnode(*p);
	if (!MTYPE_STD(typeid)) {
		/* Not a standard type */
		s = mnode_error(SE_NOT_ARRAY, "mref_array");
	} else if (length < 0) {
		/* Out of range */
		s = mnode_error(SE_WR_SIZE, "mref_array");
	} else {
		/* First call? */
		if (v == NULL) {
			v = mnode_error(SE_EMPTY, "mref_array");
			++nb_mnodes_reserved;
		}
		array = mstd_alloc(typeid, length);
		for (i = 0; i < length; i++)
			array->x[i] = copy_mnode(v);
		s = (s_mnode*) array;
	}
	return saml_check_mnode(*p = s);
}

int mref_getitem (mref_t mr, mref_t mr1, int i)
{
	s_mnode **p = get_mnaddress(mr);
	s_mnode *s = get_mnode(mr1);
	std_mnode *ss = (smn_ptr)s;

	if (!MTYPE_STD(s->type)) {
		/* Not a standard type */
		s = mnode_error(SE_NOT_ARRAY, "mref_getitem");
	} else if ((unsigned)i >= ss->length) {
		/* Index out of range */
		s = mnode_error(SE_INDEX, "mref_getitem");
	} else {
		/* Good */
		s = copy_mnode(ss->x[i]);
	}
	unlink_mnode(*p);
	return saml_check_mnode(*p = s);
}

int mref_setitem (mref_t mr, int i, mref_t mr1)
{
	s_mnode **p = get_mnaddress(mr), *mn = (*p), *s;
	std_mnode *a1 = (smn_ptr)mn, *a2;
	int len = a1->length, j;

	if (!MTYPE_STD(mn->type)) {
		/* Not a standard type */
		s = mnode_error(SE_NOT_ARRAY, "mref_setitem");
	} else if ((unsigned)i >= len) {
		/* Index out of range */
		s = mnode_error(SE_INDEX, "mref_setitem");
	} else {
		a2 = mstd_alloc(mn->type, len);
		for (j = 0; j < len; j++)
		    if (j != i)
			a2->x[j] = copy_mnode(a1->x[j]);
		/* But the i-th element is different */
		a2->x[i] = copy_mnode(get_mnode(mr1));
		s = (s_mnode*) a2;
	}
	unlink_mnode(*p);
	return saml_check_mnode(*p = s);
}
