/*
 * Copyright 1995,96,97 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: Upoly.c,v 2.4 1997/04/18 15:48:57 bousch Exp $
 *
 * Dense Univariate Polynomials
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "saml.h"
#include "saml-errno.h"
#include "mnode.h"
#include "builtin.h"

static gr_string* upoly_stringify (std_mnode*);
static s_mnode* upoly_make (s_mnode*);
static s_mnode* upoly_add (std_mnode*, std_mnode*);
static s_mnode* upoly_sub (std_mnode*, std_mnode*);
static s_mnode* upoly_mul (std_mnode*, std_mnode*);
static s_mnode* upoly_div (std_mnode*, std_mnode*);
static s_mnode* upoly_gcd (std_mnode*, std_mnode*);
static int upoly_notzero (std_mnode*);
static int upoly_differ (std_mnode*, std_mnode*);
static s_mnode* upoly_zero (std_mnode*);
static s_mnode* upoly_negate (std_mnode*);
static s_mnode* upoly_one (std_mnode*);
static s_mnode* upoly_invert (std_mnode*);
static s_mnode* upoly2upoly (std_mnode*, std_mnode*);
s_mnode* upoly_diff (std_mnode*);
s_mnode* upoly_eval (std_mnode*, s_mnode*);
s_mnode* upoly_sylvester (std_mnode*, std_mnode*);

static unsafe_s_mtype MathType_Upoly = {
	"Univariate Polynomial",
	mstd_free, NULL, upoly_stringify,
	upoly_make, NULL,
	upoly_add, upoly_sub, upoly_mul, upoly_div, upoly_gcd,
	upoly_notzero, NULL, NULL, upoly_differ, NULL,
	upoly_zero, upoly_negate, upoly_one, upoly_invert, NULL
};

void init_MathType_Upoly (void)
{
	register_mtype(ST_UPOLY, &MathType_Upoly);
	register_CV_routine(ST_UPOLY, ST_UPOLY, upoly2upoly);
}

static s_mnode* upoly_zero (std_mnode* mn1)
{
	std_mnode *mn = mstd_alloc(ST_UPOLY, 1);
	mn->x[0] = mnode_zero(mn1->x[0]);
	return (mn_ptr)mn;
}

static s_mnode* upoly_one (std_mnode* mn1)
{
	std_mnode *mn = mstd_alloc(ST_UPOLY, 1);
	mn->x[0] = mnode_one(mn1->x[0]);
	return (mn_ptr)mn;
}

static s_mnode* upoly_make (s_mnode* constant)
{
	std_mnode *mn = mstd_alloc(ST_UPOLY, 1);
	mn->x[0] = copy_mnode(constant);
	return (mn_ptr)mn;
}

static s_mnode* upoly_negate (std_mnode* mn1)
{
	std_mnode *mn;
	int i, length;

	length = mn1->length;
	mn = mstd_alloc(ST_UPOLY, length);
	for (i = 0; i < length; i++)
		mn->x[i] = mnode_negate(mn1->x[i]);
	return (mn_ptr)mn;
}

static s_mnode* upoly_invert (std_mnode* mn1)
{
	std_mnode *mn;
	s_mnode *tmp;
	
	if (mn1->length > 1)
		return mnode_error(SE_SINGULAR, "upoly_invert");
	tmp = mnode_invert(mn1->x[0]);
	if (tmp->type == ST_VOID)
		return tmp;
	mn = mstd_alloc(ST_UPOLY, 1);
	mn->x[0] = tmp;
	return (s_mnode*) mn;
}

static s_mnode* upoly2upoly (std_mnode* mn1, std_mnode* model)
{
	std_mnode *mn;
	int i, length;

	if (!model)
		return copy_mnode((mn_ptr)mn1);
	length = mn1->length;
	mn = mstd_alloc(ST_UPOLY, length);
	for (i = 0; i < length; i++)
		mn->x[i] = mnode_promote(mn1->x[i], model->x[0]);
	return (mn_ptr)mn;
}

static int upoly_notzero (std_mnode* mn1)
{
	return (mn1->length > 1) || mnode_notzero(mn1->x[0]);
}

static int upoly_differ (std_mnode* mn1, std_mnode* mn2)
{
	int i, length;

	if ((length = mn1->length) != mn2->length)
		return 1;
	for (i = 0; i < length; i++)
		if (mnode_differ(mn1->x[i],mn2->x[i]))
			return 1;
	return 0;
}

static gr_string* upoly_stringify (std_mnode* mn1)
{
	gr_string *grs, *grterm;
	int i;

	grs = new_gr_string(0);
	for (i = 0; i < mn1->length; i++) {
		grs = grs_append1(grs, i? ',' : '(');
		grterm = mnode_stringify(mn1->x[i]);
		grs = grs_append(grs, grterm->s, grterm->len);
		free(grterm);
	}
	grs = grs_append1(grs, ')');
	return grs;
}

static s_mnode* upoly_add (std_mnode* mn1, std_mnode* mn2)
{
	int len1, len2, terms, i;
	s_mnode **tmp;
	std_mnode *mn;

	len1 = mn1->length;
	len2 = mn2->length;
	if (len1 > len2) {
		mn = mstd_alloc(ST_UPOLY, len1);
		for (i = 0; i < len2; i++)
			mn->x[i] = mnode_add(mn1->x[i],mn2->x[i]);
		for (i = len2; i < len1; i++)
			mn->x[i] = copy_mnode(mn1->x[i]);
		return (mn_ptr)mn;
	}
	if (len1 < len2) {
		mn = mstd_alloc(ST_UPOLY, len2);
		for (i = 0; i < len1; i++)
			mn->x[i] = mnode_add(mn1->x[i],mn2->x[i]);
		for (i = len1; i < len2; i++)
			mn->x[i] = copy_mnode(mn2->x[i]);
		return (mn_ptr)mn;
	}
	/* Here the two polynomials have the same degree */
	terms = len1;
	tmp = alloca(terms * sizeof(s_mnode*));
	for (i = 0; i < terms; i++)
		tmp[i] = mnode_add(mn1->x[i], mn2->x[i]);
	while (terms > 1 && !mnode_notzero(tmp[terms-1]))
		unlink_mnode(tmp[--terms]);
	mn = mstd_alloc(ST_UPOLY, terms);
	memcpy(mn->x, tmp, terms * sizeof(s_mnode*));
	return (mn_ptr)mn;
}

static s_mnode* upoly_sub (std_mnode* mn1, std_mnode* mn2)
{
	int len1, len2, terms, i;
	s_mnode **tmp;
	std_mnode *mn;

	len1 = mn1->length;
	len2 = mn2->length;
	if (len1 > len2) {
		mn = mstd_alloc(ST_UPOLY, len1);
		for (i = 0; i < len2; i++)
			mn->x[i] = mnode_sub(mn1->x[i], mn2->x[i]);
		for (i = len2; i < len1; i++)
			mn->x[i] = copy_mnode(mn1->x[i]);
		return (mn_ptr)mn;
	}
	if (len1 < len2) {
		mn = mstd_alloc(ST_UPOLY, len2);
		for (i = 0; i < len1; i++)
			mn->x[i] = mnode_sub(mn1->x[i], mn2->x[i]);
		for (i = len1; i < len2; i++)
			mn->x[i] = mnode_negate(mn2->x[i]);
		return (mn_ptr)mn;
	}
	/* Here the two polynomials have the same degree */
	terms = len1;
	tmp = alloca(terms * sizeof(s_mnode*));
	for (i = 0; i < terms; i++)
		tmp[i] = mnode_sub(mn1->x[i], mn2->x[i]);
	while (terms > 1 && !mnode_notzero(tmp[terms-1]))
		unlink_mnode(tmp[--terms]);
	mn = mstd_alloc(ST_UPOLY, terms);
	memcpy(mn->x, tmp, terms * sizeof(s_mnode*));
	return (mn_ptr)mn;
}

static s_mnode* upoly_mul (std_mnode* mn1, std_mnode* mn2)
{
	int len1, len2, i, j, k, terms;
	s_mnode **tmp, *term, *new;
	std_mnode *mn;

	len1 = mn1->length;
	len2 = mn2->length;
	if (len1 == 1 && !mnode_notzero(mn1->x[0])) {
		/* mn1 is zero */
		return copy_mnode((mn_ptr)mn1);
	}
	if (len2 == 1 && !mnode_notzero(mn2->x[0])) {
		/* mn2 is zero */
		return copy_mnode((mn_ptr)mn2);
	}
	terms = len1 + len2 - 1;
	tmp = alloca(terms * sizeof(s_mnode*));
	tmp[0] = mnode_zero(mn1->x[0]);
	for (k = 1; k < terms; k++)
		tmp[k] = copy_mnode(tmp[0]);
	for (i = 0; i < len1; i++)
		for (j = 0; j < len2; j++) {
			k = i + j;
			term = mnode_mul(mn1->x[i], mn2->x[j]);
			new = mnode_add(tmp[k], term);
			unlink_mnode(term);
			unlink_mnode(tmp[k]);
			tmp[k] = new;
		}
	while (terms > 0 && !mnode_notzero(tmp[terms-1]))
		unlink_mnode(tmp[--terms]);
	mn = mstd_alloc(ST_UPOLY, terms);
	memcpy(mn->x, tmp, terms * sizeof(s_mnode*));
	return (mn_ptr)mn;
}

static void upoly_pseudo_div (std_mnode *U, std_mnode *V, std_mnode **quot,
	std_mnode **rem)
{
	s_mnode *LV, **q, **r, *fct, *n1, *n2, *n3;
	int j, k, lu, lv, terms;

	lu = U->length; lv = V->length;
	assert(lu >= lv);
	LV = V->x[lv-1];
	q = alloca((lu - lv + 1) * sizeof(s_mnode*));
	r = alloca(lu * sizeof(s_mnode*));
	for (j = 0; j < lu; j++)
		r[j] = copy_mnode(U->x[j]);

	for (k = lu - lv; k >= 0; k--) {
		fct = q[k] = copy_mnode(r[lv-1+k]);
		for (j = lv+k-2; j >= 0; j--) {
			n1 = mnode_mul(LV, r[j]);
			if (j >= k) {
				n2 = mnode_mul(fct, V->x[j-k]);
				n3 = mnode_sub(n1, n2);
				unlink_mnode(n1); unlink_mnode(n2);
				n1 = n3;
			}
			unlink_mnode(r[j]);
			r[j] = n1;
		}
	}
	/* Is someone interested in the quotient? */
	if (quot) {
		terms = lu - lv + 1;
		*quot = mstd_alloc(ST_UPOLY, terms);
		(*quot)->x[0] = copy_mnode(q[0]);
		n3 = mnode_one(LV);
		for (k = 1; k < terms; k++) {
			n1 = mnode_mul(n3, LV);
			unlink_mnode(n3);
			n3 = n1;
			(*quot)->x[k] = mnode_mul(q[k], n3);
		}
		unlink_mnode(n3);
	}
	/* Free the array pointed to by "q" */
	for (k = 0; k <= lu - lv; k++)
		unlink_mnode(q[k]);
	/* Is someone interested in the remainder? (I hope so) */
	if (rem) {
		terms = lv - 1;
		while (terms > 0 && !mnode_notzero(r[terms-1]))
			--terms;
		if (terms) {
			*rem = mstd_alloc(ST_UPOLY, terms);
			for (k = 0; k < terms; k++)
				(*rem)->x[k] = copy_mnode(r[k]);
		} else
			*rem = (smn_ptr)upoly_zero(U);
	}
	/* Free the array pointed to by "r" */
	for (k = 0; k < lu; k++)
		unlink_mnode(r[k]);
}

static std_mnode* upoly_scalar_div (std_mnode* up1, s_mnode* scalar)
{
	int i, terms;
	std_mnode *quot;
	s_mnode **tmp;

	if (!mnode_notzero(scalar))
		return (smn_ptr)mnode_error(SE_DIVZERO, "upoly_scalar_div");
	terms = up1->length;
	tmp = alloca(terms * sizeof(s_mnode*));
	for (i = 0; i < terms; i++)
		tmp[i] = mnode_div(up1->x[i], scalar);
	while (terms > 0 && !mnode_notzero(tmp[terms-1]))
		unlink_mnode(tmp[--terms]);
	quot = mstd_alloc(ST_UPOLY, terms);
	memcpy(quot->x, tmp, terms * sizeof(s_mnode*));
	return quot;
}

static std_mnode* upoly_scalar_mul (std_mnode* up1, s_mnode* scalar)
{
	int i, terms;
	std_mnode *quot;
	s_mnode **tmp;

	if (!mnode_notzero(scalar))
		return (smn_ptr)upoly_zero(up1);
	terms = up1->length;
	tmp = alloca(terms * sizeof(s_mnode*));
	for (i = 0; i < terms; i++)
		tmp[i] = mnode_mul(up1->x[i], scalar);
	while (terms > 0 && !mnode_notzero(tmp[terms-1]))
		unlink_mnode(tmp[--terms]);
	quot = mstd_alloc(ST_UPOLY, terms);
	memcpy(quot->x, tmp, terms * sizeof(s_mnode*));
	return quot;
}

static s_mnode* upoly_div (std_mnode* mn1, std_mnode* mn2)
{
	std_mnode *quot1, *quot;
	s_mnode *LV, *LV2;

	LV = mn2->x[mn2->length - 1];
	if (!mnode_notzero(LV))
		return mnode_error(SE_DIVZERO, "upoly_div");
	if (mn1->length < mn2->length)
		return upoly_zero(mn1);
	LV2 = mnode_power(LV, (mn1->length) - (mn2->length) + 1);
	upoly_pseudo_div(mn1, mn2, &quot1, NULL);
	quot = upoly_scalar_div(quot1, LV2);
	unlink_mnode((mn_ptr)quot1);
	unlink_mnode(LV2);
	return (mn_ptr)quot;
}

s_mnode* upoly_eval (std_mnode* poly, s_mnode* x)
{
	s_mnode *ev, *tmp;
	int i, terms;

	if (!mnode_notzero(x))
		return copy_mnode(poly->x[0]);
	terms = poly->length;
	ev = copy_mnode(poly->x[terms-1]);
	for (i = terms-2; i >= 0; i--) {
		tmp = mnode_mul(ev, x);
		unlink_mnode(ev);
		ev = mnode_add(poly->x[i], tmp);
		unlink_mnode(tmp);
	}
	return ev;
}

static s_mnode* upoly_cont (std_mnode *mn1)
{
	int i, terms;
	s_mnode *n1, *n2;

	terms = mn1->length;
	n1 = copy_mnode(mn1->x[0]);
	for (i = 1; i < terms; i++) {
		n2 = mnode_gcd(mn1->x[i], n1);
		unlink_mnode(n1);
		n1 = n2;
	}
	return n1;
}

static s_mnode* upoly_gcd (std_mnode *mn1, std_mnode *mn2)
{
	s_mnode *d, *cu, *cv, *a, *b;
	std_mnode *r, *pu, *pv, *tmp;

	if (!upoly_notzero(mn1)) {
		/* mn1 is zero */
		return copy_mnode((mn_ptr)mn2);
	}
	if (!upoly_notzero(mn2)) {
		/* mn2 is zero */
		return copy_mnode((mn_ptr)mn1);
	}
	cu = upoly_cont(mn1);
	pu = upoly_scalar_div(mn1, cu);
	cv = upoly_cont(mn2);
	pv = upoly_scalar_div(mn2, cv);
	d = mnode_gcd(cu, cv);
	unlink_mnode(cu); unlink_mnode(cv);
	a = mnode_one(d);
	if (pu->length < pv->length)
		tmp = pu, pu = pv, pv = tmp;
	while(1) {
		b = mnode_power(pv->x[pv->length - 1],
			(pu->length) - (pv->length) + 1);
		upoly_pseudo_div(pu, pv, NULL, &r);
		if (r->length == 1) {
			if (mnode_notzero(r->x[0])) {
				unlink_mnode((mn_ptr)pv);
				pv = (smn_ptr)upoly_one(pu);
			} else {
				cv = upoly_cont(pv);
				tmp = upoly_scalar_div(pv, cv);
				unlink_mnode((mn_ptr)cv);
				unlink_mnode((mn_ptr)pv);
				pv = tmp;
			}
			unlink_mnode(a); unlink_mnode(b);
			unlink_mnode((mn_ptr)r);
			unlink_mnode((mn_ptr)pu);
			tmp = upoly_scalar_mul(pv, d);
			unlink_mnode(d);
			unlink_mnode((mn_ptr)pv);
			return (mn_ptr)tmp;
		}
		unlink_mnode((mn_ptr)pu);
		pu = pv;
		tmp = upoly_scalar_div(r, a);
		unlink_mnode(a);
		unlink_mnode((mn_ptr)r);
		pv = tmp;
		a = b;
	}
}

s_mnode* upoly_sylvester (std_mnode *p1, std_mnode *p2)
{
	int i, j, d1, d2, dd;
	s_mnode *zero, *dmat;
	std_mnode *m, *mi;

	zero = mnode_zero(p1->x[0]);
	if (!upoly_notzero(p1) || !upoly_notzero(p2))
		return zero;
	d1 = p1->length - 1;
	d2 = p2->length - 1;
	dd = d1 + d2;
	m = mstd_alloc(ST_MATRIX, dd);
	for (i = 0; i < d2; i++) {
		mi = mstd_alloc(ST_LINE, dd);
		m->x[i] = (mn_ptr)mi;
		for (j = 0; j < i; j++)
			mi->x[j] = copy_mnode(zero);
		for (j = 0; j <= d1; j++)
			mi->x[j+i] = copy_mnode(p1->x[j]);
		for (j = 1; j < d2-i; j++)
			mi->x[j+i+d1] = copy_mnode(zero);
	}
	for (i = 0; i < d1; i++) {
		mi = mstd_alloc(ST_LINE, dd);
		m->x[i+d2] = (mn_ptr)mi;
		for (j = 0; j < i; j++)
			mi->x[j] = copy_mnode(zero);
		for (j = 0; j <= d2; j++)
			mi->x[j+i] = copy_mnode(p2->x[j]);
		for (j = 1; j < d1-i; j++)
			mi->x[j+i+d2] = copy_mnode(zero);
	}
	dmat = mnode_det((mn_ptr)m);
	unlink_mnode(zero);
	unlink_mnode((mn_ptr)m);
	return dmat;
}

s_mnode* upoly_diff (std_mnode *p)
{
	s_mnode *f1, *f2, **qtmp;
	std_mnode *q;
	char buff[12];
	int i, len, terms;

	if ((len = p->length) <= 1)
		return upoly_zero(p);
	terms = len - 1;
	qtmp = (s_mnode **)alloca (terms * sizeof(s_mnode*));
	for (i = 0; i < terms; i++) {
		sprintf(buff, "%d", i+1);
		f1 = mnode_build(ST_INTEGER, buff);
		f2 = mnode_promote(f1, p->x[0]);
		unlink_mnode(f1);
		qtmp[i] = mnode_mul(f2, p->x[i+1]);
		unlink_mnode(f2);
	}
	/*
	 * On a field of characteristic zero, it is guaranteed that
	 * deg(dp/dx) = deg(p)-1, but this is not necessarily
	 * the case here.
	 */
	while (terms > 0 && !mnode_notzero(qtmp[terms-1]))
		unlink_mnode(qtmp[--terms]);
	q = mstd_alloc(ST_UPOLY, terms);
	memcpy(q->x, qtmp, terms * sizeof(s_mnode*));
	return (mn_ptr)q;
}
