/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <ctype.h>
#include <math.h>
#include <float.h>
#include <stdlib.h>
#include "approx.h"
#include "common.h"
#include "error.h"
#include "julcal/julcal.h"
#include "misc.h"
#include "str.h"
#include "misc.h"
#include "settings.h"
#include "var.h"

#undef DEBUGGING
/*#define DEBUGGING 1*/
#include "debug-print.h"

#if FLT_RADIX!=2
#error Write your own floating-point output routines.
#endif

/* PORTME:

   Some of the routines in this file are likely very specific to
   base-2 representation of floating-point number, most notably the
   routines that use frexp() or ldexp().  These attempt to extract
   individual digits by setting the base-2 exponent and
   multiplying/dividing by powers of 2.  In base-2 numeration systems,
   this just nudges the exponent up or down, but in base-10 floating
   point, such multiplications/division can cause catastrophic loss of
   precision.

   The author has never personally used a machine that didn't use
   binary floating point formats, so he is unwilling, and perhaps
   unable, to code around this "problem".  */

/* In older versions, numbers got their trailing zeros stripped.
   Newer versions leave them on when there's room.  Comment this next
   line out if you wanna go retro. */
#define NEW_STYLE 1

/* Scratch buffer for formatting. */
static char buf[128];

/* Converts a number between 0 and 15 inclusive to a `hexit'
   [0-9A-F]. */
static inline int
make_hexit (int x)
{
  /* PORTME: This routine might be broken on non-ASCII character sets. */
  return (x < 10) ? (x + '0') : (x - 10 + 'A');
}

static void
convert_E (const fmt_spec *fp, double v)
{
  /* There's no easy, obvious, precise, portable way to do scientific
     notation without postprocessing the output of sprintf.  So that's
     what we'll do.  */

  /* Actual number of decimal places.  PSPP allows the user to input
     an impossible number, so this is a sanity check. */
  int d;

  d = min (fp->d, fp->w - 6);
  if (v < 0)
    d--;
  if (d < 0)
    d = 0;
  sprintf (buf, "%*.*E", fp->w, d, v);

  /* What we do here is force the exponent part to have four
     characters whenever possible.  That is, 1.00E+99 is okay (`E+99')
     but 1.00E+100 (`E+100') must be coerced to 1.00+100 (`+100').  On
     the other hand, 1.00E1000 (`E+100') cannot be canonicalized.
     Note that ANSI C guarantees at least two digits in the
     exponent. */
  if (fabs (v) > 1e99)
    {
      /* Pointer to the `E' in buf. */
      char *cp;

      /* Base-10 exponent value of v */
      int exp;

      cp = strchr (buf, 'E');
      if (cp)
	exp = atoi (cp + 1);	/* screw you if your exponent's bigger'n an int */
      if (cp && abs (exp) > 99 && abs (exp) < 1000)
	{
	  /* shift everything left one place: 1.00e+100 -> 1.00+100 */
	  cp[0] = cp[1];
	  cp[1] = cp[2];
	  cp[2] = cp[3];
	  cp[3] = cp[4];
	}
      else if (abs (exp) > 1000)
	memset (buf, '*', fp->w);
    }

  /* The C locale always uses a period `.' as a decimal point.
     Translate that to a comma `,' for the benefit of our foreign
     friends. */
  if ((set_decimal == ',' && fp->type != FMT_DOT)
      || (set_decimal == '.' && fp->type == FMT_DOT))
    {
      char *cp = strchr (buf, '.');
      if (cp)
	*cp = ',';
    }
}

/* I played with different implementations of this routine for more
   than a week, 'til I was blue in the face.  Let's instead rely on
   the underlying implementation of sprintf:

   1. If the number has a magnitude 1e40 or greater, then we
   needn't bother with it, since it's guaranteed to need processing
   in scientific notation.

   2. Otherwise, do a binary search for the base-10 magnitude of
   the thing.  log10() is not accurate enough, and the alternatives
   are frightful.  Besides, we never need as many as 6 (pairs of)
   comparisons.

   3. The algorithm used for searching is Knuth's Algorithm 6.2.1C
   (Uniform binary search).  Look it up in _The Art of Computer
   Programming_--if you don't own it, why not?  You should!

   DON'T CHANGE ANYTHING HERE UNLESS YOU'VE THOUGHT ABOUT IT FOR A
   LONG TIME!  The rest of the program is heavily dependent on
   specific properties of this routine's output.  LOG ALL CHANGES!

   FIXME?  Another possible implementation would simply, after testing
   for magnitude > 1e40, sprintf with %f, then postprocess.  Might be
   _a_lot_ simpler, but there are many pitfalls in this routine. */
static void
convert_F (const fmt_spec *fp, double v)
{
  /* Used for determining whether v is between two powers of 10.
     This is Ki, 1<=i<=40, from Knuth. */
  static const double tab[] =
  {
    0,
    1e01, 1e02, 1e03, 1e04, 1e05, 1e06, 1e07, 1e08, 1e09, 1e10,
    1e11, 1e12, 1e13, 1e14, 1e15, 1e16, 1e17, 1e18, 1e19, 1e20,
    1e21, 1e22, 1e23, 1e24, 1e25, 1e26, 1e27, 1e28, 1e29, 1e30,
    1e31, 1e32, 1e33, 1e34, 1e35, 1e36, 1e37, 1e38, 1e39, 1e40,
  };

  /* This is the DELTA array from Knuth.
     DELTA[j] = floor((40+2**(j-1))/(2**j)). */
  static const int delta[8] =
  {
    0, (40 + 1) / 2, (40 + 2) / 4, (40 + 4) / 8, (40 + 8) / 16,
    (40 + 16) / 32, (40 + 32) / 64, (40 + 64) / 128,
  };

  /* The number of digits in floor(v), including sign.  This is `i'
     from Knuth. */
  int nint = (40 + 1) / 2;

  /* Used to step through delta[].  This is `j' from Knuth. */
  int j = 2;

  /* Magnitude of v.  This is `K' from Knuth. */
  double mag;

  /* Number of characters for the fractional part, including the
     decimal point. */
  int ndec;

  /* Pointer into buf used for formatting. */
  char *cp;

  /* Used to count characters formatted by nsprintf(). */
  int n;

#if DEBUGGING
  static int seq = 0;
  seq++;
#endif

  /* First check for infinities and NaNs.  12/13/96. */
  if (!finite (v))
    {
      n = nsprintf (buf, "%f", v);
      if (n == fp->w)
	return;
      else if (n > fp->w)
	memset (buf, '*', fp->w);
      else
	{
	  memmove (&buf[fp->w - n], buf, n);
	  memset (buf, ' ', fp->w - n);
	}
      return;
    }

  /* Then check for radically out-of-range values. */
  mag = fabs (v);
  if (mag >= tab[fp->w])
    goto scientific;

  if (mag < 1.0)
    {
      nint = 0;

      /* Avoid printing `-.000'. 7/6/96. */
      if (approx_eq (v, 0.0))
	v = 0.0;
    }
  else
    /* Now perform a `uniform binary search' based on the tables tab[]
       and delta[].  After this step, nint is the number of digits in
       floor(v), including any sign.  */
    while (1)
      {
	if (mag >= tab[nint])	/* Should this be approx_ge()? */
	  {
	    assert (delta[j]);
	    nint += delta[j++];
	  }
	else if (mag < tab[nint - 1])
	  {
	    assert (delta[j]);
	    nint -= delta[j++];
	  }
	else
	  break;
      }

  /* If we have any decimal places, then there is a decimal point,
     too. */
  ndec = fp->d;
  if (ndec)
    ndec++;

  /* 1/10/96: If there aren't any digits at all, add one.  This occurs
     only when fabs(v) < 1.0. */
  if (nint + ndec == 0)
    nint++;

  /* Give space for a minus sign.  Moved 1/10/96. */
  if (v < 0)
    nint++;

  /* Normally we only go through the loop once; occasionally twice.
     Three times or more indicates a very serious bug somewhere. */
  while (1)
    {
      /* Check out the total length of the string. */
      cp = buf;
      if (nint + ndec > fp->w)
	{
	  /* The string is too long.  Let's see what can be done. */
	  if (nint <= fp->w)
	    /* If we can, just reduce the number of decimal places. */
	    ndec = fp->w - nint;
	  else
	    goto scientific;
	}
      else if (nint + ndec < fp->w)
	{
	  /* The string is too short.  Left-pad with spaces. */
	  int n_spaces = fp->w - nint - ndec;
	  memset (cp, ' ', n_spaces);
	  cp += n_spaces;
	}

      /* Finally, format the number. */
      /* 22 April 1997: There appears to be a serious bug in the
	 Checker 0.8 libc that causes lotsa spew and eventually a
	 segfault when very large numbers are printed in %f format.
	 How can this be accounted for? */
      if (ndec)
	n = nsprintf (cp, "%.*f", ndec - 1, v);
      else
	n = nsprintf (cp, "%.0f", v);

      /* If v is positive and its magnitude is less than 1...  */
      if (nint == 0)
	{
	  if (*cp == '0')
	    {
	      /* The value rounds to `.###'. */
	      memmove (cp, &cp[1], n - 1);
	      n--;
	    }
	  else
	    {
	      /* The value rounds to `1.###'. */
	      nint = 1;
	      continue;
	    }
	}
      /* Else if v is negative and its magnitude is less than 1...  */
      else if (v < 0 && nint == 1)
	{
	  if (cp[1] == '0')
	    {
	      /* The value rounds to `-.###'. */
	      memmove (&cp[1], &cp[2], n - 2);
	      n--;
	    }
	  else
	    {
	      /* The value rounds to `-1.###'. */
	      nint = 2;
	      continue;
	    }
	}

#if DEBUGGING
      if (n != nint + ndec)
	printf ("%d:data-out:convert_F(): v=%.20f cp=\"%s\" nint=%d ndec=%d\n",
		seq, v, cp, nint, ndec);
#endif

      /* Check for a correct number of digits & decimal places & stuff.
         This is just a desperation check.  Hopefully it won't fail too
         often, because then we have to run through the whole loop again:
         sprintf() is not a fast operation with floating-points! */
      if (n == nint + ndec)
	{
	  /* Convert periods `.' to commas `,' for our foreign friends. */
	  if ((set_decimal == ',' && fp->type != FMT_DOT)
	      || (set_decimal == '.' && fp->type == FMT_DOT))
	    {
	      cp = strchr (cp, '.');
	      if (cp)
		*cp = ',';
	    }
	  return;
	}

      nint = n - ndec;		/* FIXME?  Need an idiot check on resulting nint? */
    }

  /* The rest of the function is normally unreachable code except by
     goto's. */

scientific:
  /* We arrive at this point if there is not enough room in the string
     to print it without scientific notation.  So test if the string
     is long enough to handle scientific notation: */
  if (fp->w >= 6)
    {
      /* Go scientific if there's room. */
      fmt_spec f;
      f.type = fp->type;
      f.w = fp->w;
      f.d = f.w - 6;
      convert_E (&f, v);
      return;
    }
  else
    {
      /* If there's not, we're out of luck. */
      memset (buf, '*', fp->w);
      return;
    }
}

/* Copies buf[] to S, inserting commas and dollar signs as appropriate
   for format spec *FP.  */
static inline void
insert_commas (char *s, const fmt_spec *fp)
{
  /* Number of leading spaces in the number.  This is the amount of
     room we have for inserting commas and dollar signs. */
  int nspaces;

  /* Number of digits before the decimal point.  This is used to
     determine the number of commas to insert. */
  int ndigits;

  /* Number of commas to insert. */
  int ncommas;

  /* Number of items (commas and dollar signs) to insert. */
  int nitems;

  /* Digit iterator. */
  int i;

  /* Destination pointer. */
  char *dp = s;

  /* Source pointer. */
  char *sp = buf;

  /* Count spaces and digits. */
  nspaces = strspn (buf, " ");
  sp = &buf[nspaces];
  if (*sp == '-')
    sp++;
  ndigits = strspn (sp, "0123456789");
  ncommas = (ndigits - 1) / 3;
  nitems = ncommas + (fp->type == FMT_DOLLAR);

  if (!nspaces)
    {
      strcpy (s, buf);
      return;
    }
  if (nitems > nspaces)
    {
      nitems -= ncommas;
      if (!nitems)
	{
	  strcpy (s, buf);
	  return;
	}
    }
  if (nspaces > nitems)
    {
      memset (dp, ' ', nspaces - nitems);
      dp += nspaces - nitems;
    }
  if (fp->type == FMT_DOLLAR)
    {
      *dp++ = '$';
      nitems--;
    }
  if (sp - buf > nspaces)
    *dp++ = '-';
  for (i = ndigits; i; i--)
    {
      if (i % 3 == 0 && ndigits > i && nitems)
	{
	  nitems--;
	  *dp++ = fp->type == FMT_COMMA ? set_grouping : set_decimal;
	}
      *dp++ = *sp++;
    }
  memcpy (dp, sp, strlen (sp));	/* Avoid copying a terminating null. */
}

static inline void
convert_PIBHEX (char *s, const fmt_spec *fp, const value *v)
{
  /* Strategy: Use frexp() to create a normalized result (but mostly
     to find the base-2 exponent), then change the base-2 exponent to
     (-4*fp->w) using multiplication and division by powers of two.
     Extract each hexit by multiplying by 16. */

  /* Fraction (mantissa). */
  double frac;

  /* Exponent. */
  int exp;

  /* Difference between exponent and (-4*fp->w). */
  int diff;

  /* Counter. */
  int i;

  /* Make the exponent (-4*fp->w). */
  frac = frexp (fabs (v->f), &exp);
  diff = exp - (-4 * fp->w);
  exp -= diff;
  frac *= ldexp (1.0, diff);

  /* Extract each hexit. */
  for (i = 0; i < fp->w; i++)
    {
      modf (frac, &frac);
      frac *= 16.0;
      *s++ = make_hexit (floor (frac));
    }
}

static inline void
convert_RBHEX (char *s, const fmt_spec *fp, const value *v)
{
  union
  {
    double d;
    unsigned char c[8];
  }
  u;

  int i;

  u.d = v->f;
  for (i = 0; i < fp->w / 2; i++)
    {
      *s++ = make_hexit (u.c[i] >> 4);
      *s++ = make_hexit (u.c[i] & 15);
    }
}

static inline void
convert_Z (char *s, const fmt_spec *fp, const value *v)
{
  static int warned = 0;
  double d = fabs (floor (v->f));
  int i;

  if (!warned)
    {
      msg (MW, _("Quality of zoned decimal (Z) output format code is suspect.  "
	   "Check your results three times, report bugs to author."));
      warned = 1;
    }

  if (d == SYSMIS)
    {
      msg (ME, _("The system-missing value cannot be output as a zoned "
	   "decimal number."));
      return;
    }

  if (d < pow (10.0, fp->w))
    sprintf (s, "%*.0f", fp->w, v->f);
  else
    {
      msg (ME, _("Number %g too big to fit in field with format Z%d.%d."),
	   v->f, fp->w, fp->d);
      return;
    }

  for (i = 0; i < fp->w; i++)
    s[i] = (s[i] - '0') | 0xf0;
  if (v->f < 0)
    s[fp->w - 1] &= 0xdf;
}

static inline void
convert_IB (char *s, const fmt_spec *fp, const value *v)
{
  /* Strategy: Basically the same as convert_PIBHEX() but with base
     256. Then it's necessary to negate the two's-complement result if
     v->f is negative. */

  /* Used for constructing the two's-complement result. */
  unsigned temp[8];

  /* Fraction (mantissa). */
  double frac;

  /* Exponent. */
  int exp;

  /* Difference between exponent and (-8*fp->w-1). */
  int diff;

  /* Counter. */
  int i;

  /* Make the exponent (-8*fp->w-1). */
  frac = frexp (fabs (v->f), &exp);
  diff = exp - (-8 * fp->w - 1);
  exp -= diff;
  frac *= ldexp (1.0, diff);

  /* Extract each base-256 digit. */
  for (i = 0; i < fp->w; i++)
    {
      modf (frac, &frac);
      frac *= 256.0;
      temp[i] = floor (frac);
    }

  /* Perform two's-complement negation if v->f is negative. */
  if (v->f < 0)
    {
      /* Perform NOT operation. */
      for (i = 0; i < fp->w; i++)
	temp[i] = ~temp[i];
      /* Add 1 to the whole number. */
      for (i = fp->w - 1; i >= 0; i--)
	{
	  temp[i]++;
	  if (temp[i])
	    break;
	}
    }
  memcpy (s, temp, fp->w);
  if (endian == LITTLE)
    memrev (s, fp->w);
}

static inline void
convert_PIB (char *s, const fmt_spec *fp, const value *v)
{
  /* Strategy: Basically the same as convert_IB(). */

  /* Fraction (mantissa). */
  double frac;

  /* Exponent. */
  int exp;

  /* Difference between exponent and (-8*fp->w). */
  int diff;

  /* Counter. */
  int i;

  /* Make the exponent (-8*fp->w). */
  frac = frexp (fabs (v->f), &exp);
  diff = exp - (-8 * fp->w);
  exp -= diff;
  frac *= ldexp (1.0, diff);

  /* Extract each base-256 digit. */
  for (i = 0; i < fp->w; i++)
    {
      modf (frac, &frac);
      frac *= 256.0;
      ((unsigned char *) s)[i] = floor (frac);
    }
  if (endian == LITTLE)
    memrev (s, fp->w);
}

/* PORTME: May need to be adjusted with odd non-ASCII charsets. */
static inline int
xtract_digit (int ch)
{
  return ch - '0';
}

static inline void
convert_P (char *s, const fmt_spec *fp, const value *v)
{
  /* Buffer for v->f*2-1 characters + a decimal point if library is
     not quite compliant + a null. */
  char buf[17];

  /* Counter. */
  int i;

  /* Main extraction. */
  sprintf (buf, "%0*.0f", fp->w * 2 - 1, floor (fabs (v->f)));

  for (i = 0; i < fp->w; i++)
    ((unsigned char *) s)[i]
      = (xtract_digit (buf[i * 2]) << 4) + xtract_digit (buf[i * 2 + 1]);

  /* Set sign. */
  s[fp->w - 1] &= 0xf0;
  if (v->f >= 0.0)
    s[fp->w - 1] |= 0xf;
  else
    s[fp->w - 1] |= 0xd;
}

static inline void
convert_PK (char *s, const fmt_spec *fp, const value *v)
{
  /* Buffer for v->f*2 characters + a decimal point if library is not
     quite compliant + a null. */
  char buf[18];

  /* Counter. */
  int i;

  /* Main extraction. */
  sprintf (buf, "%0*.0f", fp->w * 2, floor (fabs (v->f)));

  for (i = 0; i < fp->w; i++)
    ((unsigned char *) s)[i]
      = (xtract_digit (buf[i * 2]) << 4) + xtract_digit (buf[i * 2 + 1]);
}

static inline void
convert_RB (char *s, const fmt_spec *fp, const value *v)
{
  union
    {
      double d;
      unsigned char c[8];
    }
  u;

  u.d = v->f;
  memcpy (s, u.c, fp->w);
}

static inline int
convert_N (char *s, const fmt_spec *fp, const value *v)
{
  double d = floor (v->f);

  if (d < 0 || d == SYSMIS)
    return msg (ME, _("The N output format cannot be used to output a "
		"negative number or the system-missing value."));
  if (d < pow (10.0, fp->w))
    sprintf (s, "%0*.0f", fp->w, v->f);
  else
    memset (s, '*', fp->w);
  return 1;
}

static inline void
convert_A (char *s, const fmt_spec *fp, const value *v)
{
  memcpy (s, v->c, fp->w);
}

static inline void
convert_AHEX (char *s, const fmt_spec *fp, const value *v)
{
  int i;

  for (i = 0; i < fp->w / 2; i++)
    {
      ((unsigned char *) s)[i * 2] = make_hexit ((v->c[i]) >> 4);
      ((unsigned char *) s)[i * 2 + 1] = make_hexit ((v->c[i]) & 0xf);
    }
}

/* Returns 1 if YEAR (as an offset from 1900) can be represented in
   two digits (when the 19- prefix is omitted), 0 otherwise. */
static int
year2 (int year)
{
  if (year >= 0 && year <= 99)
    return 1;
  return msg (ME, _("Year %d cannot be represented in two digits for "
	      "output formatting purposes."), year + 1900);
}

/* Returns 1 if YEAR (as an offset from 1900) can be represented in
   four digits, 0 otherwise. */
static int
year4 (int year)
{
  if (year + 1900 >= 0 && year + 1900 <= 9999)
    return 1;
  return msg (ME, _("Year %d cannot be represented in four digits for "
	      "output formatting purposes."), year + 1900);
}

static inline int
convert_date (char *s, const fmt_spec *fp, const value *v)
{
  static const char *months[12] =
    {
      "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
      "JUL", "AUG", "SEP", "OCT", "NOV", "DEC",
    };

  struct tm *date = unjul (v->f);
  char temp_buf[32] =
  {0};

  switch (fp->type)
    {
    case FMT_DATE:
      if (fp->w >= 11)
	sprintf (temp_buf, "%02d-%s-%04d",
		 date->tm_mday, months[date->tm_mon], date->tm_year + 1900);
      else if (year2 (date->tm_year))
	sprintf (temp_buf, "%02d-%s-%02d",
		 date->tm_mday, months[date->tm_mon], date->tm_year);
      break;
    case FMT_EDATE:
      if (fp->w >= 10)
	sprintf (temp_buf, "%02d.%02d.%04d",
		 date->tm_mday, date->tm_mon + 1, date->tm_year + 1900);
      else
	sprintf (temp_buf, "%02d.%02d.%02d",
		 date->tm_mday, date->tm_mon + 1, date->tm_year);
      break;
    case FMT_SDATE:
      if (fp->w >= 10)
	sprintf (temp_buf, "%04d/%02d/%02d",
		 date->tm_year + 1900, date->tm_mon + 1, date->tm_mday);
      else
	sprintf (temp_buf, "%02d/%02d/%02d",
		 date->tm_year, date->tm_mon + 1, date->tm_mday);
      break;
    case FMT_ADATE:
      if (fp->w >= 10)
	sprintf (temp_buf, "%02d/%02d/%04d",
		 date->tm_mon + 1, date->tm_mday, date->tm_year + 1900);
      else if (year2 (date->tm_year))
	sprintf (temp_buf, "%02d/%02d/%02d",
		 date->tm_mon + 1, date->tm_mday, date->tm_year);
      break;
    case FMT_JDATE:
      if (fp->w >= 7)
	{
	  if (year4 (date->tm_year))
	    sprintf (temp_buf, "%04d%03d", date->tm_year + 1900, date->tm_yday + 1);
	}
      else if (year2 (date->tm_year))
	sprintf (temp_buf, "%02d%03d", date->tm_year, date->tm_yday + 1);
      break;
    case FMT_QYR:
      if (fp->w >= 8)
	sprintf (temp_buf, "%d Q %04d", date->tm_mon / 3 + 1, date->tm_year + 1900);
      else if (year2 (date->tm_year))
	sprintf (temp_buf, "%d Q %02d", date->tm_mon / 3 + 1, date->tm_year);
      break;
    case FMT_MOYR:
      if (fp->w >= 8)
	sprintf (temp_buf, "%s %04d", months[date->tm_mon], date->tm_year + 1900);
      else if (year2 (date->tm_year))
	sprintf (temp_buf, "%s %02d", months[date->tm_mon], date->tm_year);
      break;
    case FMT_WKYR:
      if (fp->w >= 10)
	sprintf (temp_buf, "%02d WK %04d",
		 date->tm_yday / 7 + 1, date->tm_year + 1900);
      else if (year2 (date->tm_year))
	sprintf (temp_buf, "%02d WK %02d", date->tm_yday / 7 + 1, date->tm_year);
      break;
    case FMT_DATETIME:
      {
	char *cp;

	cp = spprintf (temp_buf, "%02d-%s-%04d %02d:%02d",
		       date->tm_mday, months[date->tm_mon], date->tm_year + 1900,
		       (int) fmod (floor (v->f / 60. / 60.), 24.),
		       (int) fmod (floor (v->f / 60.), 60.));
	if (fp->w >= 20)
	  {
	    int w, d;

	    if (fp->w >= 22 && fp->d > 0)
	      d = min (fp->d, fp->w - 21), w = 3 + d;
	    else
	      w = 2, d = 0;

	    cp = spprintf (cp, ":%0*.*f", w, d, fmod (v->f, 60.));
	  }
      }
      break;
#if __CHECKER__
    case 42000:
      assert (0);
#endif
    default:
      assert (0);
    }

  if (temp_buf[0] == 0)
    return 0;
  strbarepadcpy (s, temp_buf, fp->w, ' ');
  return 1;
}

static inline int
convert_time (char *s, const fmt_spec *fp, const value *v)
{
  char temp_buf[40];
  char *cp;

  double time;
  int width;

  if (fabs (v->f) > 1e20)
    {
      msg (ME, _("Time value %g too large in magnitude to convert to "
	   "alphanumeric time."), v->f);
      return 0;
    }

  time = v->f;
  width = fp->w;
  cp = temp_buf;
  if (time < 0)
    *cp++ = '-', time = -time;
  if (fp->type == FMT_DTIME)
    {
      double days = floor (time / 60. / 60. / 24.);
      cp = spprintf (temp_buf, "%02.0f ", days);
      time = time - days * 60. * 60. * 24.;
      width -= 3;
    }
  else
    cp = temp_buf;

  cp = spprintf (cp, "%02.0f:%02.0f",
		 fmod (floor (time / 60. / 60.), 24.),
		 fmod (floor (time / 60.), 60.));

  if (width >= 8)
    {
      int w, d;

      if (width >= 10 && fp->d >= 0 && fp->d != 0)
	d = min (fp->d, width - 9), w = 3 + d;
      else
	w = 2, d = 0;

      cp = spprintf (cp, ":%0*.*f", w, d, fmod (time, 60.));
    }
  strbarepadcpy (s, temp_buf, fp->w, ' ');

  return 1;
}

static inline int
convert_WKDAY (char *s, const fmt_spec *fp, const value *v)
{
  static const char *weekdays[7] =
    {
      "SUNDAY", "MONDAY", "TUESDAY", "WEDNESDAY",
      "THURSDAY", "FRIDAY", "SATURDAY",
    };

  int x = v->f;

  if (x < 1 || x > 7)
    return msg (ME, _("Weekday index %d does not lie between 1 and 7."), x);
  strbarepadcpy (s, weekdays[x - 1], fp->w, ' ');

  return 1;
}

static inline int
convert_MONTH (char *s, const fmt_spec *fp, const value *v)
{
  static const char *months[12] =
    {
      "JANUARY", "FEBRUARY", "MARCH", "APRIL", "MAY", "JUNE",
      "JULY", "AUGUST", "SEPTEMBER", "OCTOBER", "NOVEMBER", "DECEMBER",
    };

  int x = v->f;

  if (x < 1 || x > 12)
    return msg (ME, _("Month index %d does not lie between 1 and 12."), x);
  strbarepadcpy (s, months[x - 1], fp->w, ' ');

  return 1;
}

static inline void
convert_CCx (char *s, const fmt_spec *fp, double v, int x)
{
  set_cust_currency *cc = &set_cc[x];

  fmt_spec f;

  char buf2[128];
  char *cp;

  /* Determine length available, decimal character for number
     proper. */
  f = *fp;
  f.w -= strlen (cc->prefix) + strlen (cc->suffix);
  if (v < 0)
    f.w -= strlen (cc->neg_prefix) + strlen (cc->neg_suffix) - 1;
  f.type = cc->decimal == set_decimal ? FMT_COMMA : FMT_DOT;

  if (f.w <= 0)
    goto fallback;

  /* There's room for all that currency crap.  Let's do the F
     conversion first. */
  convert_F (&f, v);
  if (buf[0] == '*')
    goto fallback;
  insert_commas (buf2, &f);

  /* Postprocess into buf. */
  cp = buf;
  if (v < 0)
    cp = stpcpy (cp, cc->neg_prefix);
  cp = stpcpy (cp, cc->prefix);
  {
    char *bp = buf2;
    while (*bp == ' ')
      bp++;
    assert (v >= 0 || *bp == '-');
    if (v < 0)
      bp++;

    memcpy (cp, bp, f.w - (bp - buf2));
    cp += f.w - (bp - buf2);
  }
  cp = stpcpy (cp, cc->suffix);
  if (v < 0)
    cp = stpcpy (cp, cc->neg_suffix);

  /* Make the final copy. */
  assert (cp - buf <= fp->w);
  if (cp - buf < fp->w)
    {
      memcpy (&s[fp->w - (cp - buf)], buf, cp - buf);
      memset (s, ' ', fp->w - (cp - buf));
    }
  else
    memcpy (s, buf, fp->w);
  return;

  /* If we run out of alternatives, control comes here to try standard
     COMMA format as a fallback. */
fallback:
  f = *fp;
  f.type = FMT_COMMA;
  convert_F (fp, v);
  memcpy (s, buf, fp->w);
}

/* Converts binary value V into printable form in string S according
   to format specification FP.  The string as written has exactly
   FP->W characters.  It is not null-terminated.  Returns 1 on
   success, 0 on failure. */
int
convert_format_to_string (char *s, const fmt_spec *fp, const value *v)
{
  value tmp_val;
  int cat;

  cat = formats[fp->type].cat;
  if (!(cat & FCAT_STRING))
    {
      if ((cat & FCAT_BLANKS_SYSMIS) && v->f == SYSMIS)
	{
	  memset (s, ' ', fp->w);
	  s[fp->w - fp->d - 1] = '.';
	  return 1;
	}
      if ((cat & FCAT_SHIFT_DECIMAL) && v->f != SYSMIS && fp->d)
	{
	  tmp_val.f = v->f * pow (10.0, fp->d);
	  v = &tmp_val;
	}
    }

  switch (fp->type)
    {
      /* Numeric formats. */
    case FMT_F:
    case FMT_COMMA:
    case FMT_DOT:
    case FMT_DOLLAR:
    case FMT_PCT:
      convert_F (fp, v->f);

      if (fp->type == FMT_PCT)
	{
	  if (buf[0] == ' ')
	    {
	      memmove (&buf[0], &buf[1], fp->w);
	      buf[fp->w - 1] = '%';
	    }
	}
      else if (fp->type != FMT_F)
	{
	  insert_commas (s, fp);
	  return 1;
	}
      break;
    case FMT_PIBHEX:
      convert_PIBHEX (s, fp, v);
      return 1;
    case FMT_RBHEX:
      convert_RBHEX (s, fp, v);
      return 1;
    case FMT_Z:
      convert_Z (s, fp, v);
      return 1;
    case FMT_IB:
      convert_IB (s, fp, v);
      return 1;
    case FMT_PIB:
      convert_PIB (s, fp, v);
      return 1;
    case FMT_N:
      return convert_N (s, fp, v);
    case FMT_P:
      convert_P (s, fp, v);
      return 1;
    case FMT_E:
      convert_E (fp, v->f);
      break;
    case FMT_PK:
      convert_PK (s, fp, v);
      return 1;
    case FMT_RB:
      convert_RB (s, fp, v);
      return 1;

      /* Custom currency formats. */
    case FMT_CCA:
      convert_CCx (s, fp, v->f, 0);
      return 1;
    case FMT_CCB:
      convert_CCx (s, fp, v->f, 1);
      return 1;
    case FMT_CCC:
      convert_CCx (s, fp, v->f, 2);
      return 1;
    case FMT_CCD:
      convert_CCx (s, fp, v->f, 3);
      return 1;
    case FMT_CCE:
      convert_CCx (s, fp, v->f, 4);
      return 1;

      /* String formats. */
    case FMT_A:
      convert_A (s, fp, v);
      return 1;
    case FMT_AHEX:
      convert_AHEX (s, fp, v);
      return 1;

      /* Time formats. */
    case FMT_TIME:
    case FMT_DTIME:
      convert_time (s, fp, v);
      return 1;

      /* Lookup-table formats. */
    case FMT_WKDAY:
      return convert_WKDAY (s, fp, v);
    case FMT_MONTH:
      return convert_MONTH (s, fp, v);

      /* Date formats. */
    case FMT_DATE:
    case FMT_EDATE:
    case FMT_SDATE:
    case FMT_ADATE:
    case FMT_JDATE:
    case FMT_QYR:
    case FMT_MOYR:
    case FMT_WKYR:
    case FMT_DATETIME:
      return convert_date (s, fp, v);

    default:
      assert (0);
    }
  memcpy (s, buf, fp->w);
  return 1;
}

/* Converts V into S in F format with width W and D decimal places,
   then deletes trailing zeros.  S is not null-terminated. */
void
num_to_string (double v, char *s, int w, int d)
{
  /* Dummies to pass to convert_F. */
  value val;
  fmt_spec f;

#if !NEW_STYLE
  /* Pointer to `.' in S. */
  char *decp;

  /* Pointer to `E' in S. */
  char *expp;

  /* Number of characters to delete. */
  int n = 0;
#endif

  f.w = w;
  f.d = d;
  val.f = v;

  /* Cut out the jokers. */
  if (!finite (v))
    {
      char temp[9];
      int len;

      if (isnan (v))
	{
	  memcpy (temp, "NaN", 3);
	  len = 3;
	}
      else if (isinf (v))
	{
	  memcpy (temp, "+Infinity", 9);
	  if (v < 0)
	    temp[0] = '-';
	  len = 9;
	}
      else
	{
	  memcpy (temp, _("Unknown"), 7);
	  len = 7;
	}
      if (w > len)
	{
	  int pad = w - len;
	  memset (s, ' ', pad);
	  s += pad;
	  w -= pad;
	}
      memcpy (s, temp, w);
      return;
    }

  convert_F (&f, val.f);
  memcpy (s, buf, w);

#if !NEW_STYLE
  decp = memchr (s, set_decimal, w);
  if (!decp)
    return;

  /* If there's an `E' we can only delete 0s before the E. */
  expp = memchr (s, 'E', w);
  if (expp)
    {
      while (expp[-n - 1] == '0')
	n++;
      if (expp[-n - 1] == set_decimal)
	n++;
      memmove (&s[n], s, expp - s - n);
      memset (s, ' ', n);
      return;
    }

  /* Otherwise delete all trailing 0s. */
  n++;
  while (s[w - n] == '0')
    n++;
  if (s[w - n] != set_decimal)
    {
      /* Avoid stripping `.0' to `'. */
      if (w == n || !isdigit ((unsigned char) s[w - n - 1]))
	n -= 2;
    }
  else
    n--;
  memmove (&s[n], s, w - n);
  memset (s, ' ', n);
#endif
}
