/*
                         Configuration File IO

	Functions:

	int RCLoadFromFile(char *filename)

	void RCMixerParseMixerValueString(   
	        char *s,
	        Coefficient *value1,
	        Coefficient *value2
	)
	void MixerFreeFData(
	        mixer_device_fdata_struct **ptr,
	        int total
	)
	mixer_device_fdata_struct **MixerRCLoadAllFromFile(
	        char *filename,
	        int *total
	)
	int MixerRCLoadFromFile(char *filename, Recorder *recorder)
        int MixerRCSaveToFile(char *filename, Recorder *recorder)

	---

 */

#include <stdio.h>
#include <db.h>
#include <sys/types.h>
#include <malloc.h>
#include <string.h>
#include <sys/stat.h>

#include "../include/cfgfmt.h"
#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/fio.h"

#include "ymode.h"
#include "soundpaths.h"
#include "ysound.h"
#include "ymixer.h"
#include "rcfile.h"
#include "options.h"



#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define MAX(a,b)	(((a) > (b)) ? (a) : (b))



extern int antishift(int in);	/* In main.c */


/*
 *	Loads configuration.
 *
 *	Any current Audio modes and sound paths will be deleted
 *	and then loaded with the new ones from the file.
 */
int RCLoadFromFile(char *filename)
{
	int status;
        char *strptr, *strptr2;

        FILE *fp;
        struct stat stat_buf;

        char parm[CFG_PARAMETER_MAX];
        char val[CFG_VALUE_MAX];
        int lines_read = 0;

	YMode *ymode_ptr;
	SoundPath *sp_ptr;


	/* Check if filename exists. */
	if(filename == NULL)
	    return(-1);
	if(stat(filename, &stat_buf))
	{
	    fprintf(stderr, "%s: No such file.\n", filename);
	    return(-1);
	}

        /* Open filename. */
        fp = fopen(filename, "r");
        if(fp == NULL)
        {
            fprintf(stderr, "%s: Cannot open.\n", filename);
            return(-1);
        }


	/* ************************************************************* */
	/* Free any allocated resources as needed. */

	YModeDeleteAll();
	SoundPathDeleteAll();

	/* Do not delete connections and yhosts. */


	/* ******************************************************** */

	strptr = NULL;

        while(1)
        {
	    /* Free previous line and allocate/read next line. */
            free(strptr); strptr = NULL;
            strptr = FReadNextLineAlloc(fp, UNIXCFG_COMMENT_CHAR);
            if(strptr == NULL) break;
            lines_read++;

            /* Fetch parameter. */
            strptr2 = StringCfgParseParm(strptr);
            if(strptr2 == NULL) continue;
            strncpy(parm, strptr2, CFG_PARAMETER_MAX);
            parm[CFG_PARAMETER_MAX - 1] = '\0';

            /* Fetch value. */
            strptr2 = StringCfgParseValue(strptr);
            if(strptr2 == NULL) strptr2 = "0";  /* Set it to "0" if NULL. */
            strncpy(val, strptr2, CFG_VALUE_MAX);
            val[CFG_VALUE_MAX - 1] = '\0';


	    /* VersionMajor */
            if(!strcasecmp(parm, "VersionMajor"))
            {

	    }
            /* VersionMinor */
            else if(!strcasecmp(parm, "VersionMinor"))
            {

            }

	    /* Port */
            else if(!strcasecmp(parm, "Port"))
            {
		option.port = atoi(val);
	    }
	    /* RefreshInterval */
            else if(!strcasecmp(parm, "RefreshInterval"))
            {
		option.refresh_int.ms = atol(val) / 1000;
                option.refresh_int.us = atol(val) % 1000;
            }
	    /* Device */
	    else if(!strcasecmp(parm, "Device"))
	    {
		strncpy(
		    fname.device,
		    val,
		    PATH_MAX + NAME_MAX
		);
                if(stat(fname.device, &stat_buf))
                {
                    fprintf(stderr,
   "%s: Line %i: Warning: %s: No such device.\n",
                            filename,
                            lines_read,
                            fname.device
                    );
                }
	    }
            /* Mixer */
            else if(!strcasecmp(parm, "Mixer"))
            {
                strncpy(
                    fname.mixer,
                    val,
                    PATH_MAX + NAME_MAX
                );
                if(stat(fname.mixer, &stat_buf))
                {
                    fprintf(stderr,
   "%s: Line %i: Warning: %s: No such device.\n",
                            filename,
                            lines_read,
                            fname.mixer
                    );
                }
            }
	    /* MIDIPlayCommand */
            else if(!strcasecmp(parm, "MIDIPlayCommand"))
	    {
		free(option.midi_play_cmd);
		option.midi_play_cmd = StringCopyAlloc(val);
	    }

            /* **************************************************** */
	    /* BeginAudioMode */
            else if(!strcasecmp(parm, "BeginAudioMode") ||
		    !strcasecmp(parm, "BeginYMode") ||
                    !strcasecmp(parm, "BeginMode")
	    )
            {
		status = YModeAllocate(NULL);
		ymode_ptr = YModeGetPtr(status);
		if(ymode_ptr == NULL)
                    continue;

                while(1)
                {
                    /* Free previous line and allocate/read next line. */
                    free(strptr); strptr = NULL;
                    strptr = FReadNextLineAlloc(fp, UNIXCFG_COMMENT_CHAR);
                    if(strptr == NULL) break;
                    lines_read++;
                    
                    /* Fetch parameter. */
                    strptr2 = StringCfgParseParm(strptr);
                    if(strptr2 == NULL) continue;
                    strncpy(parm, strptr2, CFG_PARAMETER_MAX);
                    parm[CFG_PARAMETER_MAX - 1] = '\0';
                        
                    /* Fetch value. */
                    strptr2 = StringCfgParseValue(strptr);  
                    if(strptr2 == NULL) strptr2 = "0";
                    strncpy(val, strptr2, CFG_VALUE_MAX);
                    val[CFG_VALUE_MAX - 1] = '\0';


                    /* Name */
                    if(!strcasecmp(parm, "Name"))
                    {
			free(ymode_ptr->name);
			ymode_ptr->name = StringCopyAlloc(val);
		    }
                    /* Cycle */
                    else if(!strcasecmp(parm, "Cycle"))
                    {
                        ymode_ptr->cycle.ms = atol(val) / 1000;
                        ymode_ptr->cycle.us = atol(val) % 1000;
                    }
                    /* WriteAhead */
                    else if(!strcasecmp(parm, "WriteAhead"))
                    {
                        ymode_ptr->write_ahead.ms = atol(val) / 1000;
                        ymode_ptr->write_ahead.us = atol(val) % 1000;
                    }
                    /* SampleSize */
                    else if(!strcasecmp(parm, "SampleSize"))
                    {
                        ymode_ptr->sample_size = atoi(val);

			if(ymode_ptr->sample_size == 16)
			    ymode_ptr->sample_size = 16;
			else
			    ymode_ptr->sample_size = 8;
                    }
                    /* Channels */
                    else if(!strcasecmp(parm, "Channels"))
                    {
                        ymode_ptr->channels = atoi(val);

                        if(ymode_ptr->channels == 2)
                            ymode_ptr->channels = 2;
                        else
                            ymode_ptr->channels = 1;
                    }
                    /* SampleRate */
                    else if(!strcasecmp(parm, "SampleRate"))
                    {
                        ymode_ptr->sample_rate = atoi(val);
                    }
#ifdef OSS_BUFFRAG
                    /* AllowFragmenting */
                    else if(!strcasecmp(parm, "AllowFragmenting"))
                    {
                        ymode_ptr->allow_fragments = StringIsYes(val);
                    }
                    /* Fragments */
                    else if(!strcasecmp(parm, "Fragments"))
                    {
                        ymode_ptr->num_fragments = atoi(val);
                    }
                    /* FragmentSize */
                    else if(!strcasecmp(parm, "FragmentSize"))
                    {
			/*   Note: On file, it's bytes, in memory it's
                         *   shifts.
                         */
                        ymode_ptr->fragment_size = antishift(atoi(val) - 1);
                    }
#endif  /* OSS_BUFFRAG */
                    /* FlipStereo */
                    else if(!strcasecmp(parm, "FlipStereo"))
                    {
                        ymode_ptr->flip_stereo = StringIsYes(val);
                    }
                    /* Direction */
                    else if(!strcasecmp(parm, "Direction"))
                    {
			if(!strcasecmp(val, "Record"))
			    ymode_ptr->direction = AUDIO_DIRECTION_RECORD;
			else
                            ymode_ptr->direction = AUDIO_DIRECTION_PLAY;
                    }

                    /* EndMode */
                    else if(!strcasecmp(parm, "EndAudioMode") ||
			    !strcasecmp(parm, "EndYMode") ||
                            !strcasecmp(parm, "EndMode")
		    )
                    {
                        break;
                    }

                    /* Unknown parameter. */
                    else
                    { 
                        fprintf(stderr,
                            "%s: Line %i: Unknown parameter: %s\n",
                            filename,
                            lines_read,
                            parm
                        );
                        continue;
                    }

                }
            }

            /* **************************************************** */
            /* BeginSoundPath */
            else if(!strcasecmp(parm, "BeginSoundPath"))
            {
                status = SoundPathAllocate();
                sp_ptr = SoundPathGetPtr(status);
                if(sp_ptr == NULL)
                    continue;

                while(1)
                {
                    /* Free previous line and allocate/read next line. */
                    free(strptr); strptr = NULL;
                    strptr = FReadNextLineAlloc(fp, UNIXCFG_COMMENT_CHAR);
                    if(strptr == NULL) break;
                    lines_read++;

                    /* Fetch parameter. */
                    strptr2 = StringCfgParseParm(strptr);
                    if(strptr2 == NULL) continue;
                    strncpy(parm, strptr2, CFG_PARAMETER_MAX);
                    parm[CFG_PARAMETER_MAX - 1] = '\0';

                    /* Fetch value. */
                    strptr2 = StringCfgParseValue(strptr);
                    if(strptr2 == NULL) strptr2 = "0";
                    strncpy(val, strptr2, CFG_VALUE_MAX);
                    val[CFG_VALUE_MAX - 1] = '\0';  


                    /* Path */   
                    if(!strcasecmp(parm, "Path"))
                    {
                        free(sp_ptr->path);
                        sp_ptr->path = StringCopyAlloc(val);
                    }
                    /* EndSoundPath */
                    else if(!strcasecmp(parm, "EndSoundPath"))
                    {
                        break;
                    }
                    /* Unknown parameter. */  
                    else
                    {
                        fprintf(stderr,
                            "%s: Line %i: Unknown parameter: %s\n",
                            filename,
                            lines_read,
                            parm
                        );
                        continue;
                    } 
                }
            }

            /* Unknown parameter. */
            else
            {
                fprintf(stderr, "%s: Line %i: Unknown parameter: %s\n",
                    filename,
                    lines_read,
                    parm
                );
            }
	}


	/* Close the file. */
	fclose(fp);


	return(0);
}



/*
 *      Structure to contain a mixer device settings read from
 *	file.
 */
typedef struct {
            
        char *name;
        Coefficient val[YTotalMixers][2];
                
} mixer_device_fdata_struct;

/*
 *	Parses two values seperated by a white space from string
 *	s and puts them into the Coefficient values.
 */
void RCMixerParseMixerValueString(
	char *s,
	Coefficient *value1,
	Coefficient *value2
)
{
	char *strptr;
	const int len = 256;
	char ls[len];


	/* Get value 1. */
	strptr = s;
	if(strptr != NULL)
        {
            while(*strptr == ' ')
                strptr++;

            strncpy(ls, strptr, len);
	    ls[len - 1] = '\0';

	    strptr = strchr(ls, ' ');
	    if(strptr != NULL)
		*strptr = '\0';
	    *value1 = atof(ls);
	}

	/* Get value 2. */
	strptr = strchr(s, ' ');
	if(strptr != NULL)
	{
            while(*strptr == ' ')
                strptr++;

	    strncpy(ls, strptr, len);
            ls[len - 1] = '\0';

            strptr = strchr(ls, ' ');
            if(strptr != NULL) 
                *strptr = '\0';
            *value2 = atof(ls);
	}


	return;
}

/*
 *	Deallocates mixer device file data list.
 */
void MixerFreeFData(
	mixer_device_fdata_struct **ptr,
	int total
)
{
	int i;

	for(i = 0; i < total; i++)
	{
	    if(ptr[i] == NULL)
		continue;

	    free(ptr[i]->name);
	    free(ptr[i]);
	}
	free(ptr);

	return;
}

/*
 *	Loads mixer devices and their values from the file,
 *	allocates them in the returned pointer array.
 *
 *	This array needs to be free'ed by teh calling function.
 */
mixer_device_fdata_struct **MixerRCLoadAllFromFile(
	char *filename,
	int *total
)
{
        int i, n, status;
        char *strptr, *strptr2;

        FILE *fp;
        int lines_read = 0;
        struct stat stat_buf;
        char parm[CFG_PARAMETER_MAX];
        char val[CFG_VALUE_MAX];
   
        Coefficient value1, value2;
        char *mixer_name[] = YMixerConicalNames;

	mixer_device_fdata_struct **ptr = NULL;
	mixer_device_fdata_struct *mdfd_ptr;


        /* Error checks. */
        if((filename == NULL) ||
           (total == NULL)
        )
            return(NULL);
        if(stat(filename, &stat_buf))
            return(NULL);

	/* Reset total. */
	*total = 0;

        /* Open file for reading. */
        fp = fopen(filename, "r");
        if(fp == NULL)
            return(NULL);

        /* Begin reading file. */
        strptr = NULL;
        while(1)
        {
            /* Free previous line and allocate/read next line. */
            free(strptr); strptr = NULL;
            strptr = FReadNextLineAlloc(fp, UNIXCFG_COMMENT_CHAR);
            if(strptr == NULL) break;
            lines_read++;

            /* Fetch parameter. */
            strptr2 = StringCfgParseParm(strptr);
            if(strptr2 == NULL) continue;
            strncpy(parm, strptr2, CFG_PARAMETER_MAX);
            parm[CFG_PARAMETER_MAX - 1] = '\0';

            /* Fetch value. */  
            strptr2 = StringCfgParseValue(strptr);
            if(strptr2 == NULL) strptr2 = "0";  /* Set it to "0" if NULL. */
            strncpy(val, strptr2, CFG_VALUE_MAX);
            val[CFG_VALUE_MAX - 1] = '\0';

            /* BeginMixer */
            if(!strcasecmp(parm, "BeginMixer"))
            {
		/* Allocate a new mixer device structure. */
		i = *total;
		*total = i + 1;
		ptr = (mixer_device_fdata_struct **)realloc(
		    ptr,
		    *total * sizeof(mixer_device_fdata_struct *)
		);
		if(ptr == NULL)
		{
		    *total = 0;
		    continue;
		}

		ptr[i] = (mixer_device_fdata_struct *)calloc(
		    1,
		    sizeof(mixer_device_fdata_struct)
		);
		if(ptr[i] == NULL)
		{
		    *total = i;
		    continue;
		}
		mdfd_ptr = ptr[i];

		/* Record mixer device's name. */
		mdfd_ptr->name = StringCopyAlloc(val);


                while(1)
		{
                    /* Free previous line and allocate/read next line. */
                    free(strptr); strptr = NULL;
                    strptr = FReadNextLineAlloc(fp, UNIXCFG_COMMENT_CHAR);
                    if(strptr == NULL) break;
                    lines_read++;
        
                    /* Fetch parameter. */
                    strptr2 = StringCfgParseParm(strptr);
                    if(strptr2 == NULL) continue;
                    strncpy(parm, strptr2, CFG_PARAMETER_MAX);
                    parm[CFG_PARAMETER_MAX - 1] = '\0';
            
                    /* Fetch value. */
                    strptr2 = StringCfgParseValue(strptr);
                    if(strptr2 == NULL) strptr2 = "0";
                    strncpy(val, strptr2, CFG_VALUE_MAX);
                    val[CFG_VALUE_MAX - 1] = '\0';


		    /* Go through mixer device channel names. */  
                    for(n = 0; n < YTotalMixers; n++)
                    {
                        if(!strcasecmp(parm, mixer_name[n]))
                        {
                            RCMixerParseMixerValueString(
                                val,
				&mdfd_ptr->val[n][0],
				&mdfd_ptr->val[n][1]
                            );
			}
		    }
                    /* EndMixer */
                    if(!strcasecmp(parm, "EndMixer"))
                    {
                        break;
                    }       
                } 
            }
            /* Unknown parameter. */
            else
            {
                fprintf(stderr, "%s: Line %i: Unknown parameter: %s\n",
                    filename,
                    lines_read,
                    parm 
                );   
            }       
        }           

        /* Close the file. */
        fclose(fp);


        return(ptr);
}


/*
 *	Loads the mixer device channel values from the specified
 *	file and updates the recorder's mixer device channel values.
 *
 *	The recorder's mixer must already be opened for this
 *	function to have any affect.
 */
int MixerRCLoadFromFile(char *filename, Recorder *recorder)
{
	int i, n, total;
	mixer_device_fdata_struct **ptr, *mdfd_ptr;
	char *this_mixer_dev_name;
        char *mixer_name[] = YMixerConicalNames; 


	/* Error checks. */
        if((filename == NULL) ||
           (recorder == NULL)
	)
            return(-1);

        /* Get pointer to our mixer device's name. */
        this_mixer_dev_name = fname.mixer;


	/* Read all mixer device names from file. */
	ptr = MixerRCLoadAllFromFile(
	    filename,
	    &total
	);
	if((ptr == NULL) || (total <= 0))
	    return(-1);


	/* Go through mixer device listing. */
	for(i = 0; i < total; i++)
	{
	    mdfd_ptr = ptr[i];

	    if(mdfd_ptr == NULL)
		continue;
	    if(mdfd_ptr->name == NULL)
		continue;


	    /* Is this device listing the same as the one
	     * we're looking for?
	     */
	    if(strcasecmp(mdfd_ptr->name, this_mixer_dev_name))
		continue;

	    /* Go through mixer device channel settings. */
	    for(n = 0; n < YTotalMixers; n++)
	    {
                /* Set mixer values for our mixer. */
		YMixerSet(
		    recorder,
		    n + YMixerCodeBaseOffset,
		    mdfd_ptr->val[n][0],
		    mdfd_ptr->val[n][1]
		);
	    }
	}

	/* Deallocate mixer data listing. */
	MixerFreeFData(ptr, total);


	return(0);
}

/*
 *	Saves the mixer device channel values specified in the
 *	recorder to file.
 *
 *	Other mixer device channel values in the file will also
 *	be read and rewritten to file (but their values unmodified).
 *
 *	The recorder's mixer must already be opened for this
 *	function to have any affect.
 */
int MixerRCSaveToFile(char *filename, Recorder *recorder)
{
        int i, n, total;
	FILE *fp;
        mixer_device_fdata_struct **ptr, *mdfd_ptr;
        char *this_mixer_dev_name;
        char *mixer_name[] = YMixerConicalNames;


        /* Error checks. */
        if((filename == NULL) ||
           (recorder == NULL)
        )
            return(-1);


        /* Set this mixer name pointer. */
        this_mixer_dev_name = fname.mixer;
 

        /* Read all mixer device names from file. */
        ptr = MixerRCLoadAllFromFile(
            filename,
            &total
        );
        if((ptr == NULL) || (total <= 0))   
            return(-1);


        /* Go through mixer device listing, search for our
	 * mixer device by name. If it is found, then update
	 * its values with the current mixer device channel values.
	 */
        for(i = 0; i < total; i++)
        {
            mdfd_ptr = ptr[i];

            if(mdfd_ptr == NULL)
                continue;
            if(mdfd_ptr->name == NULL)
                continue;


            /* Is this device listing the same as the one
             * we're looking for?  
             */
            if(strcasecmp(mdfd_ptr->name, this_mixer_dev_name))
                continue;

            /* Go through mixer device channel settings. */
            for(n = 0; n < YTotalMixers; n++)
            {
		/* Get current mixer values for our mixer. */
                YMixerGet(
                    recorder,
                    n + YMixerCodeBaseOffset,
                    &mdfd_ptr->val[n][0],
                    &mdfd_ptr->val[n][1]
                );
            }
        }    

	/* Do not deallocate mixer device listing just yet, we need
	 * to write them back to file below.
	 */

	/* ************************************************************ */

        /* Open file for writing */
        fp = fopen(filename, "w");
        if(fp != NULL)
	{
	    /* Write commented heading. */
	    fprintf(fp,
"# Mixer device channel settings.\n\
# This file is automatically generated by the YIFF Sound Server.\n\
# You may manually edit this file as needed.\n\
#\n"
	    );

	    /* Write each mixer device channel values to file. */
	    for(i = 0; i < total; i++)
	    {
                mdfd_ptr = ptr[i];

                if(mdfd_ptr == NULL)
                    continue;
                if(mdfd_ptr->name == NULL)
                    continue;

		fprintf(fp, "BeginMixer = %s\n",
		    mdfd_ptr->name
		);

		/* Write each mixer channel values. */
		for(n = 0; n < YTotalMixers; n++)
		{
		    fprintf(fp, "    %s = %.4lf %.4lf\n",
			mixer_name[n],
			mdfd_ptr->val[n][0],
			mdfd_ptr->val[n][1]
		    );
		}

		fprintf(fp, "EndMixer\n");
	    }

	    /* Close the file. */
	    fclose(fp);
	}

        /* Deallocate mixer data listing. */
        MixerFreeFData(ptr, total);


	return(0);
}

