/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysvideo.cc,v 1.9 1999/07/15 18:42:02 slouken Exp $";
#endif

/* BWindow based framebuffer implementation */

#include <stdlib.h>
#include <string.h>

#include <stdio.h>
#include <unistd.h>

#include "SDL.h"
#include "SDL_BeApp.h"
#include "SDL_BWin.h"
#include "SDL_timer.h"
#include "blank_cursor.h"
#include "SDL_lowvideo.h"

extern "C" {

#include "SDL_sysvideo.h"
#include "SDL_sysmouse_c.h"
#include "SDL_syswm_c.h"

#define BEOS_HIDDEN_SIZE	32	/* starting hidden window size */

/* Local data members -- actually covers the entire window interaction */
SDL_BWin   *SDL_Win = NULL;
static BBitmap *SDL_Bitmap = NULL;
WMcursor *SDL_BlankCursor = NULL;

static void SDL_UpdateVideoInfo(SDL_VideoInfo *info)
{
	return;
}

int SDL_SYS_VideoInit(SDL_PixelFormat *vformat)
{
	BRect bounds;

	/* Initialize the Be Application for appserver interaction */
	if ( SDL_InitBeApp() < 0 ) {
		return(-1);
	}

	/* Determine the screen depth */
	{
	  BScreen bscreen;
	  switch(bscreen.ColorSpace()) {
		case B_CMAP8:
			vformat->BitsPerPixel = 8;
			break;
		case B_RGB15:
		case B_RGBA15:
		case B_RGB15_BIG:
		case B_RGBA15_BIG:
			vformat->BitsPerPixel = 15;
			break;
		case B_RGB16:
		case B_RGB16_BIG:
			vformat->BitsPerPixel = 16;
			break;
		case B_RGB32:
		case B_RGBA32:
		case B_RGB32_BIG:
		case B_RGBA32_BIG:
			vformat->BitsPerPixel = 32;
			break;
		default:
			SDL_SetError("Unknown BScreen colorspace: 0x%x",
							bscreen.ColorSpace());
			return(-1);
	  }
	}

	/* Create the window and view */
	bounds.top = 0; bounds.left = 0;
	bounds.right = BEOS_HIDDEN_SIZE;
	bounds.bottom = BEOS_HIDDEN_SIZE;
	SDL_Win = new SDL_BWin(bounds);

	/* Create the clear cursor */
	SDL_BlankCursor = SDL_CreateWMCursor(NULL, blank_cdata, blank_cmask,
			BLANK_CWIDTH, BLANK_CHEIGHT, BLANK_CHOTX, BLANK_CHOTY);

	/* Fill in some window manager capabilities */
	SDL_HWCaps.info.wm_available = 1;
	SDL_WMCaps.SetCaption = SDL_SYS_SetWMCaption;

	/* Fill in the video hardware capabilities */
	SDL_UpdateVideoInfo(&SDL_HWCaps.info);

	/* We're done! */
	return(0);
}

/* We support any dimension at our bit-depth */
SDL_Rect **SDL_SYS_ListModes(SDL_Surface *screen,
					SDL_PixelFormat *format, Uint32 flags)
{
	if ( format->BitsPerPixel == screen->format->BitsPerPixel ) {
		return((SDL_Rect **)-1);
	} else {
		return((SDL_Rect **)0);
	}
}

/* Various screen update functions available */
static void SDL_NormalUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects);

/* FIXME: check return values and cleanup here */
SDL_Surface *SDL_SYS_SetVideoMode(SDL_Surface *current,
				int width, int height, int bpp, Uint32 flags)
{
	BScreen bscreen;
	BRect bounds;
	int needs_unlock;

	/* Create the BBitmap framebuffer */
	if ( SDL_Bitmap != NULL ) {
		delete SDL_Bitmap;
	}
	bounds.top = 0; bounds.left = 0;
	bounds.right = width-1;
	bounds.bottom = height-1;
	SDL_Bitmap = new BBitmap(bounds, bscreen.ColorSpace());
	if ( ! SDL_Bitmap->IsValid() ) {
		SDL_SetError("Couldn't create screen bitmap");
		delete SDL_Bitmap;
		return(NULL);
	}
	current->flags = 0;		/* Clear flags */
	current->w = width;
	current->h = height;
	current->pitch = SDL_Bitmap->BytesPerRow();
	current->pixels = (void *)SDL_Bitmap->Bits();
	SDL_Win->SetBitmap(SDL_Bitmap);

	/* Determine what blit function to use */
	SDL_SYS_UpdateRects = SDL_NormalUpdate;

	/* Hide the window for resizing */
	if ( SDL_Win->Lock() ) {
		if ( SDL_Win->Shown() ) {
			needs_unlock = 1;
			SDL_Win->Hide();
		} else {
			needs_unlock = 0;
		}

		/* Resize and show! */
		SDL_Win->ResizeTo(width, height);
		bounds = bscreen.Frame();
		SDL_Win->MoveTo((bounds.Width()-width)/2,
				(bounds.Height()-height)/2);
		SDL_Win->Show();

		/* Unlock the window manually after the first Show() */
		if ( needs_unlock ) {
			SDL_Win->Unlock();
		}
	}

	/* We're done */
	return(current);
}

static void SDL_NormalUpdate(SDL_Surface *screen,int numrects,SDL_Rect *rects)
{
	if ( SDL_BWin::Alive() && SDL_Win->BeginDraw() ) {
		int i;

		for ( i=0; i<numrects; ++i ) {
			BRect rect;

			rect.top = rects[i].y;
			rect.left = rects[i].x;
			rect.bottom = rect.top+rects[i].h-1;
			rect.right = rect.left+rects[i].w-1;
			SDL_Win->DrawAsync(rect);
		}
		SDL_Win->EndDraw();
	}
}

/* Is the system palette settable? */
int SDL_SYS_SetColors(SDL_Surface *screen, int firstcolor, int ncolors)
{
	int i;
	SDL_Palette *palette;
	const color_map *cmap = BScreen().ColorMap();

	/* Get the screen colormap */
	palette = screen->format->palette;
	for ( i=0; i<256; ++i ) {
		palette->colors[i].r = cmap->color_list[i].red;
		palette->colors[i].g = cmap->color_list[i].green;
		palette->colors[i].b = cmap->color_list[i].blue;
	}
	return(0);
}

void SDL_SYS_VideoQuit(SDL_Surface *screen)
{
	if ( SDL_BlankCursor != NULL ) {
		SDL_FreeWMCursor(screen, SDL_BlankCursor);
		SDL_BlankCursor = NULL;
	}
	if ( screen ) {
		screen->pixels = NULL;
	}
	SDL_QuitBeApp();
}
void SDL_SYS_FinalQuit(void) { }


}; /* Extern C */
