/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_blit_1.c,v 1.7 1999/05/13 19:13:07 slouken Exp $";
#endif

#include <stdio.h>

#include "SDL_types.h"
#include "SDL_video.h"
#include "SDL_blit.h"
#include "SDL_sysvideo.h"
#include "SDL_endian.h"

/* Functions to blit from 8-bit surfaces to other surfaces */

static void SDL_Blit1to1(SDL_BlitInfo *info)
{
	int c;
	int width, height;
	Uint8 *src, *map, *dst;
	int srcskip, dstskip;

	/* Set up some basic variables */
	width = info->d_width;
	height = info->d_height;
	src = info->s_pixels;
	srcskip = info->s_skip;
	dst = info->d_pixels;
	dstskip = info->d_skip;
	map = info->table;

	while ( height-- ) {
		for ( c=width; c; --c ) {
			if ( 1 ) {
			  *dst = map[*src];
			}
			dst++;
			src++;
		}
		src += srcskip;
		dst += dstskip;
	}
}
/* This is now endian dependent */
#if ( SDL_BYTEORDER == SDL_LIL_ENDIAN )
#define HI	1
#define LO	0
#else /* ( SDL_BYTEORDER == SDL_BIG_ENDIAN ) */
#define HI	0
#define LO	1
#endif
static void SDL_Blit1to2(SDL_BlitInfo *info)
{
	int c;
	int width, height;
	Uint8 *src, *dst;
	Uint16 *map;
	int srcskip, dstskip;

	/* Set up some basic variables */
	width = info->d_width;
	height = info->d_height;
	src = info->s_pixels;
	srcskip = info->s_skip;
	dst = info->d_pixels;
	dstskip = info->d_skip;
	map = (Uint16 *)info->table;

	/* Memory align at 4-byte boundary, if necessary */
	if ( (long)dst & 0x03 ) {
		/* Don't do anything if width is 0 */
		if ( width == 0 ) {
			return;
		}
		--width;

		while ( height-- ) {
			/* Perform copy alignment */
			*(Uint16 *)dst = map[*src++];
			dst += 2;

			/* Copy in 4 pixel chunks */
			for ( c=width/4; c; --c ) {
				*(Uint32 *)dst =
					(map[src[HI]]<<16)|(map[src[LO]]);
				src += 2;
				dst += 4;
				*(Uint32 *)dst =
					(map[src[HI]]<<16)|(map[src[LO]]);
				src += 2;
				dst += 4;
			}
			/* Get any leftovers */
			switch (width % 4) {
				case 3:
					*(Uint16 *)dst = map[*src++];
					dst += 2;
				case 2:
					*(Uint32 *)dst =
					  (map[src[HI]]<<16)|(map[src[LO]]);
					src += 2;
					dst += 4;
					break;
				case 1:
					*(Uint16 *)dst = map[*src++];
					dst += 2;
					break;
			}
			src += srcskip;
			dst += dstskip;
		}
	} else { 
		while ( height-- ) {
			/* Copy in 4 pixel chunks */
			for ( c=width/4; c; --c ) {
				*(Uint32 *)dst =
					(map[src[HI]]<<16)|(map[src[LO]]);
				src += 2;
				dst += 4;
				*(Uint32 *)dst =
					(map[src[HI]]<<16)|(map[src[LO]]);
				src += 2;
				dst += 4;
			}
			/* Get any leftovers */
			switch (width % 4) {
				case 3:
					*(Uint16 *)dst = map[*src++];
					dst += 2;
				case 2:
					*(Uint32 *)dst =
					  (map[src[HI]]<<16)|(map[src[LO]]);
					src += 2;
					dst += 4;
					break;
				case 1:
					*(Uint16 *)dst = map[*src++];
					dst += 2;
					break;
			}
			src += srcskip;
			dst += dstskip;
		}
	}
}
static void SDL_Blit1to3(SDL_BlitInfo *info)
{
	int c, o;
	int width, height;
	Uint8 *src, *map, *dst;
	int srcskip, dstskip;

	/* Set up some basic variables */
	width = info->d_width;
	height = info->d_height;
	src = info->s_pixels;
	srcskip = info->s_skip;
	dst = info->d_pixels;
	dstskip = info->d_skip;
	map = info->table;

	while ( height-- ) {
		for ( c=width; c; --c ) {
			if ( 1 ) {
				o = *src * 4;
				dst[0] = map[o++];
				dst[1] = map[o++];
				dst[2] = map[o++];
			}
			src++;
			dst += 3;
		}
		src += srcskip;
		dst += dstskip;
	}
}
static void SDL_Blit1to4(SDL_BlitInfo *info)
{
	int c;
	int width, height;
	Uint8 *src;
	Uint32 *map, *dst;
	int srcskip, dstskip;

	/* Set up some basic variables */
	width = info->d_width;
	height = info->d_height;
	src = info->s_pixels;
	srcskip = info->s_skip;
	dst = (Uint32 *)info->d_pixels;
	dstskip = info->d_skip/4;
	map = (Uint32 *)info->table;

	while ( height-- ) {
		for ( c=width/4; c; --c ) {
			*dst++ = map[*src++];
			*dst++ = map[*src++];
			*dst++ = map[*src++];
			*dst++ = map[*src++];
		}
		switch ( width % 4 ) {
			case 3:
				*dst++ = map[*src++];
			case 2:
				*dst++ = map[*src++];
			case 1:
				*dst++ = map[*src++];
		}
		src += srcskip;
		dst += dstskip;
	}
}

static SDL_loblit one_blit[] = {
	NULL, SDL_Blit1to1, SDL_Blit1to2, SDL_Blit1to3, SDL_Blit1to4
};
static SDL_loblit complex_blit[] = {
	NULL, SDL_BlitKey, SDL_BlitAlpha, SDL_BlitAlphaKey
};

SDL_loblit SDL_CalculateBlit1(SDL_Surface *surface, int complex)
{
	int which;

	if ( surface->map->dst->format->BitsPerPixel < 8 ) {
		which = 0;
	} else {
		which = surface->map->dst->format->BytesPerPixel;
	}
	if ( complex ) {
		return(complex_blit[complex]);
	} else {
		return(one_blit[which]);
	}
}
