# File "selector.tcl":
# A file selector box.

# This file is part of Malaga, a system for Left Associative Grammars.
# Copyright (C) 1995-1998 Bjoern Beutel
#
# Bjoern Beutel
# Universitaet Erlangen-Nuernberg
# Abteilung fuer Computerlinguistik
# Bismarckstrasse 12
# D-91054 Erlangen
# e-mail: malaga@linguistik.uni-erlangen.de
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Global variables:
# $selector(name) -- name of file that has been selected
# $selector(dir) -- path of directory that has been selected
# $selector(finished) -- 1 if selector can finish now, else 0

#------------------------------------------------------------------------------

# Select a path name. The path is returned.

proc select_file {{title "Select file:"} {default_name ""} {default_dir ""}} {

  global selector

  set font "-*-helvetica-medium-r-normal-*-17-*-*-*-*-*-iso8859-1"

  if {$default_dir == ""} {set default_dir [pwd]}

  # Build file selector window.
  toplevel .selector
  wm geometry .selector 350x300
  wm title .selector $title
  wm maxsize .selector 1000 1000
  wm minsize .selector 100 100

  # Create "ok" and "cancel" widget
  frame .selector.ok_cancel
  button .selector.ok_cancel.ok \
  -text "OK" \
  -font $font \
  -command {
    global selector
    set selector(name) [.selector.file.file get]
    set selector(dir) [.selector.dir.dir get]
    set selector(finished) 1
  }
  button .selector.ok_cancel.cancel \
  -text "Cancel" \
  -font $font \
  -command {
    global selector
    set selector(name) ""
    set selector(dir) ""
    set selector(finished) 1
   }
  pack append .selector.ok_cancel \
  .selector.ok_cancel.ok {left fill expand} \
  .selector.ok_cancel.cancel {left fill expand}
  
  # Create the dir string widget
  frame .selector.dir
  label .selector.dir.dir_label -font $font -text "Dir:"
  entry .selector.dir.dir -relief raised -font $font
  bind .selector.dir.dir <Return> {show_files}
  .selector.dir.dir insert 0 $default_dir
  pack append .selector.dir \
  .selector.dir.dir_label {left} \
  .selector.dir.dir {left fill expand}

  # Create the file listbox.
  frame .selector.files -relief raised
  scrollbar .selector.files.hscroll \
  -orient horiz \
  -command ".selector.files.listbox xview"
  scrollbar .selector.files.vscroll \
  -orient vert \
  -command ".selector.files.listbox yview"
  listbox .selector.files.listbox \
  -font $font \
  -exportselection false \
  -relief raised \
  -xscrollcommand {.selector.files.hscroll set} \
  -yscrollcommand {.selector.files.vscroll set}
  pack append .selector.files \
  .selector.files.vscroll {right filly} \
  .selector.files.hscroll {bottom fillx} \
  .selector.files.listbox {left fill expand}
  bind .selector.files.listbox <ButtonPress-1> "file_select_click %y"
  bind .selector.files.listbox <Button1-Motion> "file_select_click %y"
  bind .selector.files.listbox <Double-Button-1> "file_select_double_click %y"

  frame .selector.file
  label .selector.file.file_label -font $font -text "File:"
  entry .selector.file.file -font $font -relief raised
  .selector.file.file delete 0 end
  .selector.file.file insert 0 $default_name
  bind .selector.file.file <Return> {
    global selector
    set selector(name) [.selector.file.file get]
    set selector(dir) [.selector.dir.dir get]
    set selector(finished) 1
  }
  pack append .selector.file \
  .selector.file.file_label {left} \
  .selector.file.file {left fill expand}

  # packing
  pack append .selector \
  .selector.ok_cancel {bottom fill} \
  .selector.file {bottom fill} \
  .selector.dir {top fill} \
  .selector.files {top fill expand}

  show_files
  focus .selector.file.file

  # Let user interact.
  update idletask
  grab .selector
  set selector(finished) 0
  tkwait variable selector(finished)
  
  # Return result.
  destroy .selector
  set selector(name) [string trim $selector(name)]
  set selector(dir) [string trimright [string trim $selector(dir)] "/"]
  if {$selector(name) == ""} {
    return $selector(dir)
  } else {return "$selector(dir)/$selector(name)"}
}

#------------------------------------------------------------------------------

# Select a file name by mouse click.

proc file_select_click {position} {
  set entry_number [.selector.files.listbox nearest $position]
  if {$entry_number >= 0} {
    set entry [string trimright [string trim [.selector.files.listbox get \
					      $entry_number]] "/@*"]
    set dir [string trimright [string trim [.selector.dir.dir get]] "/"]
    
    if [is_file "$dir/$entry"] {
      .selector.file.file delete 0 end
      .selector.file.file insert 0 $entry
    }
  }
}

#------------------------------------------------------------------------------

# Select a file name by mouse double click.

proc file_select_double_click {position} {
  global selector

  set entry_number [.selector.files.listbox nearest $position]
  if {$entry_number >= 0} {
    set entry [string trimright [string trim [.selector.files.listbox get \
					      $entry_number]] "/@*"]
    set dir [string trimright [string trim [.selector.dir.dir get]] "/"]
    
    if [is_file "$dir/$entry"] {
      set selector(name) $entry
      set selector(dir) $dir
      set selector(finished) 1
    } elseif {[is_directory "$dir/$entry"] \
	      && [file executable "$dir/$entry"]} {
      .selector.dir.dir delete 0 end
      if {$entry == ".."} {
	.selector.dir.dir insert 0 [file dirname $dir]
      } else {.selector.dir.dir insert 0 "$dir/$entry"}
      show_files
    }
  }
}

#------------------------------------------------------------------------------

# Show the file list.

proc show_files {} {

  set dir "[string trimright [string trim [.selector.dir.dir get]] "/"]/"
  .selector.dir.dir delete 0 end
  .selector.dir.dir insert 0 $dir
  
  .selector.files.listbox delete 0 end
  
  if {! [catch {exec ls -F $dir} directory]} {
    if {"$dir" != "/"} {.selector.files.listbox insert end "../"}
    foreach dir_entry [lsort $directory] {
      if {"$dir_entry" != "../" && "$dir_entry" != "./"} {
	.selector.files.listbox insert end $dir_entry
      }
    }
  }
}

#------------------------------------------------------------------------------

# Return 1 if "path_name" is a directory or a symbolic link to a directory.

proc is_directory {path_name} {

  # Go through link list...
  if [catch "file type $path_name" file_type] {return 0}
  while {"$file_type" == "link"} {
    set path_name [file readlink $path_name]
    if [catch "file type $path_name" file_type] {return 0}
  }

  return [file isdirectory $path_name]
}

#------------------------------------------------------------------------------

# Return 1 if "path_name" is a file or a symbolic link to a file.

proc is_file {path_name} {

  # Go through link list.
  if [catch "file type $path_name" file_type] {return 0}
  while {"$file_type" == "link"} {
    set path_name [file readlink $path_name]
    if [catch "file type $path_name" file_type] {return 0}
  }

  return [file isfile $path_name]
}

#------------------------------------------------------------------------------
