/* File "malaga_files.c":
 * Defines the structure of compiled Malaga files. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include <stdio.h>
#include <time.h>
#include "basic.h"

#undef GLOBAL
#define GLOBAL
#include "malaga_files.h"

/* constants ================================================================*/

LOCAL char malaga[] = "MALAGA"; /* magic key */

/* variables ================================================================*/

LOCAL time_t symbol_compile_time = 0; /* time the symbol file was compiled */

/* functions ================================================================*/

GLOBAL void check_header (common_header_t *header, 
			  string_t file_name, 
			  short_t file_type, 
			  long_t code_version)
/* Check if <header> is of <file_type> and <code_version>.
 * <file_name> is needed for error messages. */
{
  if (COMP_MEM (header->malaga, malaga, sizeof (char), MALAGA_LEN) != NULL)
    error ("\"%s\" is not a Malaga file", file_name);

  if (header->file_type != file_type)
    error ("\"%s\" is of wrong file type", file_name);

  if (header->code_version != code_version)
    error ("\"%s\" is of wrong code version", file_name);

  if (file_type == SYMBOL_FILE)
    symbol_compile_time = header->symbol_compile_time;
  else if (header->symbol_compile_time < symbol_compile_time)
    error ("\"%s\" uses older symbol file", file_name);
  else if (header->symbol_compile_time > symbol_compile_time)
    error ("\"%s\" uses newer symbol file", file_name);
}

/*---------------------------------------------------------------------------*/

GLOBAL void set_header (common_header_t *header, 
			short_t file_type,
  			long_t code_version)
/* Set <header> to be of <file_type> and <code_version>. */
{
  if (symbol_compile_time == 0)
    time (&symbol_compile_time);

  COPY_MEM (header->malaga, malaga, sizeof (char), MALAGA_LEN);
  header->file_type = file_type;
  header->code_version = code_version;
  header->symbol_compile_time = symbol_compile_time;
}
