/* File "instr_type.h":
 * The rule instructions. */

/* This file is part of Malaga, a system for Left Associative Grammars.
 * Copyright (C) 1995-1998 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

typedef u_long_t instr_t;
/* An instruction is an u_long_t whose lower 8 bits are used
 * to store the opcode. The upper 24 bits are used to store a signed value. */

#define INSTR_INFO_MIN (-1L << 23)
#define INSTR_INFO_MAX ((1L << 23) - 1)
/* Never use an instruction info smaller than INSTR_INFO_MIN and
 * greater than INSTR_INFO_MAX. */

#define OPCODE_MAX 255 /* the largest opcode possible */

#define INSTR(opcode, instr_info) \
  ((opcode) | (u_long_t) (instr_info) << 8)
/* Use this macro to create an instruction. */ 

#define OPCODE(instr) \
  ((instr) & OPCODE_MAX)
/* Use this macro to get the opcode of an instruction. */
/* Use this macro to get the opcode of an instruction. */

#define INSTR_INFO(instr) \
  ((long_t) (instr) >> 8)
/* Use this macro to get the info of an instruction. */


/* The rule-internal state is described by the following variables:
 * <S> is the rule stack, a stack of values.
 * <SP> (stack pointer) is the index of the first free element in <S>.
 * <BP> (base pointer) is the index of the first element in <S> local to the
 *      current rule.
 * <I> is information that is stored in the instruction itself.
 * <PC> is the program counter, i.e. index of the next node to be executed.
 * <B>, the backtrace stack, is a stack of triples: (PC, BP, S).
 * <BTP> is the top index of <B>. */
enum
{ /* These are the opcodes of the rule instructions. */

  /* control instructions */
  INS_ERROR,             /* terminate all paths and report Error(I). */
  INS_TERMINATE,         /* terminate this path. */
  INS_NOP,               /* do nothing. */
  INS_TERMINATE_IF_NULL, /* SP--; if (S[SP] == NULL) then terminate; */

  /* result instructions */
  INS_ADD_END_STATE,     /* result = S[SP-1]; rule_successful = TRUE; SP-- */
  INS_ADD_STATE,         /* result = S[SP]; rules = rules[I];
			  * rule_successful = TRUE; SP-- */
  INS_ADD_ALLO,          /* surf = S[SP-2]; allo = S[SP-1];
			  * rule_successful = TRUE; SP-= 2 */
  INS_SUCCEED,           /* rule_successful = TRUE; */

  /* stack management instructions */
  INS_PUSH_NULL,         /* for (i = 0; i < I; i++) { S[SP+i] = NULL; }
			  * SP += I; */
  INS_PUSH_VAR,          /* S[SP] = S[I]; SP++; */
  INS_PUSH_CONST,        /* S[SP] = Const(I); SP++; */
  INS_PUSH_SYMBOL,       /* S[SP] = Symbol(I); SP++; */
  INS_PUSH_PATTERN_VAR,  /* S[SP] = Pattern_Var(I); SP++; */
  INS_POP,               /* SP -= I; */

  /* value instructions */
  INS_BUILD_LIST,        /* S[SP-I] = <S[SP-I],..,S[SP-1]>; SP -= (I-1); */
  INS_BUILD_RECORD,      /* S[SP-2I] = "[" S[SP-2I] : S[SP-2I+1] ,..,
			  *                S[SP-2 ] : S[SP-1] "]"; */
       	                 /* SP -= 2I-1; */
  INS_BUILD_PATH,        /* SP[SP-I]= "." S[SP-I] "." .. "." S[SP-1]; 
			  * SP -= (I-1); */
  INS_DOT_OPERATION,     /* S[SP-2] = S[SP-2] "." S[SP-1]; SP--; */
  INS_PLUS_OPERATION,    /* S[SP-2] := S[SP-2] "+" S[SP-1]; SP--; */
  INS_MINUS_OPERATION,   /* S[SP-2] = S[SP-2] "-" S[SP]; SP--; */
  INS_ASTERISK_OPERATION,/* S[SP-2] = S[SP-2] "*" S[SP]; SP--; */
  INS_SLASH_OPERATION,   /* S[SP-2] = S[SP-2] "/" S[SP]; SP--; */
  INS_UNARY_MINUS_OP,    /* S[SP-1] = "-" S[SP-1]; */
  INS_GET_ATTRIBUTE,     /* S[SP-1] = S[SP-1] "." Symbol(I); */
  INS_REMOVE_ATTRIBUTE,  /* S[SP-1] = S[SP-1] "-" Symbol(I); */
  INS_STD_FUNCTION,      /* S[SP-1] = function_I (S[SP-1]); */
  INS_MATCH,             /* S[SP-1] = S[SP-1] "match" String(I); (yes or no) */

  /* instructions to modify variables */
  INS_SET_VAR,           /* S[I] = S[SP-1]; SP--; */
  INS_PLUS_VAR,          /* S[I] = S[I] "+" S[SP-1]; SP--; */
  INS_MINUS_VAR,         /* S[I] = S[I] "-" S[SP-1]; SP--; */
  INS_ASTERISK_VAR,      /* S[I] = S[I] "*" S[SP-1]; SP--; */
  INS_SLASH_VAR,         /* S[I] = S[I] "/" S[SP-1]; SP--; */
  INS_SET_VAR_PATH,      /* (S[I] "." S[SP]) = S[SP+1]; SP -= 2; */
  INS_PLUS_VAR_PATH,     /* (S[I] "." S[SP]) = (S[I] "." S[SP]) "+" S[SP+1]; 
			  * SP -= 2; */
  INS_MINUS_VAR_PATH,    /* (S[I] "." S[SP]) = (S[I] "." S[SP]) "-" S[SP+1]; 
			  * SP -= 2; */
  INS_ASTERISK_VAR_PATH, /* (S[I] "." S[SP]) = (S[I] "." S[SP]) "*" S[SP+1]; 
			  * SP -= 2; */
  INS_SLASH_VAR_PATH,    /* (S[I] "." S[SP]) = (S[I] "." S[SP]) "/" S[SP+1]; 
			  * SP -= 2; */

  /* instructions to support loops */
  INS_PUSH_1ST_ELEMENT,  /* S[SP] = first element/attribute/ordinal of S[I]; 
			  * SP++; */
  INS_ITERATE,           /* S[I-1] must be a list, a record, or a number
                          * and S[I] the n-th element/attrib/ordinal of S[I-1].
                          * S[I] = (n+1)-th element/attr/ordinal of S[I-1]; */

  /* jump instructions */
  INS_JUMP,                /* PC = I. */
  INS_JUMP_IF_EQUAL,       /* if (  S[SP-2] "="  S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_NOT_EQUAL,   /* if (! S[SP-2] "="  S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_CONGR,       /* if (  S[SP-2] "~"  S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_NOT_CONGR,   /* if (! S[SP-2] "~"  S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_IN,          /* if (  S[SP-2] "in" S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_NOT_IN,      /* if (! S[SP-2] "in" S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_LESS,        /* if (  S[SP-2] <    S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_NOT_LESS,    /* if (! S[SP-2] <    S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_GREATER,     /* if (  S[SP-2] >    S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_NOT_GREATER, /* if (! S[SP-2] >    S[SP-1]) {PC = I;} SP -= 2; */
  INS_JUMP_IF_NULL,        /* if (  S[SP-1] == NULL)  {PC = I;} SP--; */
  INS_JUMP_IF_NOT_NULL,    /* if (! S[SP-1] == NULL)  {PC = I;} SP--; */
  INS_JUMP_IF_YES,         /* if (  S[SP-1] == yes)   {PC = I;} SP--; */
  INS_JUMP_IF_NO,          /* if (  S[SP-1] == no)    {PC = I;}  SP--; */
  INS_JUMP_NOW,            /* B[BTP] = {PC, BP, S}; PC = I; BTP++; */
  INS_JUMP_LATER,          /* B[BTP] = {I, BP, S}; BTP++; */
  INS_JUMP_SUBRULE,        /* Push BP; Push PC+1; BP = TOP; 
			    * PC = first_instr (I); */
  INS_RETURN               /* SP = BP; Pop PC; Pop BP; Pop (I-1); */
};

/* The possible error values for INS_ERROR */
enum {ASSERTION_ERROR, NO_RETURN_ERROR};

/* The possible standard functions for <standard_function>. */
enum {FUNC_TO_ATOMS, FUNC_IS_CAPITAL, FUNC_GET_LENGTH, FUNC_TO_MULTI, 
      FUNC_TO_SET, FUNC_GET_SWITCH, FUNC_GET_VALUE_TYPE, FUNC_GET_SYMBOL_NAME,
      FUNC_TRANSMIT};

/*---------------------------------------------------------------------------*/
