#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include "dialog.h"
#include <diajava.h>
#include "internal.h"
#include "../diajava/proto.h"
#include "../diajava/protorev.h"
#include "../paths.h"
#include "dialog.m"

static DIAJAVA *java = NULL;

static void diagui_end()
{
	delete java;
	java = NULL;
}

static DIAGUI_MODE diagui_mode=DIAGUI_AUTO;
static int diagui_handlein=-1;
static int diagui_handleout=-1;
static const char *diagui_lines;

/*
	Disable the GUI mode for the current session
*/
void diagui_setmode(DIAGUI_MODE mode)
{
	diagui_mode = mode;
}
/*
	Set the socket handle used to talk with the GUI front end
*/
void diagui_sethandle(int handlein, int handleout, const char *lines)
{
	diagui_handlein = handlein;
	diagui_handleout = handleout;
	diagui_lines = lines;
}

/*
	Initialise communication with the user interface server
*/
int diagui_init ()
{
	int ret = -1;
	atexit (diagui_end);
	bool guiok = true;
	if (diagui_handlein == -1){
		if (diagui_mode == DIAGUI_AUTO){
			guiok = linuxconf_getguimode();
		}else if (diagui_mode == DIAGUI_NOGUI){
			guiok = false;
		}
	}
	uithread_init (20);
	if (diagui_handlein != -1){
		java = new DIAJAVA(diagui_handleout,diagui_handlein,diagui_lines);
	}else{
		java = new DIAJAVA(guiok);
	}
	if (java->is_ok()){
		char tmp[1000];
		diagui_send ("Version %d %s\n",PROTOGUI_REV
			,diagui_quote(MSG_U(E_GUIVER,"Invalid GUI protocol version")
				,tmp));
		char dianame[200],actionid[100];
		int menu,but;
		POPENWAITS tbo;
		java->wait (tbo,dianame,actionid,menu,but);
		if (strcmp(actionid,"ncurses")!=0){
			ret = 0;
		}else{
			delete java;
			java = NULL;
		}
	}else{
		delete java;
		java = NULL;
	}
	return ret;
}

static int diagui_lastbut,diagui_lastmenu;
static char actionid[100];
static char diapath[200];
static SSTRINGS valids;
static SSTRING_KEYS messages;
static SSTRING lastmsg;

class POPENWAITID: public POPENWAIT{
public:
	int threadid;
	bool mustreturn;
	POPENWAITID (POPENFD &_po, int _timeout, int _threadid)
		: POPENWAIT (_po,_timeout){
		threadid = _threadid;
		mustreturn = false;
	}
};

static POPENWAITS tbpopen;

static int diagui_wait()
{
	int ret = -1;	// Can't return -1
	lastmsg.setfrom ("");
	while (messages.getnb() > 0){
		SSTRING_KEY *msg = messages.getitem(0);
		const char *dianame = msg->getobjval();
		if (valids.lookup(dianame)!=-1){
			ret = atoi (dianame+5);
			lastmsg.setfrom (msg->get());
			messages.remove_del (0);
			break;
		}
		messages.remove_del (0);
	}
	for (int i=0; i<tbpopen.getnb(); i++){
		POPENWAITID *p = (POPENWAITID*)tbpopen.getitem(i);
		if (p->mustreturn){
			p->mustreturn = false;
			ret = p->threadid;
			break;
		}
	}
	if (ret == -1){
		static bool wakeupfront = false;
		if (wakeupfront){
			wakeupfront = false;
			if (diajava_alive) diagui_sendcmd (P_Alive,"\n");
		}
		while (ret == -1){
			int retsel = java->wait(tbpopen,diapath,actionid
				,diagui_lastmenu,diagui_lastbut);
			if (retsel == 0){
				// fprintf (stderr,"Must return\n");
				for (int i=0; i<tbpopen.getnb(); i++){
					POPENWAITID *p = (POPENWAITID*)tbpopen.getitem(i);
					p->mustreturn = true;
				}
			}
			for (int i=0; i<tbpopen.getnb(); i++){
				POPENWAITID *p = (POPENWAITID*)tbpopen.getitem(i);
				if (p->retcode > 0 || p->mustreturn){
					p->mustreturn = false;
					ret = p->threadid;
					break;
				}
			}
			if (ret == -1){
				//fprintf (stderr,"diagui_wait %d %d %s\n",diagui_lastmenu,diagui_lastbut,diapath);
				// Check if the dialog is listening
				char *pt = strchr (diapath,'.');
				if (pt != NULL) *pt = '\0';
				if (valids.lookup(diapath)!=-1){
					ret = atoi (diapath+5);
					if (pt != NULL) *pt = '.';
					// Something was received from the front-end
					// We will have to tell it when we are back listening
					wakeupfront = true;
				}
			}
		}
	}
	return ret;
}


EXPORT const char *diagui_quote (const char *s, char tmp[1000])
{
	const char *ret = s;
	if (s[0] == '\0' || strchr(s,' ')!=NULL || s[0] == '"'){
		ret = tmp;
		char *pt = tmp;
		*pt++ = '"';
		while (*s != '\0' && (pt-tmp) < 997){
			if (*s == '"' || *s == '\\'){
				*pt++ = '\\';
				*pt++ = *s;
			}else{
				*pt++ = *s;
			}
			s++;
		}
		*pt++ = '"';
		*pt++ = '\0';
	}
	return ret;
}

/*
	Send a command to the user interface server
*/
void diagui_send (const char *ctl, ...)
{
	char buf[1000];
	va_list list;
	va_start (list,ctl);
	vsnprintf (buf,sizeof(buf)-1,ctl,list);
	java->send ("%s",buf);
	//fprintf (stderr,"guisend: %s",buf);
	va_end (list);
}

/*
	Send a command to the user interface server
*/
EXPORT void diagui_sendcmd (int cmd, const char *ctl, ...)
{
	char buf[1000];
	va_list list;
	va_start (list,ctl);
	vsprintf (buf,ctl,list);
	java->sendcmd (cmd,"%s",buf);
	//fprintf (stderr,"guisend: %s",buf);
	va_end (list);
}

/*
	Flush the command sent to the front-end
*/
EXPORT void diagui_flush ()
{
	if (java != NULL) java->flush();
}

void diagui_send_Label(const char *str)
{
	char tmp[1000];
	diagui_sendcmd (P_Label,"%s\n",diagui_quote(str,tmp));
}

/*
	Send an Icon_xpm command to the user interface server
*/
int diagui_sendxpm (
	const char *name,	// Name of the icon to send
	char *name_sent)	// Name selected if this one was missing
{
	int ret  = -1;
	static SSTRINGS sofar;	// List of icons already transmitted
	strcpy (name_sent,name);
	if (sofar.lookup(name)!=-1){
		ret = 0;
	}else{
		char path[PATH_MAX];
		sprintf (path,"%s/images/%s.xpm",USR_LIB_LINUXCONF,name);
		FILE *fin = fopen (path,"r");
		if (fin == NULL){
			strcpy (name_sent,"missing_icon");
			if (sofar.lookup(name_sent)!=-1){
				ret = 0;
			}else{
				sprintf (path,"%s/images/%s.xpm",USR_LIB_LINUXCONF,name_sent);
				fin = fopen (path,"r");
			}
		}
		if (fin != NULL){
			char buf[1000];
			while (fgets(buf,sizeof(buf)-1,fin)!=NULL){
				int last = strlen (buf)-1;
				if (last >=0 && buf[last] == '\n') buf[last] = '\0';
				char tmp[1000];
				diagui_sendcmd (P_Str,"%s\n",diagui_quote(buf,tmp));
			}
			fclose (fin);
			diagui_sendcmd (P_Xfer_xpm,"%s\n",name_sent);
			sofar.add (new SSTRING(name_sent));
			ret = 0;
		}
	}
	return ret;
}

/*
	Get the dialog ID already used to setup the FORM object in the front-end
	Return NULL if the front-end do not support the setval primitive
	or if the FORM has not been created yet.

	Return tmp if ok.
*/
const char *DIALOG::setguiname(char tmp[200])
{
	const char *dianame = NULL;
	if (internal->guidone_once && diajava_setval){
		extern int uithread_id;
		int n = 0;
		if (!internal->context.is_empty()){
			n = snprintf (tmp,200-1,"%s.",internal->context.get());
		}
		snprintf (tmp+n,200-n-1,"main-%d-%d",uithread_id,internal->gui_id);
		dianame = tmp;
	}
	return dianame;
}

static SSTRING default_context;		// Default context for DIALOGs

/*
	Record the default context in which dialog may be inserted.
	A context is a path inside an existing dialog.
*/
EXPORT void diagui_setdefaultctx (const char *s)
{
	default_context.setfrom (s);
}

/*
	Ask the front-end to delete (forget) this DIALOG
*/
PRIVATE void DIALOG::guidelete()
{
	if (internal->guidone_once){
		internal->guidone_once = false;
		extern int uithread_id;
		if (internal->context.is_empty()){
			diagui_sendcmd (P_Delete,"main-%d-%d\n",uithread_id
				,internal->gui_id);
		}else{
			diagui_sendcmd (P_Delete,"%s.main-%d-%d\n"
				,internal->context.get()
				,uithread_id,internal->gui_id);
		}
	}
}

/*
	Hide the DIALOG from the screen. Only useful in graphic mode.
	No effect for other mode.
*/
PUBLIC void DIALOG::hide()
{
	guidelete();
}

PRIVATE void DIALOG::showgui(int &nof, int but_options)
{
	extern int uithread_id;
	if (!internal->guidone){
		internal->guidone = true;
		if (internal->guidone_once && !diajava_reconfdia){
			guidelete();
			// We allocate a new id as this seem to confuse gnome-linuxconf
			internal->gui_id = multi_alloc_gui_id();
		}
		internal->guidone_once = true;
		char tmp[1000];
		DIALOG_TYPE diatype = internal->diatype;
		if (getnb()==1 && (but_options & MENUBUT_ACCEPT)!=0
			&& internal->diatype == DIATYPE_STD){
			diatype = DIATYPE_POPUP;
		}
		static char *tbtype[]={"std","error","notice","popup"};
		if (diatype == DIATYPE_STD && !internal->context_wasset){
			setcontext (default_context.get());
		}
		bool context_doend = false;
		if (!internal->context.is_empty()){
			diagui_sendcmd (P_Setcontext,"%s\n",internal->context.get());
			context_doend = true;
		}
		diagui_sendcmd (P_MainForm,"main-%d-%d %s %s\n",uithread_id
			,internal->gui_id,diagui_quote(internal->title.get(),tmp)
			,tbtype[diatype]);
		if (diatype == DIATYPE_POPUP){
			diagui_sendcmd (P_Enteraction,"B98\n");
		}else{
			//diagui_sendcmd (P_Sidetitle,"%s\n",diagui_quote(internal->sidetitle.get(),tmp));
		}
		if (!internal->intro.is_empty()){
			diagui_sendcmd (P_Form,"intro\n");		  	
			if (!internal->icon.is_empty()){
				char name_sent[PATH_MAX];
				diagui_sendxpm (internal->icon.get(),name_sent);
				diagui_sendcmd (P_Icon_xpm,"%s\n",name_sent);
				diagui_sendcmd (P_Dispolast,"l 1 c 1\n");
				diagui_sendcmd (P_Form,"subintro\n");
			}
			// Extract the lines and convert the tabs
			const char *para = internal->intro.get();
			while (*para != '\0'){
				char tmp[100];
				char *pt = tmp;
				int pos = 0;
				while (*para != '\0' && *para != '\n'){
					char car = *para++;
					if (car == '\t'){
						if ((pos & 7)==0){
							*pt++ = ' ';
							pos++;
						}
						while ((pos & 7) != 0){
							*pt++ = ' ';
							pos++;
						}
					}else{
						*pt++ = car;
						pos++;
					}
				}
				*pt = '\0';
				if (*para == '\n') para++;
				diagui_send_Label (tmp);
				diagui_sendcmd (P_Newline,"\n");
			}
			diagui_sendcmd (P_End,"\n");
			if (!internal->icon.is_empty()){
				diagui_sendcmd (P_End,"\n");
			}
			diagui_sendcmd (P_Dispolast,"c 1 t 1\n");
			diagui_sendcmd (P_Newline,"\n");
		}
		int lastf = getnb();
		if (lastf > 0){
			diagui_sendcmd (P_Group,"panel\n");

			/* #Specification: GUI / layout / first section
				It is possible to create a dialog with a first visible
				section followed by a notebook containing "less" visible
				sections (options). This is done by putting
				DIALOG::newf_title() calls after the first section.

				We can also do a normal notebook dialog where all
				sections (pages) are selectable with the page tab. This
				is done by starting the dialog with DIALOG::newf_title
				(Each page starts with this).

				For the first case (When the dialog does not start
				with the newf_title() function), linuxconf puts the
				first section inside of a Form centered horizontally
				in the dialog.
			*/
			int first_form_end = -1;
			bool autonewline = internal->autonewline;
			for (int i=0; i<lastf; i++){
				FIELD *f = getitem(i);
				if (f->getnotepadlevel() > 0 && first_form_end == -1){
					first_form_end = i;
				}else if (f->is_passthrough()){
					autonewline = 0;
				}
			}

			int booklevel = 0;
			if (first_form_end > 0) diagui_sendcmd (P_Form,"first\n");
			for (int i=0; i<lastf; i++){
				if (i != 0 && i == first_form_end){
					diagui_sendcmd (P_End,"\n");
					diagui_sendcmd (P_Dispolast,"c 1 t 1\n");
					diagui_sendcmd (P_Newline,"\n");
				}
				FIELD *f = getitem(i);
				f->gui_draw (i,booklevel);
				if (autonewline) diagui_sendcmd (P_Newline,"\n");
			}
			for (; booklevel > 0; booklevel--){
				diagui_sendcmd (P_End,"\n");
				diagui_sendcmd (P_End,"\n");
			}
			diagui_sendcmd (P_End,"\n");
			diagui_sendcmd (P_Dispolast,"c 1 t 1\n");
			diagui_sendcmd (P_Newline,"\n");
		}
		diagui_sendcmd (P_Formbutton,"button\n");
		internal->buttons->gui_draw ();
		diagui_sendcmd (P_End,"\n");
		diagui_sendcmd (P_End,"\n");
		if (context_doend){
			diagui_sendcmd (P_End,"\n");
		}
	}
}

PRIVATE MENU_STATUS DIALOG::editgui_thread(int &nof, int but_options)
{
	extern int uithread_id;
	{
		// Select the first editable field
		while (nof < getnb()){
			FIELD *f = getitem(nof);
			if (f != NULL && f->readonly && !f->may_select){
				nof++;
			}else{
				break;
			}
		}
		if (nof == getnb()) nof--;
	}
	FIELD *f = getitem(nof);
	char prefix = f == NULL ? ' ' : f->getidprefix();
	diagui_sendcmd (P_Curfield,"main-%d-%d %c%d\n",uithread_id
		,internal->gui_id,prefix,nof);
	MENU_STATUS ret = MENU_NULL;
	nof = 0;
	char dianame[20];
	// Same name we are building above
	sprintf (dianame,"main-%d-%d",uithread_id,internal->gui_id);
	valids.add (new SSTRING (dianame));
	while (1){
		uithread_sync (diagui_wait);
		if (lastmsg.is_empty()){
			break;
		}else if (internal->waitmsg.lookup(lastmsg.get())!=-1){
			ret = MENU_MESSAGE;
			break;
		}
	}
	valids.remove_del (valids.lookup(dianame));
	if (ret == MENU_NULL){
		if (diagui_lastbut != -1){
			ret = internal->buttons->bid2status(diagui_lastbut);
			int n = getnb();
			for (int i=0; i<n; i++){
				FIELD *f = getitem(i);
				f->gui_get (i);
			}
		}else{
			nof = diagui_lastmenu;
			ret = MENU_OK;
		}
	}
	return ret;
}

/*
	Wait for an event for this uithread.
	This function is used by dialogs which completly bypass the DIALOG
	object and build their GUI by talking directly to the GUI front-end
	(using diagui_sendcmd). Those dialogs are really "on their own". So
	far only the treemenu module is using that as the tree widget is
	not integrated in the DIALOG object.

	This function returns 0 when something was selected in the dialog
	(on element of the treemenu for example). In that case, action contains
	the selected item.

	The function returns > 0 when a button is selected. The value is the
	ID of the button as passed using diagui_sendcmd().
*/
EXPORT int diagui_sync (
	const char *dianame,	// Base name of the dialog
	SSTRING &path,		// path of the selected component
							// (button, treemenu)
	SSTRING &action)		// Action associated with the component
{
	int ret = -1;
	valids.add (new SSTRING (dianame));
	uithread_sync (diagui_wait);
	valids.remove_del (valids.lookup(dianame));
	path.setfrom (diapath);
	action.setfrom ("");
	if (diagui_lastbut != -1){
		ret = diagui_lastbut;
	}else{
		action.setfrom (actionid);
		ret = 0;
	}
	return ret;
}

/*
	Wait for an event on a POPEN connection for this uithread.
*/
EXPORT int diagui_sync (POPENFD &po, int timeout)
{
	extern int uithread_id;
	POPENWAITID *w = new POPENWAITID (po,timeout,uithread_id);
	tbpopen.add (w);
	uithread_sync (diagui_wait);
	int ret = w->retcode;
	tbpopen.remove_del (w);
	return ret;
}

static bool help_html = false;

/*
	Record the fact that the GUI frontend support html helps
*/
void diagui_sethtmlhelp()
{
	help_html = true;
}

/*
	Transmit an html help to the GUI frontend
*/
void diagui_sendhtmlhelp (const char *path)
{
	FILE *fin = fopen (path,"r");
	if (fin != NULL){
		char buf[500];
		diagui_sendcmd (P_Html,"%s\n",path);
		while (fgets_strip(buf,sizeof(buf)-1,fin,NULL)!=NULL){
			char tmp[1000];
			diagui_sendcmd (P_Str,"%s\n",diagui_quote(buf,tmp));
		}
		diagui_sendcmd (P_End,"\n");
	}
}

static void ft (void *p)
{
	char *rpath = (char *)p;
	char path[PATH_MAX];
	if (html_locatefile (rpath,help_html ? ".html" : ".help"
		,path,PATH_MAX)!=-1){
		if (help_html){
			diagui_sendhtmlhelp (rpath);
		}else{
			dialog_textbox (path,path);
		}
	}else{
		xconf_error (MSG_R(E_NOHELPFILE),rpath);
	}
	free (rpath);
}

/*
	Present a help screen.
	Function used internally and by some module handling the GUI themselves
*/
void diagui_showhelp (const char *relpath)
{
	char *pt = strdup(relpath);
	uithread (ft,(void*)pt);
}

/*
	Present a help screen.
	Function used internally and by some module handling the GUI themselves
*/
EXPORT void diagui_showhelp (HELP_FILE &helpfile)
{
	char rpath[PATH_MAX];
	helpfile.getrpath(rpath);
	diagui_showhelp (rpath);
}

PRIVATE MENU_STATUS DIALOG::editgui(
	int &nof,
	int but_options)
{
	MENU_STATUS ret = MENU_NULL;
	while (1){
		ret = editgui_thread (nof,but_options);
		if (ret != MENU_HELP) break;
		internal->buttons->help (NULL);
	}
	return ret;
}
/*
	Return the current value of a field
	All field have an ID in the java frontend. This id
	is generally a letter followed by a number.
*/
EXPORT const char *diagui_getval (char prefix, int nof)
{
	return diagui_getval (NULL,prefix,nof);
}
EXPORT const char *diagui_getval (const char *diapath, char prefix, int nof)
{
	char id[100];
	sprintf (id,"%c%d",prefix,nof);
	return java->getval(diapath,id);
}
EXPORT const char *diagui_getval (
	const char *diapath,
	char prefix,
	const char *str)
{
	char id[100];
	sprintf (id,"%c%s",prefix,str);
	return java->getval(diapath,id);
}
EXPORT const char *diagui_getval (char prefix, const char *str)
{
	return diagui_getval(NULL,prefix,str);
}

/*
	Return the current values of a field
	All field have an ID in the java frontend. This ID
	is generally a letter followed by a number. Some fields (textarea)
	are returned by the front-end as multiple lines with the same ID.

	Return the number of lines places in tb[].
*/
EXPORT int diagui_getvals (
	const char *diapath,
	char prefix,
	int nof,
	SSTRINGS &tb)
{
	char id[100];
	sprintf (id,"%c%d",prefix,nof);
	return java->getvals(diapath,id,tb);
}
EXPORT int diagui_getvals (char prefix, int nof, SSTRINGS &tb)
{
	return diagui_getvals (NULL,prefix,nof,tb);
}


/*
	Wakeup dialogs which are waiting for this message
*/
EXPORT void dialog_sendmessage(const char *msg)
{
	// We make all dialogs receive the message. When
	// the dialog wakeup (in its own thread), it checks if it needs
	// the message. This means that most dialog wake up for nothing.
	// should do the trick anyway.
	for (int i=0; i<valids.getnb(); i++){
		const char *dianame = valids.getitem(i)->get();
		messages.add (msg,dianame);
	}
}

/*
	Return the last message received
*/
EXPORT const char *dialog_getmessage()
{
	return lastmsg.get();
}

