// curses++.h  (this is -*-c++-*-)
//
//  Copyright 1999 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Simple class wrappers around various CURSES functions.

#ifndef CURSES_PLUSPLUS_H
#define CURSES_PLUSPLUS_H

#include <string>
#include <curses.h>

inline chtype _getbkgd(WINDOW *win)
{
  return getbkgd(win);
}

inline int _box(WINDOW *win, chtype verch, chtype horch)
{
  return box(win, verch, horch);
}

inline void _getyx(WINDOW *win, int &y, int &x)
{
  getyx(win, y, x);
}

inline void _getparyx(WINDOW *win, int &y, int &x)
{
  getparyx(win, y, x);
}

inline void _getbegyx(WINDOW *win, int &y, int &x)
{
  getbegyx(win, y, x);
}

inline void _getmaxyx(WINDOW *win, int &y, int &x)
{
  getmaxyx(win, y, x);
}

#undef getbkgd
#undef box

#undef getyx
#undef getparyx
#undef getbegyx
#undef getmaxyx

#undef addch
#undef addchnstr
#undef addchstr
#undef addnstr
#undef addstr
#undef attroff
#undef attron
#undef attrset
#undef attr_set
#undef bkgd
#undef bkgdset
#undef clear
#undef clrtobot
#undef clrtoeol
#undef delch
#undef deleteln
#undef echochar
#undef erase
#undef getch
#undef getstr
#undef inch
#undef inchnstr
#undef innstr
#undef insch
#undef insdelln
#undef insertln
#undef insnstr
#undef insstr
#undef instr
#undef move
#undef refresh
#undef scrl
#undef scroll
#undef setscrreg
#undef standend
#undef standout
#undef timeout

#undef mvaddch
#undef mvaddchnstr
#undef mvaddchstr
#undef mvaddnstr
#undef mvaddstr
#undef mvdelch
#undef mvgetch
#undef mvgetnstr
#undef mvgetstr
#undef mvhline
#undef mvinch
#undef mvinchnstr
#undef mvinchstr
#undef mvinnstr
#undef mvinsch
#undef mvinsnstr
#undef mvinsstr
#undef mvinstr
#undef mvvline

#undef border
#undef hline
#undef vline

#undef touchline

//  The following class encapsulates a CURSES window, mostly with inlined
// versions of w* and mvw*.  subwin and newwin are encapsulated with
// constructors; casting to WINDOW * is also supported.
//
//  A note of caution: deleting or destroying one of these will NOT call
// delwin on the associated WINDOW * value.  That must be explicitly done with
// delwin() [ not too pretty, but..]
class cwindow
{
  WINDOW *win;
  // The actual curses window

public:
  cwindow(WINDOW *_win):win(_win) {}
  cwindow(int nlines, int ncols, int begin_y, int begin_x)
    :win(newwin(nlines, ncols, begin_y, begin_x)) {}
  cwindow(WINDOW *parent, int nlines, int ncols, int begin_y, int begin_x)
    :win(subwin(parent, nlines, ncols, begin_y, begin_x)) {}

  int mvwin(int y, int x) {return ::mvwin(win, y, x);}
  int delwin() {return ::delwin(win);}

  void syncup() {wsyncup(win);}
  int syncok(bool bf) {return ::syncok(win, bf);}
  void cursyncup() {wcursyncup(win);}
  void syncdown() {wsyncdown(win);}

  int scroll(int n=1) {return wscrl(win, n);}
  // Does both scroll() and wscsrl()

  int addch(chtype ch) {return waddch(win, ch);}
  int mvaddch(int y, int x, chtype ch) {return mvwaddch(win, y, x, ch);}

  int addstr(const char *str) {return waddstr(win, str);}
  int addnstr(const char *str, int n) {return waddnstr(win, str, n);}
  int mvaddstr(int y, int x, const char *str) {return mvwaddstr(win, y, x, str);}
  int mvaddnstr(int y, int x, const char *str, int n) {return mvwaddnstr(win, y, x, str, n);}

  int attroff(int attrs) {return wattroff(win, attrs);}
  int attron(int attrs) {return wattron(win, attrs);}
  int attrset(int attrs) {return wattrset(win, attrs);}
  //  int attr_set(int attrs, void *opts) {return wattr_set(win, attrs, opts);}

  void bkgdset(const chtype ch) {wbkgdset(win, ch);}
  int bkgd(const chtype ch) {return wbkgd(win, ch);}
  chtype getbkgd() {return _getbkgd(win);}

  int border(chtype ls, chtype rs, chtype ts, chtype bs, chtype tl, chtype tr, chtype bl, chtype br)
    {return wborder(win, ls, rs, ts, bs, tl, tr, bl, br);}

  int box(chtype verch, chtype horch) {return _box(win, verch, horch);}
  int hline(chtype ch, int n) {return whline(win, ch, n);}
  int vline(chtype ch, int n) {return wvline(win, ch, n);}
  int mvhline(int y, int x, chtype ch, int n) {return mvwhline(win, y, x, ch, n);}
  int mvvline(int y, int x, chtype ch, int n) {return mvwvline(win, y, x, ch, n);}

  int delch() {return wdelch(win);}
  int mvdelch(int y, int x) {return mvwdelch(win, y, x);}

  int deleteln() {return wdeleteln(win);}
  int insdelln(int n) {return winsdelln(win,n);}
  int insertln() {return winsertln(win);}

  int echochar(chtype ch) {return wechochar(win, ch);}

  int getch() {return wgetch(win);}
  int mvgetch(int y, int x) {return mvwgetch(win, y, x);}

  int move(int y, int x) {return wmove(win, y, x);}
  void getyx(int &y, int &x) {_getyx(win, y, x);}
  void getparyx(int &y, int &x) {_getparyx(win, y, x);}
  void getbegyx(int &y, int &x) {_getbegyx(win, y, x);}
  void getmaxyx(int &y, int &x) {_getmaxyx(win, y, x);}

  void show_string_as_progbar(int x, int y, string s, int attr1, int attr2, int size1, int totalsize);
  // Glitz bit :) Displays the given string with a progress bar behind it.

  void display_header(string s, int attr);
  void display_status(string s, int attr);
  // Make it easier to write interfaces that have a header and status line..
  // they do what they say :)

  int overlay(cwindow &dstwin) {return ::overlay(win, dstwin);}
  int overwrite(cwindow &dstwin) {return ::overwrite(win, dstwin);}
  int copywin(cwindow &dstwin, int sminrow, int smincol, int dminrow, int dmincol, int dmaxrow, int dmaxcol, int overlay)
    {return ::copywin(win, dstwin, sminrow, smincol, dminrow, dmincol, dmaxrow, dmaxcol, overlay);}

  int refresh() {return wrefresh(win);}
  int noutrefresh() {return wnoutrefresh(win);}

  int touch() {return touchwin(win);}
  int untouch() {return untouchwin(win);}
  int touchln(int y, int n, int changed) {return ::wtouchln(win, y, n, changed);}
  int touchline(int start, int count) {return touchln(start, count, 1);}
  int untouchline(int start, int count) {return touchln(start, count, 0);}

  int erase() {return werase(win);}
  int clear() {return wclear(win);}
  int clrtobot() {return wclrtobot(win);}
  int clrtoeol() {return wclrtoeol(win);}

  int keypad(bool bf) {return ::keypad(win,bf);}
  int meta(bool bf) {return ::meta(win,bf);}
  int nodelay(bool bf) {return ::nodelay(win, bf);}
  int notimeout(bool bf) {return ::notimeout(win, bf);}
  void timeout(int delay) {wtimeout(win, delay);}

  int clearok(bool bf) {return ::clearok(win, bf);}
  int idlok(bool bf) {return ::idlok(win, bf);}
  void idcok(bool bf) {::idcok(win, bf);}
  void immedok(bool bf) {::immedok(win, bf);}
#if defined(NCURSES_VERSION_MAJOR) && NCURSES_VERSION_MAJOR>=5
  int leaveok(bool bf) {int rval=::leaveok(win, bf); curs_set(bf?0:1); return rval;}
#else
  int leaveok(bool bf) {return ::leaveok(win, bf);}
#endif
  int setscrreg(int top, int bot) {return wsetscrreg(win, top, bot);}
  int scrollok(bool bf) {return ::scrollok(win,bf);}

  int printw(char *str, ...);
  /* You guessed it.. :) */

  bool enclose(int y, int x) {return wenclose(win, y, x);}

  WINDOW *getwin() {return win;}
  operator WINDOW * () {return win;}
  operator const WINDOW * () const {return win;}
  operator bool () {return win!=NULL;}
  WINDOW *operator =(WINDOW *_win) {return win=_win;}
  bool operator ==(cwindow &other) {return win==other.win;}
  bool operator !=(cwindow &other) {return win!=other.win;}
};

extern cwindow rootwin;
// This is stdscr, but calling it 'stdscr' would confuse the compiler, so I'm
// confusing the programmer instead :)

void init_curses();
// Initializes curses and sets rootwin to the correct value

void resize();
// Called when a terminal resize is detected.
#endif
